/*
* Copyright (c) 2022 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:      
*
*/








#include <e32std.h>
#include <e32base.h>
#include <e32test.h>
#include <f32file.h>
#include <charconv.h>

///////////////////////////////////////////////////////////////////////////////////////

RTest TheTest(_L("TISO2022JP"));
	
///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
//Tests macroses and functions.
//If (!aValue) then the test will be panicked, the test data files will be deleted.
static void Check(TInt aValue, TInt aLine)
	{
	if(!aValue)
		{
		TheTest(EFalse, aLine);
		}
	}
//If (aValue != aExpected) then the test will be panicked, the test data files will be deleted.
static void Check(TInt aValue, TInt aExpected, TInt aLine)
	{
	if(aValue != aExpected)
		{
		RDebug::Print(_L("*** Expected error: %d, got: %d\r\n"), aExpected, aValue);
		TheTest(EFalse, aLine);
		}
	}
//Use these to test conditions.
#define TEST(arg) ::Check((arg), __LINE__)
#define TEST2(aValue, aExpected) ::Check(aValue, aExpected, __LINE__)

///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
	
const TInt KBufferLength=100;
const TInt KNotEnoughBufferLength=17;
/**
@SYMTestCaseID          SYSLIB-CHARCONV-CT-0522
@SYMTestCaseDesc        Tests for conversion from Unicode to ISO2022JP
@SYMTestPriority        Medium
@SYMTestActions         Tests for conversion from Unicode to ISO2022JP and back to Unicode
@SYMTestExpectedResults Test must not fail
@SYMREQ                 REQ0000
*/
LOCAL_C void TestConversionFromUnicodeToIso(
                CCnvCharacterSetConverter& aCharacterSetConverter, 
                const TDesC16& aOriginalUnicode)
 	{
 	TBuf8<KNotEnoughBufferLength> generatedIso2022Jp;
	generatedIso2022Jp.FillZ(KNotEnoughBufferLength);
 	const TInt returnValue=aCharacterSetConverter.ConvertFromUnicode(generatedIso2022Jp, aOriginalUnicode);
 	TEST(returnValue>=0);
 	TInt state=CCnvCharacterSetConverter::KStateDefault;
 	TBuf16<KBufferLength> generatedUnicode;
	TInt testresult;
	testresult=aCharacterSetConverter.ConvertToUnicode(generatedUnicode, generatedIso2022Jp, state);
 	TEST(testresult==0);
 //	TEST(generatedUnicode==aOriginalUnicode);
 	}
/**
@SYMTestCaseID          SYSLIB-CHARCONV-CT-0523
@SYMTestCaseDesc        Tests for truncated conversion from Unicode to ISO2022JP 
@SYMTestPriority        Medium
@SYMTestActions         Tests for truncated conversion from Unicode to ISO2022JP and back to Unicode
@SYMTestExpectedResults Test must not fail
@SYMREQ                 REQ0000
*/
LOCAL_C void TestTruncatedConversionFromUnicodeToIso2022Jp(
                CCnvCharacterSetConverter& aCharacterSetConverter, 
                const TDesC16& aOriginalUnicode)
	{
	for (TInt i=aOriginalUnicode.Length(); i>=0; --i)
		{
		TBuf8<KBufferLength> generatedIso2022Jp;
		const TInt returnValue=aCharacterSetConverter.ConvertFromUnicode(generatedIso2022Jp, aOriginalUnicode.Left(i));
		TEST(returnValue>=0);
		TBuf8<KBufferLength> generatedsecondPartOfIso2022Jp;
		TEST(aCharacterSetConverter.ConvertFromUnicode(generatedsecondPartOfIso2022Jp, aOriginalUnicode.Mid(i-returnValue))==0);
		generatedIso2022Jp.Append(generatedsecondPartOfIso2022Jp);
		TInt state=CCnvCharacterSetConverter::KStateDefault;
		TBuf16<KBufferLength> generatedUnicode;
		TEST(aCharacterSetConverter.ConvertToUnicode(generatedUnicode, generatedIso2022Jp, state)==0);
		TEST(generatedUnicode==aOriginalUnicode);
		}
	}
/**
@SYMTestCaseID          SYSLIB-CHARCONV-CT-0524
@SYMTestCaseDesc        Splitting and converting from Unicode to ISO2022JP test
@SYMTestPriority        Medium
@SYMTestActions         Tests for conversion after splitting, from Unicode to ISO2022JP and back to Unicode
@SYMTestExpectedResults Test must not fail
@SYMREQ                 REQ0000
*/
LOCAL_C void TestSplittingConvertingFromUnicodeToIso2022Jp(
                CCnvCharacterSetConverter& aCharacterSetConverter, 
                TInt aMaximumLengthLowerLimit, 
                TInt aMaximumLengthUpperLimit, 
                TInt aExpectedNumberOfUnicodeCharactersNotConvertedAtSplit, 
                const TDesC8& aExpectedFirstPartOfIso2022Jp, 
                const TDesC8& aExpectedSecondPartOfIso2022Jp, 
                const TDesC16& aOriginalUnicode)
	{
	TEST(aMaximumLengthLowerLimit<=aMaximumLengthUpperLimit);
	TEST(aMaximumLengthUpperLimit<=KBufferLength);
	TUint8 iso2022JpBuffer[KBufferLength];
	for (TInt i=aMaximumLengthLowerLimit; i<=aMaximumLengthUpperLimit; ++i)
		{
		TPtr8 generatedFirstPartOfIso2022Jp(iso2022JpBuffer, i);
		TEST(aCharacterSetConverter.ConvertFromUnicode(generatedFirstPartOfIso2022Jp, aOriginalUnicode)==aExpectedNumberOfUnicodeCharactersNotConvertedAtSplit);
		TEST(generatedFirstPartOfIso2022Jp==aExpectedFirstPartOfIso2022Jp);
		TBuf8<KBufferLength> generatedSecondPartOfIso2022Jp;
		TEST(aCharacterSetConverter.ConvertFromUnicode(generatedSecondPartOfIso2022Jp, aOriginalUnicode.Right(aExpectedNumberOfUnicodeCharactersNotConvertedAtSplit))==0);
		TEST(generatedSecondPartOfIso2022Jp==aExpectedSecondPartOfIso2022Jp);
		TInt state=CCnvCharacterSetConverter::KStateDefault;
		TBuf16<KBufferLength> generatedUnicode;
		TEST(aCharacterSetConverter.ConvertToUnicode(generatedUnicode, generatedFirstPartOfIso2022Jp, state)==0);
		TBuf16<KBufferLength> generatedSecondPartOfUnicode;
		TEST(aCharacterSetConverter.ConvertToUnicode(generatedSecondPartOfUnicode, generatedSecondPartOfIso2022Jp, state)==0);
		generatedUnicode.Append(generatedSecondPartOfUnicode);
		TEST(generatedUnicode==aOriginalUnicode);
		}
	}
/**
@SYMTestCaseID          SYSLIB-CHARCONV-CT-0525
@SYMTestCaseDesc        Tests for truncated conversion from ISO2022JP to Unicode test
@SYMTestPriority        Medium
@SYMTestActions         Tests for truncated conversion from ISO2022JP to Unicode and back to ISO2022JP
@SYMTestExpectedResults Test must not fail
@SYMREQ                 REQ0000
*/
LOCAL_C void TestTruncatedConversionToUnicodeFromIso2022Jp(
                CCnvCharacterSetConverter& aCharacterSetConverter, 
                const TDesC16& aExpectedUnicode, 
                const TDesC8& aOriginalIso2022Jp)
	{
	for (TInt i=aOriginalIso2022Jp.Length(); i>=3; --i) // 3 is the length of ISO-2022-JP's longest escape sequence
		{
		TInt state=CCnvCharacterSetConverter::KStateDefault;
		TBuf16<KBufferLength> generatedUnicode;
		const TInt returnValue=aCharacterSetConverter.ConvertToUnicode(generatedUnicode, aOriginalIso2022Jp.Left(i), state);
		TEST(returnValue>=0);
		TBuf16<KBufferLength> generatedsecondPartOfUnicode;
		TEST(aCharacterSetConverter.ConvertToUnicode(generatedsecondPartOfUnicode, aOriginalIso2022Jp.Mid(i-returnValue), state)==0);
		generatedUnicode.Append(generatedsecondPartOfUnicode);
		TEST(generatedUnicode==aExpectedUnicode);
		}
	}
/**
@SYMTestCaseID          SYSLIB-CHARCONV-CT-0526
@SYMTestCaseDesc        Splitting and converting from ISO2022JP to Unicode test.  
@SYMTestPriority        Medium
@SYMTestActions         Tests for conversion after splitting, from ISO2022JP to Unicode and back to ISO2022JP
@SYMTestExpectedResults Test must not fail
@SYMREQ                 REQ0000
*/
LOCAL_C void TestSplittingConvertingToUnicodeFromIso2022Jp(
                CCnvCharacterSetConverter& aCharacterSetConverter, 
                TInt aMaximumLengthLowerLimit, 
                TInt aMaximumLengthUpperLimit, 
                TInt aExpectedNumberOfIso2022JpBytesNotConvertedAtSplit, 
                TInt aExpectedLengthOfFirstPartOfUnicode, 
                const TDesC16& aExpectedUnicode, 
                const TDesC8& aOriginalIso2022Jp)
	{
	TEST(aMaximumLengthLowerLimit<=aMaximumLengthUpperLimit);
	TEST(aMaximumLengthUpperLimit<=KBufferLength);
	TUint16 unicodeBuffer[KBufferLength];
	for (TInt i=aMaximumLengthLowerLimit; i<=aMaximumLengthUpperLimit; ++i)
		{
		TPtr16 generatedFirstPartOfUnicode(unicodeBuffer, i);
		TInt state=CCnvCharacterSetConverter::KStateDefault;
		const TInt returnValue=aCharacterSetConverter.ConvertToUnicode(generatedFirstPartOfUnicode, aOriginalIso2022Jp, state);
		TEST(generatedFirstPartOfUnicode==aExpectedUnicode.Left(aExpectedLengthOfFirstPartOfUnicode));
		TEST(returnValue==aExpectedNumberOfIso2022JpBytesNotConvertedAtSplit);
		TBuf16<KBufferLength> generatedSecondPartOfUnicode;
		TEST(aCharacterSetConverter.ConvertToUnicode(generatedSecondPartOfUnicode, aOriginalIso2022Jp.Right(aExpectedNumberOfIso2022JpBytesNotConvertedAtSplit), state)==0);
		TEST(generatedSecondPartOfUnicode==aExpectedUnicode.Mid(aExpectedLengthOfFirstPartOfUnicode));
		TBuf8<KBufferLength> generatedIso2022Jp;
		TEST(aCharacterSetConverter.ConvertFromUnicode(generatedIso2022Jp, generatedFirstPartOfUnicode)==0);
		TBuf8<KBufferLength> generatedSecondPartOfIso2022Jp;
		TEST(aCharacterSetConverter.ConvertFromUnicode(generatedSecondPartOfIso2022Jp, generatedSecondPartOfUnicode)==0);
		generatedIso2022Jp.Append(generatedSecondPartOfIso2022Jp);
		TBuf16<KBufferLength> regeneratedUnicode;
		state=CCnvCharacterSetConverter::KStateDefault;
		TEST(aCharacterSetConverter.ConvertToUnicode(regeneratedUnicode, generatedIso2022Jp, state)==0);
		TEST(regeneratedUnicode==aExpectedUnicode);
		state=CCnvCharacterSetConverter::KStateDefault;
		TEST(aCharacterSetConverter.ConvertToUnicode(regeneratedUnicode, aOriginalIso2022Jp, state)==0);
		TEST(regeneratedUnicode==aExpectedUnicode);
		}
	}
/**
@SYMTestCaseID          SYSLIB-CHARCONV-CT-0527
@SYMTestCaseDesc        Conversion of bad ISO2022JP format to Unicode test
@SYMTestPriority        Medium
@SYMTestActions         Tests to convert bad format ISO2022JP input to Unicode.
@SYMTestExpectedResults Test must not fail
@SYMREQ                 REQ0000
*/
LOCAL_C void TestIsIllFormedIso2022Jp(CCnvCharacterSetConverter& aCharacterSetConverter, const TDesC8& aIso2022Jp)
	{
	TBuf16<50> generatedUnicode;
	TInt state=CCnvCharacterSetConverter::KStateDefault;
	TPtrC8 remainderOfIso2022Jp(aIso2022Jp);
	TInt lastReturnValue=KMaxTInt;
	FOREVER
		{
		const TInt returnValue=aCharacterSetConverter.ConvertToUnicode(generatedUnicode, remainderOfIso2022Jp, state);
		if (returnValue==CCnvCharacterSetConverter::EErrorIllFormedInput)
			{
			break;
			}
		TEST(returnValue>0);
		TEST(returnValue<lastReturnValue);
		lastReturnValue=returnValue;
		remainderOfIso2022Jp.Set(remainderOfIso2022Jp.Right(returnValue));
		}
	}
/**
@SYMTestCaseID          SYSLIB-CHARCONV-CT-0528
@SYMTestCaseDesc        ISO2022JP to Unicode and Unicode to ISO2022JP conversion tests 
@SYMTestPriority        Medium
@SYMTestActions         Call up conversion of ISO2022JP to Unicode test functions
@SYMTestExpectedResults Test must not fail
@SYMREQ                 REQ0000
*/
LOCAL_C void DoE32MainL()
	{
	RFs fileServerSession;
	CleanupClosePushL(fileServerSession);
	User::LeaveIfError(fileServerSession.Connect());
	CCnvCharacterSetConverter* characterSetConverter=CCnvCharacterSetConverter::NewLC();
	CArrayFix<CCnvCharacterSetConverter::SCharacterSet>* arrayOfCharacterSetsAvailable=CCnvCharacterSetConverter::CreateArrayOfCharacterSetsAvailableLC(fileServerSession);
	TheTest.Printf(_L("Available:\n"));
	TInt i;
	for (i=arrayOfCharacterSetsAvailable->Count()-1; i>=0; --i)
		{
		const CCnvCharacterSetConverter::SCharacterSet& charactersSet=(*arrayOfCharacterSetsAvailable)[i];
		characterSetConverter->PrepareToConvertToOrFromL(charactersSet.Identifier(), *arrayOfCharacterSetsAvailable, fileServerSession);
		TPtrC charactersSetName(charactersSet.Name());
		if (charactersSet.NameIsFileName())
			{
			charactersSetName.Set(TParsePtrC(charactersSetName).Name());
			}
		TheTest.Printf(_L("    %S\n"), &charactersSetName);
		}
	TheTest.Start(_L("Testing ISO-2022-JP conversions"));
	characterSetConverter->PrepareToConvertToOrFromL(KCharacterSetIdentifierIso2022Jp, *arrayOfCharacterSetsAvailable, fileServerSession);
	//
	TheTest.Next(_L("Empty descriptor"));
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 0, 10, 0, KNullDesC8, KNullDesC8, KNullDesC16);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 0, 10, 0, 0, KNullDesC16, KNullDesC8);
	TheTest.Next(_L("Testing converting to ISO-2022-JP"));

	_LIT(KTestUnicode,">.@>0l90"); 			
	_LIT(KTestUnicode2,"\x0393\x03b1\x03c3\x03bf\x03c5\x3055\x3088");
 	TestConversionFromUnicodeToIso(*characterSetConverter, KTestUnicode2);
 	TestConversionFromUnicodeToIso(*characterSetConverter, KTestUnicode);

	TBuf16<50> originalUnicode;
	originalUnicode.Format(_L16("%c%c%c%c%c\xa5\\%c%c%c%c%c"), 0x0393, 0x03b1, 0x03c3, 0x03bf, 0x03c5, 0x3055, 0x3088, 0x3046, 0x306a, 0x3089);
	TestTruncatedConversionFromUnicodeToIso2022Jp(*characterSetConverter, originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 3, 7, 12, _L8(""), _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 8, 9, 11, _L8("\x1b\x24\x42\x26\x23\x1b\x28\x42"), _L8("\x1b\x24\x42\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 10, 11, 10, _L8("\x1b\x24\x42\x26\x23\x26\x41\x1b\x28\x42"), _L8("\x1b\x24\x42\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 12, 13, 9, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x1b\x28\x42"), _L8("\x1b\x24\x42\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 14, 15, 8, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x1b\x28\x42"), _L8("\x1b\x24\x42\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 16, 19, 7, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x42"), _L8("\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
    TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 20, 23, 6, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42"), _L8("\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 24, 28, 5, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\"), _L8("\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 29, 30, 4, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x1b\x28\x42"), _L8("\x1b\x24\x42\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 31, 32, 3, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x1b\x28\x42"), _L8("\x1b\x24\x42\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 33, 34, 2, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x1b\x28\x42"), _L8("\x1b\x24\x42\x24\x4a\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 35, 36, 1, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x1b\x28\x42"), _L8("\x1b\x24\x42\x24\x69\x1b\x28\x42"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 37, 50, 0, _L8("\x1b\x24\x42\x26\x23\x26\x41\x26\x52\x26\x4f\x26\x54\x1b\x28\x4a\\\x1b\x28\x42\\\x1b\x24\x42\x24\x35\x24\x68\x24\x26\x24\x4a\x24\x69\x1b\x28\x42"), KNullDesC8, originalUnicode);
	originalUnicode.Format(_L16("%cX%cY%cZ"), 0x6153, 0x6376, 0x65d9);
	TestTruncatedConversionFromUnicodeToIso2022Jp(*characterSetConverter, originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 3, 7, 6, _L8(""), _L8("\x1b\x24\x42XX\x1b\x28\x42X\x1b\x24\x42YY\x1b\x28\x42Y\x1b\x24\x42ZZ\x1b\x28\x42Z"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 8, 11, 5, _L8("\x1b\x24\x42XX\x1b\x28\x42"), _L8("X\x1b\x24\x42YY\x1b\x28\x42Y\x1b\x24\x42ZZ\x1b\x28\x42Z"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 12, 16, 4, _L8("\x1b\x24\x42XX\x1b\x28\x42X"), _L8("\x1b\x24\x42YY\x1b\x28\x42Y\x1b\x24\x42ZZ\x1b\x28\x42Z"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 17, 20, 3, _L8("\x1b\x24\x42XX\x1b\x28\x42X\x1b\x24\x42YY\x1b\x28\x42"), _L8("Y\x1b\x24\x42ZZ\x1b\x28\x42Z"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 21, 25, 2, _L8("\x1b\x24\x42XX\x1b\x28\x42X\x1b\x24\x42YY\x1b\x28\x42Y"), _L8("\x1b\x24\x42ZZ\x1b\x28\x42Z"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 26, 29, 1, _L8("\x1b\x24\x42XX\x1b\x28\x42X\x1b\x24\x42YY\x1b\x28\x42Y\x1b\x24\x42ZZ\x1b\x28\x42"), _L8("Z"), originalUnicode);
	TestSplittingConvertingFromUnicodeToIso2022Jp(*characterSetConverter, 30, 40, 0, _L8("\x1b\x24\x42XX\x1b\x28\x42X\x1b\x24\x42YY\x1b\x28\x42Y\x1b\x24\x42ZZ\x1b\x28\x42Z"), KNullDesC8, originalUnicode);
	TheTest.Next(_L("Testing converting to Unicode"));
	const TPtrC8 originalIso2022Jp(_S8("\x1b\x24\x40\x1b\x28\x4aMy name is \x1b\x28\x4a\x1b\x28\x42\x1b\x24\x40\x25\x47\x25\x23\x25\x53\x25\x45\x1b\x28\x4a in \x1b\x24\x42\x46\x7c\x4b\x5c\x38\x6c\x1b\x28\x42\\~\x1b\x28\x4a\\~"));
	TBuf16<50> expectedUnicode;
	expectedUnicode.Format(_L16("My name is %c%c%c%c in %c%c%c\\~%c%c"), 0x30c7, 0x30a3, 0x30d3, 0x30c5, 0x65e5, 0x672c, 0x8a9e, 0x00a5, 0x203e);
	TestTruncatedConversionToUnicodeFromIso2022Jp(*characterSetConverter, expectedUnicode, originalIso2022Jp);
	TestTruncatedConversionToUnicodeFromIso2022Jp(*characterSetConverter, _L16(" Hello"), _L8("\x1b\x24\x42\x1b\x28\x4a\x1b\x24\x42\x1b\x28\x4a\x1b\x24\x42\x1b\x28\x4a Hello"));
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 0, 0, 60, 0, expectedUnicode, originalIso2022Jp);
	for (i=1; i<=10; ++i)
		{
		TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, i, i, 54-i, i, expectedUnicode, originalIso2022Jp);
		}
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 11, 11, 34, 11, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 12, 12, 32, 12, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 13, 13, 30, 13, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 14, 14, 28, 14, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 15, 15, 23, 15, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 16, 16, 22, 16, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 17, 17, 21, 17, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 18, 18, 20, 18, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 19, 19, 16, 19, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 20, 20, 14, 20, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 21, 21, 12, 21, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 22, 22, 7, 22, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 23, 23, 6, 23, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 24, 24, 2, 24, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 25, 25, 1, 25, expectedUnicode, originalIso2022Jp);
	TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, 26, 40, 0, 26, expectedUnicode, originalIso2022Jp);
	TheTest.Next(_L("Testing the default ISO-2022-JP state"));
	for (i=0; i<=6; ++i)
		{
		TestSplittingConvertingToUnicodeFromIso2022Jp(*characterSetConverter, i, i, 6-i, i, _L16("Hello\xa5"), _L8("Hello\\"));
		}
	TheTest.Next(_L("Testing ill-formed ISO-2022-JP"));
	TestIsIllFormedIso2022Jp(*characterSetConverter, _L8("abc \x1b\x24\x42\x21\x1b\x28\x4a def"));
	TestIsIllFormedIso2022Jp(*characterSetConverter, _L8("abc \x1b\x24\x42\x21\x21\x21\x1b\x28\x4a def"));
	TestIsIllFormedIso2022Jp(*characterSetConverter, _L8("abc \x1b\x24\x42\x21\x21\x21\x21\x21\x1b\x28\x4a def"));
	TestIsIllFormedIso2022Jp(*characterSetConverter, _L8("abc \x1b"));
	TestIsIllFormedIso2022Jp(*characterSetConverter, _L8("abc \x1b\x24"));
	TestIsIllFormedIso2022Jp(*characterSetConverter, _L8("abc \x1b\x24\xff"));
	TestIsIllFormedIso2022Jp(*characterSetConverter, _L8("abc \x1b\x26\x40"));
	CleanupStack::PopAndDestroy(3); // arrayOfCharacterSetsAvailable and characterSetConverter and fileServerSession
	}

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;

	TheTest.Title();

	CTrapCleanup* trapCleanup=CTrapCleanup::New();
	TEST(trapCleanup != NULL);

	TRAPD(error, DoE32MainL());
	TEST2(error, KErrNone);

	delete trapCleanup;

	TheTest.End();
	TheTest.Close();

	__UHEAP_MARKEND;
	return KErrNone;
	}

