#
# Copyright (c) 2002-2006 Nokia Corporation and/or its subsidiary(-ies).
# All rights reserved.
# This component and the accompanying materials are made available
# under the terms of "Eclipse Public License v1.0"
# which accompanies this distribution, and is available
# at the URL "http://www.eclipse.org/legal/epl-v10.html".
#
# Initial Contributors:
# Nokia Corporation - initial contribution.
#
# Contributors:
#
# Description:  See the usage-method below.
#
# -----------------------------
# - Parses TBS build log file -
# -----------------------------
#	The script offers parsing functionality for build logs generated by Symbian TBS build tool.
#	It takes the build log file as an input and creates list of compiled binaries as an output.
#	The parser searches hard coded key words -- abld what and ++ Finished and parses the data between the key words.
#	Optionally you can also zip the binaries. The parser is configurable by giving filter, zipping etc. parameters as an argument.
#	You can get instructions by launching the script parse_what_log.pl help
#	Note: Tested with Symbian TBS build logs only.
#
# INPUT:	mandatory:	'-i' name of the build log input file, multiple allowed
#											'-o' name of the parsed output file or '-zip' name of the zip file
#											'-filter' and '-pos' defines the string and position ( zero by default ) on the line, which we want to store
#
#					optional:		'-ex' defines a list of strings, which excludes lines containing one of these strings
#											'-zip_prefix' prefix for each secondary zip file name when "double" zipping method selected
#											'-zip2zip' enables the "double" zipping. Primary zip file name is defined in argument '-zip'
#											'-zip' full path for the primary zip file
#
# OUTPUT: mandatory:	list of compiled binaries as an output file and/or single zip file named according '-zip' parameter
# 				optional:		separate zip files named according to prefix and component's name containing one's compiled binaries
#

use warnings; 
use strict; 
use Getopt::Long;
Getopt::Long::Configure ("bundling_override","ignore_case_always");

my ($what_log, $filter_string) = '';
my ($start_found, $filter_position, $help, $i, $skip_line) = 0;
my @build_log = ();
my @lines = ();
my @exclude_list = ();
my $temp_folder = "\\temp_what_zip2zip\\";	# temporaly zip for zip2zip
my ($app, @app_path, $zip_cmd, $zip_prefix, $zip2zip, $zip) = 0;
my %components;

my $parse_start_string = "^(-- abld .*-w)";	# defines the starting point for parsing
my $parse_end_string = "++ Finished";			# defines the ending point for parsing
my $app_key_word = "Chdir ";							# defines the key word for finding the app name
$filter_position = 0;				#set default filter position to zero
###############################################
#									MAIN BEGIN									#
###############################################

	initialize();							# checks arguments, opens input file etc.
#	parse_log();							# parses the input file
	handle_output();					# zips binaries and generates the output file


###############################################
#									MAIN END										#
###############################################

# Performs initialization
sub initialize
{
	handleArgs();
	readLogFile();
}

# Reads, stores and checks arguments
sub handleArgs
{
		my $ii = 0;
		my $result = GetOptions (											# read and store arguments
							 'i=s' => \@build_log,							# path for input file
					     'o=s' => \$what_log,								# path for output file
					     'filter=s' => \$filter_string,			# saves lines containing string in $filter_string and in position in $filter_position
					     'pos:i' => \$filter_position,
					     'ex=s' => \@exclude_list,					# excludes lines containing one of these strings
					     'zip=s' => \$zip,									# name for zip file containing binaries from parsed lines
					     'zip_prefix=s' => \$zip_prefix,		# prefix for zip files named with each application's name if $zip2zip defined
					     'zip2zip' => \$zip2zip,						# double zipping
					     'h' => \$help,											# prints usage information
				  	   'help' => \$help										# prints usage information
				    	 );
	
	# check argumens are valid and prints usage if not
	if (!$result || !scalar(@build_log) || (!$what_log && !$zip) || (!$zip && $zip2zip) ||
			!$filter_string || ($filter_position < 0) || $help ) {
		print "\nUSAGE:\n";
		print "perl parse_what_log.pl\n".
					" -i\t'full path of the build log input file'\n".
					" -o\t'full path of the parsed output file'\n".
					" -filter\t'defines the string on the line, which we want to store'\n".
					" -pos\t'defines the position for filter string on the line, 0 index is used by default'\n".
					" -ex\t'defines a list of strings, which excludes lines containing one of these strings'\n".
					" -zip\t'full path for the primary zip file'\n".
					" -zip_prefix\t'prefix for each secondary zip file name when 'double' zipping method selected'\n".
					" -zip2zip\t'enables the 'double' zipping'\n\n";

		print "Example1: Parse the build log\n";
		print "perl parse_what_log.pl -i \\log\\S60_build.log -o \\Parsed_S60_build.log -filter \"epoc32\" -pos 1\n\n";

		print "Example2: Parse the build log and zip binaries\n";
		print "perl parse_what_log.pl -i \\log\\S60_build.log -o \\Parsed_S60_build.log -filter \"\\epoc32\" -pos -zip \\zip\\S60_binaries\n\n";

		print "Example3: Parse the build log and double zip binaries\n";
		print "perl parse_what_log.pl -i \\log\\S60_build.log -o \\Parsed_S60_build.log -filter \"\\epoc32\" -pos -zip \\zip\\S60_binaries -zip2zip\n\n";

		print "Example4: Parse the build log with exclude list\n";
		print "perl parse_what_log.pl -i \\log\\S60_build.log -o \\Parsed_S60_build.log -filter \"\\epoc32\" -pos -ex \"warning\" -ex \"\\build\"\n\n";

		exit;
	}

	# disable the meaning of characters used in regular expressions
#	$parse_start_string = lc("\Q$parse_start_string\E");
	$parse_end_string = lc("\Q$parse_end_string\E");
	$filter_string = lc($filter_string);
	foreach $i (@exclude_list) {
		$i = lc("\Q$i\E");
		$exclude_list[$ii++] = $i;
	}

}

# Opens and reads the log file to memory
sub readLogFile
{
  foreach my $log_file(@build_log) {

		open (IN,"$log_file") or die "Cannot open $log_file for reading: $!";
		@lines=<IN>;
		parse_log();
		close IN;
	}
}

# Parses the input file
# Uses configuration according user input
sub parse_log {
 	foreach (@lines){
 		chomp $_;

 		if ($_=~/$parse_start_string/i || $_=~/ACTION=what/i) {						# find starting point for data
 			$start_found = 1;
 		}
 		elsif ($start_found) {
   		if ($_=~/$app_key_word/i) {								# find the path for bld.inf file
  			@app_path = split /[\\||\/]/,$_;
  			$app = $app_path[scalar(@app_path)-2];	# store application's name
   		}
   		if ($_=~/$parse_end_string/i) {						# stop fetching data
   			$start_found = 0;
   			next;
   		}
			$_ =~ s/\"//g;
			my $f = $_;
			$f = lc $f;
  		if (($filter_position == index $f,$filter_string,$filter_position) && $start_found) { # fetch valid data only
  			foreach $i (@exclude_list) {						# check if data is having illegal words
  				if ($_=~/$i/i) {
  					$skip_line = 1;
  				}
  			}
				if (!$skip_line) {
 					$_ =~ s/ /\n/g;
	      	push @{$components{$app}}, $_."\n";		# store the data under each app's name
				}
  		}
  		$skip_line = 0;
		}
	}
	@lines = ();
}

# Creates output and zip files according arguements
# Takes the list of binaries and applications fetched from input file as an input
sub handle_output {
  my $zip_cmd;

  # prepares double zipping if zip2zip enabled by user
  if ($zip2zip) {
    mkdir $temp_folder;
  }
  if ($zip_prefix && $zip_prefix !~ /_$/) { 		# use the prefix from user input
    $zip_prefix .= "_";
  }

	# generates zipping command if zip enabled by user
  if ($zip) {
    $zip_cmd ="zip -r ";
    if ($zip2zip) {
      $zip_cmd .="-j -m ";
    }
    if ($zip !~ /\.zip$/i) {
    	$zip .= ".zip";
    }
    $zip_cmd .="${zip} -@ ";

  	open (ZIP, "| $zip_cmd");
  }

  if ($what_log) {
  	open (OUT_WHAT,">$what_log") or die "Cannot open $what_log for writing: $!";	# create or open output file
  }

	# goes through all apps and their binary lists
	foreach my $app (sort keys %components) {
	  if ($zip2zip) {
			my $zip2zip_cmd = "call zip -r ${temp_folder}${zip_prefix}${app}.zip -@ "; # generate double zipping command
    	open (ZIP2ZIP, "| $zip2zip_cmd");
      print ZIP2ZIP @{$components{$app}};				# performs the zipping in temp folder	
    	close ZIP2ZIP;
	  }
	  elsif ($zip) {
      print ZIP @{$components{$app}};						# performs the normal zipping
	  }
	  if ($what_log) {
      print OUT_WHAT @{$components{$app}};			# prints the binary list to output file
	  }
	}
  if ($what_log) {
    	close OUT_WHAT;
  }

  if ($zip2zip && $zip) {
    print ZIP $temp_folder."*\n";								# zips the zip files in temp folder
  }
  if ($zip) {
  	close ZIP;
  }
  if ($zip2zip) {
    rmdir $temp_folder;
  }
}
# EOF