/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/01/26 21:32:22 $
 * $Revision: 1.24 $
 */

#ifndef _MSL_MATH_K63D_H
#define _MSL_MATH_K63D_H

#ifndef _MSL_CMATH
#error This header may only be included from <cmath>
#endif

#if _MSL_C99											/*- mm 030702 -*/
#if _MSL_MATH_ERRHANDLING & MATH_ERRNO
	#include <cerrno>
#endif
#endif /* _MSL_C99 */

_MSL_BEGIN_NAMESPACE_STD

	typedef float float_t;
	typedef long double double_t;	

_MSL_END_NAMESPACE_STD

/*	Private functions. */

_MSL_BEGIN_EXTERN_C

	/* Get the relationship between two numbers */
	_MSL_INLINE short _MSL_MATH_CDECL __relation(double , double );
	_MSL_IMP_EXP_C unsigned int _MSL_MATH_CDECL __count_leading_zero(unsigned int x);


	/*	3DNow! does not support unordered comparisons, 
		so we must still use the x87 code for this. */
		
	#define __EQUAL 0x4000
	#define __GREATERTHAN 0
	#define __LESSTHAN 0x0100
	#define __UNORDERED 0x4500

	#if _MSL_USE_INLINE

		#pragma only_std_keywords off 		/* allow asm */
		#pragma volatile_asm off			/* optimize inline assembly */

		_MSL_INLINE short _MSL_MATH_CDECL __relation(double x, double y) _MSL_CANT_THROW
		{
			short 	__ret;
			asm
			{
			fld       qword ptr x
			fcomp     qword ptr y
			fnstsw    ax
			and       ax,__UNORDERED
			mov		  __ret, ax
			fclex     
			}
			return	__ret;
		}

		_MSL_INLINE unsigned int _MSL_MATH_CDECL __count_leading_zero(unsigned int x) _MSL_CANT_THROW
		{
			unsigned int bits;
			if (x == 0)
				return 32U;
			asm bsr eax, x
			asm mov bits, eax
			return 31U - bits;
		
		}

		_MSL_INLINE	int _MSL_MATH_CDECL isgreater(double x, double y)	   _MSL_CANT_THROW { return (!__relation(x, y)); }
		_MSL_INLINE	int _MSL_MATH_CDECL isgreaterequal(double x, double y) _MSL_CANT_THROW { return (!((__relation(x, y)>>8) & 1)); }
		_MSL_INLINE	int _MSL_MATH_CDECL isless(double x, double y)		   _MSL_CANT_THROW { return (__relation(x, y) == __LESSTHAN); }
		_MSL_INLINE	int _MSL_MATH_CDECL islessequal(double x, double y)    _MSL_CANT_THROW { return ((__relation(x, y)%__UNORDERED) != 0); }  /* pmk 020617 */
		_MSL_INLINE	int _MSL_MATH_CDECL islessgreater(double x, double y)  _MSL_CANT_THROW { return (__relation(x, y) <= __LESSTHAN); }
		_MSL_INLINE	int _MSL_MATH_CDECL isunordered(double x, double y)    _MSL_CANT_THROW { return (__relation(x, y) == __UNORDERED); }
	
		#pragma volatile_asm reset			/* reset inline assembly optimization */
		#pragma only_std_keywords reset		/* no more asm */

	#endif /*_MSL_USE_INLINE*/
	
_MSL_END_EXTERN_C

/*	Standard functions. */

/*	bring in x87 double-precision asm code */
#include <math_x87_inlines.h>

/*	bring in generic single-precision FP inlines */
#include <AMD_K63DNow_math.h>

_MSL_BEGIN_NAMESPACE_STD
_MSL_BEGIN_EXTERN_C

	#if _MSL_USE_INLINE

		#pragma c9x on					// allow // comments
		#pragma volatile_asm off		// optimize inline assembly
		#pragma only_std_keywords off	// allow asm
		#pragma gcc_extensions on		// allow local arrays with initializers
		
		#if 0 
		#pragma mark -
		#pragma mark === Inline 3DNow! functions ===
		#endif

		#if _MSL_USE_DEFAULT_LIBS && !defined(_MSL_BUILDING_DLL)
			#ifndef _DLL
				#pragma comment(lib, "amd_sp_mathlib.lib")
			#endif
		#endif

		static __inline void _MSL_MATH_CDECL __sincos( float x, const float* v) _MSL_CANT_THROW
		{    	                                          // sin(x)=sin(n*pi/4 + frac_part)=sin(n*pi/4)cos(frac_part)+ cos(n*pi/4)sin(frac_part)
		    float z=__four_over_pi*x ;
		    _INT32 n=(_INT32)z;
			extern _MSL_IMP_EXP_C const float __sincos_poly[];  
			extern _MSL_IMP_EXP_C const float __four_over_pi_m1[];  
			extern _MSL_IMP_EXP_C const float __tan_on_octant[];
			
			// get explicit pointers to possibly-dllimported data to
			// avoid warning from compiler
			register const float *__sincos_poly_ptr = __sincos_poly;
			register const float *__tan_on_octant_ptr = __tan_on_octant;
			
			// note fract_part is the remainder(mod(pi/4))
			// i.e. the actual arg that will be evaluated is frac_part*(pi/4)
			float frac_part= ((((x - (float)n) + __four_over_pi_m1[0]*x) + __four_over_pi_m1[1]*x)+
		                                          __four_over_pi_m1[2]*x) + __four_over_pi_m1[3]*x/*) +
		                                           __four_over_pi_m1[4]*x  */; 
			float xsq=frac_part*frac_part;
			float x_vect[2]={xsq, xsq};
			float last_mult[2]={1.0f,frac_part};

			n&=0x00000007;  // n is now the octant in which x resides

			// calculate sin and cos in parallel
			asm                  
			{
				movq mm0,x_vect[0]
				movq mm1,mm0
				mov edx, __sincos_poly_ptr
				pfmul mm0,qword ptr [edx+0]
				mov ecx,n
				pfadd mm0,qword ptr [edx+8]
				pfmul mm0,mm1
				imul ecx,12
				pfadd mm0,qword ptr [edx+16]
				pfmul mm0,mm1
				mov eax, v
				pfadd mm0,qword ptr [edx+24]
				pfmul mm0,mm1
				pfadd mm0,qword ptr [edx+32]
				pfmul mm0,last_mult[0]                
				                                  // now mm0= | cos(frac_part) | sin(frac_part) |
				                                  // we now use the trig identities
				                                  // cos(x)=cos(n*pi/4 + frac_part)=cos(n*pi/4)cos(frac_part)- sin(n*pi/4)sin(frac_part)
				                                  // sin(x)=sin(n*pi/4 + frac_part)=sin(n*pi/4)cos(frac_part)+ cos(n*pi/4)sin(frac_part)
				movq  mm1,mm0                         // note this step is very awkward on the K6 and can be completed much easier
				                                  // using the K7 instruction pfswapd
				mov edx,__tan_on_octant_ptr
				pfmul mm0,qword ptr [edx+ecx+4]   // mm0= |cos(n*pi/4)cos(frac_part) | sin(n*pi/4)sin(frac_part) |  
				pxor mm0,0x8000000000000000       // mm0= |cos(n*pi/4)cos(frac_part) | -sin(n*pi/4)sin(frac_part) |              
				pfacc mm0,mm0                         // mm0= |cos(x) | cos(x)|   
				movd  dword ptr[eax+4],mm0            // v[1]=sin(x)

				pfmul mm1,qword ptr [edx+ecx]     // mm1= |sin(n*pi/4)cos(frac_part) | cos(n*pi/4)sin(frac_part) | 
				pfacc mm1,mm1                         // mm1= |sin(x) | sin(x)|  
				movd  dword ptr[eax],mm1                        // v[0]=sin(x)
			}
			return ;  
		}

		_MSL_INLINE  float _MSL_MATH_CDECL tanf(float x) _MSL_CANT_THROW
		{
			float z[2];  // z[0] will hold sin(x) and z[1] cos(x)
			__sincos(x,&z[0]);
			return z[0]/z[1] ; 
		}

		_MSL_INLINE float _MSL_MATH_CDECL sqrtf( float x) _MSL_CANT_THROW
		{
		  asm{
				movd 		mm0, x;
				pfrsqrt 	mm1, mm0;
				movq		mm2, mm1;
				pfmul		mm1, mm1;
				pfrsqit1	mm1, mm0;
				pfrcpit2	mm1, mm2;
				pfmul		mm0, mm1;
				movd		x, mm0;
			}
			
			return x;
		}

		_MSL_INLINE float _MSL_MATH_CDECL fabsf(float x) _MSL_CANT_THROW
		{
			__HI(x)&=0x7fffffff;
			return  x;
		}

		#pragma gcc_extensions reset		/* no more local initializers */
		#pragma c9x reset					/* allow // comments */
		#pragma volatile_asm reset			/* no more optimization inline assembly */
		#pragma only_std_keywords reset		/* no more asm */

		#if 0 
		#pragma mark -
		#pragma mark === Plain old conversion stubs ===
		#endif

		/* unimplemented float routines */
		_MSL_INLINE float _MSL_MATH_CDECL
			acoshf(float x) _MSL_CANT_THROW { return (float)(acosh)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			asinhf(float x) _MSL_CANT_THROW { return (float)(asinh)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			atanhf(float x) _MSL_CANT_THROW { return (float)(atanh)((double)(x)); }
		#if 0
		_MSL_INLINE float _MSL_MATH_CDECL
			cbrtf(float x) _MSL_CANT_THROW { return (float)(cbrt)((double)(x)); }
		#endif
		_MSL_INLINE float _MSL_MATH_CDECL
			copysignf(float x, float y) _MSL_CANT_THROW { return (float)(copysign)((double)(x), (double)(y)); }
		#if !__INTEL__
		_MSL_INLINE float _MSL_MATH_CDECL
			erfcf(float x) _MSL_CANT_THROW { return (float)(erfc)((double)(x)); }
		#endif
		#if !__INTEL__
		_MSL_INLINE float _MSL_MATH_CDECL
			erff(float x) _MSL_CANT_THROW { return (float)(erf)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			exp2f(float x) _MSL_CANT_THROW { return (float)(exp2)((double)(x)); }
		#endif
		_MSL_INLINE float _MSL_MATH_CDECL
			expm1f(float x) _MSL_CANT_THROW { return (float)(expm1)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			fdimf(float x, float y) _MSL_CANT_THROW { return (float)(fdim)((double)(x), (double)(y)); }
		#if 0
		_MSL_INLINE float _MSL_MATH_CDECL
			fmaf(float x, float y, float z) _MSL_CANT_THROW { return (float)(fma)((double)(x), (double)(y), (double)(z)); }
		#endif
		_MSL_INLINE float _MSL_MATH_CDECL
			fmaxf(float x, float y) _MSL_CANT_THROW { return (float)(fmax)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			fminf(float x, float y) _MSL_CANT_THROW { return (float)(fmin)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			hypotf(float x, float y) _MSL_CANT_THROW { return (float)(hypot)((double)(x), (double)(y)); }
		_MSL_INLINE int _MSL_MATH_CDECL
			ilogbf(float x) _MSL_CANT_THROW { return (ilogb)((double)(x)); }
		_MSL_INLINE long long _MSL_MATH_CDECL
			llrintf(float x) _MSL_CANT_THROW { return (llrint)((double)(x)); }
		_MSL_INLINE long long _MSL_MATH_CDECL
			llroundf(float x) _MSL_CANT_THROW { return (llround)((double)(x)); }
		#if !__INTEL__
		_MSL_INLINE float _MSL_MATH_CDECL
			lgammaf(float x) _MSL_CANT_THROW { return (float)(lgamma)((double)(x)); }
		#endif
		_MSL_INLINE float _MSL_MATH_CDECL
			log1pf(float x) _MSL_CANT_THROW { return (float)(log1p)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			logbf(float x) _MSL_CANT_THROW { return (float)(logb)((double)(x)); }
		_MSL_INLINE long _MSL_MATH_CDECL
			lrintf(float x) _MSL_CANT_THROW { return (lrint)((double)(x)); }
		_MSL_INLINE long _MSL_MATH_CDECL
			lroundf(float x) _MSL_CANT_THROW { return (lround)((double)(x)); }
		#if 0
		_MSL_INLINE float _MSL_MATH_CDECL
			nanf(const char* x) _MSL_CANT_THROW { return (float)(nan)((x)); }
		#endif
		_MSL_INLINE float _MSL_MATH_CDECL
			nearbyintf(float x) _MSL_CANT_THROW { return (float)(nearbyint)((double)(x)); }
		#if 0
		_MSL_INLINE float _MSL_MATH_CDECL
			nextafterf(float x, float y) _MSL_CANT_THROW { return (float)(nextafter)((double)(x), (double)(y)); }
		#endif
		#if 0
		_MSL_INLINE float _MSL_MATH_CDECL
			nexttowardf(float x, long double y) _MSL_CANT_THROW { return (float)(nexttoward)((double)(x), (y)); }
		#endif
		_MSL_INLINE float _MSL_MATH_CDECL
			remainderf(float x, float y) _MSL_CANT_THROW { return (float)(remainder)((double)(x), (double)(y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			remquof(float x, float y, int* z) _MSL_CANT_THROW { return (float)(remquo)((double)(x), (double)(y), (z)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			rintf(float x) _MSL_CANT_THROW { return (float)(rint)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			roundf(float x) _MSL_CANT_THROW { return (float)(round)((double)(x)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			scalblnf(float x, long int y) _MSL_CANT_THROW { return (float)(scalbln)((double)(x), (y)); }
		_MSL_INLINE float _MSL_MATH_CDECL
			scalbnf(float x, int y) _MSL_CANT_THROW { return (float)(scalbn)((double)(x), (y)); }
		#if 0
		_MSL_INLINE float _MSL_MATH_CDECL
			tgammaf(float x) _MSL_CANT_THROW { return (float)(tgamma)((double)(x)); }
		#endif
		_MSL_INLINE float _MSL_MATH_CDECL
			truncf(float x) _MSL_CANT_THROW { return (float)(trunc)((double)(x)); }

		/* on x86, scalbn/scalbln are identical to ldexp */
		_MSL_INLINE double _MSL_MATH_CDECL
			scalbn(double x, int y) _MSL_CANT_THROW { return ldexp(x,y); }
		_MSL_INLINE double _MSL_MATH_CDECL
			scalbln(double x, long int y) _MSL_CANT_THROW { return ldexp(x,(int)y); }

		/* long double stubs */
		_MSL_INLINE long double _MSL_MATH_CDECL
			acoshl(long double x) _MSL_CANT_THROW { return (long double)(acosh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			acosl(long double x) _MSL_CANT_THROW { return (long double)(acos)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			asinhl(long double x) _MSL_CANT_THROW { return (long double)(asinh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			asinl(long double x) _MSL_CANT_THROW { return (long double)(asin)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			atan2l(long double y, long double x) _MSL_CANT_THROW { return (long double)(atan2)((double)(y), (double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			atanhl(long double x) _MSL_CANT_THROW { return (long double)(atanh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			atanl(long double x) _MSL_CANT_THROW { return (long double)(atan)((double)(x)); }
	/*	#if 0 */
		_MSL_INLINE long double _MSL_MATH_CDECL
			cbrtl(long double x) _MSL_CANT_THROW { return (long double)(cbrt)((double)(x)); }
	/*	#endif */
		_MSL_INLINE long double _MSL_MATH_CDECL
			ceill(long double x) _MSL_CANT_THROW { return (long double)(ceil)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			copysignl(long double x, long double y) _MSL_CANT_THROW { return (long double)(copysign)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			coshl(long double x) _MSL_CANT_THROW { return (long double)(cosh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			cosl(long double x) _MSL_CANT_THROW { return (long double)(cos)((double)(x)); }
		#if !__INTEL__
		_MSL_INLINE long double _MSL_MATH_CDECL
			erfcl(long double x) _MSL_CANT_THROW { return (long double)(erfc)((double)(x)); }
		#endif
		#if !__INTEL__
		_MSL_INLINE long double _MSL_MATH_CDECL
			erfl(long double x) _MSL_CANT_THROW { return (long double)(erf)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			exp2l(long double x) _MSL_CANT_THROW { return (long double)(exp2)((double)(x)); }
		#endif
		_MSL_INLINE long double _MSL_MATH_CDECL
			expl(long double x) _MSL_CANT_THROW { return (long double)(exp)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			expm1l(long double x) _MSL_CANT_THROW { return (long double)(expm1)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fabsl(long double x) _MSL_CANT_THROW { return (long double)(fabs)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fdiml(long double x, long double y) _MSL_CANT_THROW { return (long double)(fdim)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			floorl(long double x) _MSL_CANT_THROW { return (long double)(floor)((double)(x)); }
		#if 0
		_MSL_INLINE long double _MSL_MATH_CDECL
			fmal(long double x, long double y, long double z) _MSL_CANT_THROW { return (long double)(fma)((double)(x), (double)(y), (double)(z)); }
		#endif
		_MSL_INLINE long double _MSL_MATH_CDECL
			fmaxl(long double x, long double y) _MSL_CANT_THROW { return (long double)(fmax)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fminl(long double x, long double y) _MSL_CANT_THROW { return (long double)(fmin)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			fmodl(long double x, long double y) _MSL_CANT_THROW { return (long double)(fmod)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			frexpl(long double x, int* y) _MSL_CANT_THROW { return (long double)(frexp)((double)(x), (y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			hypotl(long double x, long double y) _MSL_CANT_THROW { return (long double)(hypot)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			ldexpl(long double x, int y) _MSL_CANT_THROW { return (long double)(ldexp)((double)(x), (y)); }
		#if !__INTEL__
		_MSL_INLINE long double _MSL_MATH_CDECL
			lgammal(long double x) _MSL_CANT_THROW { return (long double)(lgamma)((double)(x)); }
		#endif
		_MSL_INLINE long long _MSL_MATH_CDECL
			llrintl(long double x) _MSL_CANT_THROW { return (llrint)((double)(x)); }
		_MSL_INLINE long long _MSL_MATH_CDECL
			llroundl(long double x) _MSL_CANT_THROW { return (llround)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			log10l(long double x) _MSL_CANT_THROW { return (long double)(log10)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			log1pl(long double x) _MSL_CANT_THROW { return (long double)(log1p)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			log2l(long double x) _MSL_CANT_THROW { return (long double)(log2)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			logbl(long double x) _MSL_CANT_THROW { return (long double)(logb)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			logl(long double x) _MSL_CANT_THROW { return (long double)(log)((double)(x)); }
		_MSL_INLINE long _MSL_MATH_CDECL
			lrintl(long double x) _MSL_CANT_THROW { return (lrint)((double)(x)); }
		_MSL_INLINE long _MSL_MATH_CDECL
			lroundl(long double x) _MSL_CANT_THROW { return (lround)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			modfl(long double x, long double* iptr) _MSL_CANT_THROW {
		  double iptrd;
		  long double result = (long double)modf((double)x, &iptrd);
		  *iptr = (long double)iptrd;
		  return result;
		}
		#if 0
		_MSL_INLINE long double _MSL_MATH_CDECL
			nanl(const char* x) _MSL_CANT_THROW { return (long double)(nan)((x)); }
		#endif
		_MSL_INLINE long double _MSL_MATH_CDECL
			nearbyintl(long double x) _MSL_CANT_THROW { return (long double)(nearbyint)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			nextafterl(long double x, long double y) _MSL_CANT_THROW { return (long double)(nextafter)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			nexttowardl(long double x, long double y) _MSL_CANT_THROW { return (long double)(nexttoward)((double)(x), (y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			powl(long double x, long double y) _MSL_CANT_THROW { return (long double)(pow)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			remainderl(long double x, long double y) _MSL_CANT_THROW { return (long double)(remainder)((double)(x), (double)(y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			remquol(long double x, long double y, int* z) _MSL_CANT_THROW { return (long double)(remquo)((double)(x), (double)(y), (z)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			rintl(long double x) _MSL_CANT_THROW { return (long double)(rint)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			roundl(long double x) _MSL_CANT_THROW { return (long double)(round)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			scalblnl(long double x, long int y) _MSL_CANT_THROW { return (long double)(scalbln)((double)(x), (y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			scalbnl(long double x, int y) _MSL_CANT_THROW { return (long double)(scalbn)((double)(x), (y)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			sinhl(long double x) _MSL_CANT_THROW { return (long double)(sinh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			sinl(long double x) _MSL_CANT_THROW { return (long double)(sin)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			sqrtl(long double x) _MSL_CANT_THROW { return (long double)(sqrt)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			tanhl(long double x) _MSL_CANT_THROW { return (long double)(tanh)((double)(x)); }
		_MSL_INLINE long double _MSL_MATH_CDECL
			tanl(long double x) _MSL_CANT_THROW { return (long double)(tan)((double)(x)); }
		#if 0
		_MSL_INLINE long double _MSL_MATH_CDECL
			tgammal(long double x) _MSL_CANT_THROW { return (long double)(tgamma)((double)(x)); }
		#endif
		_MSL_INLINE long double _MSL_MATH_CDECL
			truncl(long double x) _MSL_CANT_THROW { return (long double)(trunc)((double)(x)); }

	#endif /* _MSL_USE_INLINE */

_MSL_END_EXTERN_C
_MSL_END_NAMESPACE_STD

#endif /* _MSL_MATH_K63D_H */

/* Change record: 
 * mf  000407 cast everything to gid rid of implicit type conversion warnings
 * cc  010410 updated to new namespace macros 	
 * cc  011203 Added _MSL_CDECL for new name mangling 
 * ejs 020124 Moved to new file, rearranged code
 * ejs 020130 Added float_t and double_t to namespace std
 * ejs 020314 Change __SET_ERRNO__ to check _MSL_MATH_ERRHANDLING
 * ejs 020402 Rename locals/labels to have underscores to avoid conflict with macros
 * pmk 020617 bug fix in islessequal(), output should be 0 or 1
 * pmk 021903 Added no throw spec to functions
 * ejs 020807 Added nexttoward/nextafter
 * ejs 020809 Renamed relation to __relation, removed nonstandard isequal()
 * ejs 020809 Be sure to include amd_sp_mathlib.lib
 * cc  030217 Added in the inline of cbrtl 
 * mm  030702 Added C99 wrappers
 */
