/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/01/26 21:32:34 $
 * $Revision: 1.21 $
 */

#ifndef _MSL_STRING_X86_H
#define _MSL_STRING_X86_H

#include <ansi_parms.h>
#include <cstring>

_MSL_BEGIN_EXTERN_C

// compute length of a zero terminated string

size_t _MSL_CDECL strlen(const char *s)
{
	register size_t	len = 0;
	
    asm
    {
		xor		eax, eax				// byte to scan for
    	mov		edi, s					// get the string
		mov		ecx, 0xffffffff			// max string length
		    	
    lab:
		repne	scasb					// scan for that zero
		
    out:
		neg		ecx						// length is -ecx+2
		lea		len, -2[ecx]
    }
    
    return len;
}



// Copy a string up to the zero terminating byte

char * _MSL_CDECL strcpy(char * _MSL_RESTRICT dest, const char * _MSL_RESTRICT src)
{
    asm
    {
        mov		edi, dest				// get the dest
        mov		esi, src				// get the source
        
    lab:
        lodsb							// move a byte
        stosb
		test	al, al					// was it zero
		jne		lab						// loop if not
    }
    
    return dest;
}


// Copy a string up to the zero terminating byte or a max len

char * _MSL_CDECL strncpy(char * _MSL_RESTRICT dest, const char * _MSL_RESTRICT src, size_t max)
{
    asm
    {
        mov		ecx, max				// get the max count
        mov		edi, dest				// and the dest
        mov		esi, src				// and the source
		add		ecx, edi				// adress of last byte to move
		jmp		test					// skip in case no bytes to move

		align		       
    lab:
        lodsb							// move a byte
        stosb
		test	al, al					// was it zero?
		je		pad						// if so, pad the rest
		
	test:
		cmp		edi, ecx				// check for done
		jb		lab						// and loop if more
		jmp		end

    // padding the string. ecx point to the end of the string,
    // al is zero
    
		align
    pad:
		sub		ecx, edi				// get count to pad
		rep		stosb					// and do it

	end:
    }
    
    return dest;
}


// Concat one string to another

char * _MSL_CDECL strcat(char * _MSL_RESTRICT dest, const char * _MSL_RESTRICT src)
{
    asm
    {
        mov		edi, dest				// get dest pointer
		xor		eax, eax
		mov		esi, src				// get the source
		              
        mov		ecx, 0xffffffff			// scan to end of dest string
        repne 	scasb
		dec		edi
		
	lab:
		lodsb							// move a byte
		stosb
		test	al, al					// was it zero?
		jne		lab						// loop if so
    }
    
    return dest;
}


// concat one string to another to a max length

char * _MSL_CDECL strncat(char * _MSL_RESTRICT dest, const char * _MSL_RESTRICT src, size_t max)
{
    asm 
    {
		mov		edx, max				// get count
		test	edx, edx				// get out quick if it is zero
		je		done

		mov		edi, dest				// get dest	
    	mov		esi, src				// get source
		mov		ecx, 0xffffffff			// max strlen
		xor		eax, eax
	
		repne	scasb					// scan to end of dest
		dec		edi						// back up to the zero byte
		
	copy:
	
		lodsb							// move a byte
		stosb
		test	al, al					// was it 0
		je		done					// time to quit
		
		sub		edx, 1					// decrement count
		jne		copy					// and loop if there is more
				
		xor		eax, eax				// on fall through, append a 0
		stosb
		
	done:	
    }
    
    return dest;
}


// Compare to zero termianted string

int _MSL_CDECL strcmp(const char * s1, const char *s2)
{
	int result = 0;

    asm
    {
    	mov		ecx, s1				// get str1
    	mov		edx, s2				// and str2

    lab:
    	mov		al, [ecx]			// get a byte from str1
    	cmp		al, [edx]			// compare it to str2
    	jne		out					// if not equal we are done
    	
    	inc		ecx					// bump the pointers
    	inc		edx
    	
		test	al, al				// and check for end of string
		jne		lab					// looping if not
    	jmp		done				// return 0 if they are
    	
    	align
    out:
    	movzx	edx, byte ptr [edx]
		movzx	eax, al
    	sub		eax, edx
		mov		result, eax
		
	done:
    }
    
    return result;
}


// Compare two strings up to a max len

int _MSL_CDECL strncmp(const char *s1, const char *s2, size_t n)
{
	int result = 0;
	
    asm
    {
		mov		edx, n				// get the count
    	mov		ebx, s1				// get str1
    	mov		ecx, s2				// and str2
    	
		test	edx, edx			// if count is zero
		je		done				// they are equal
		
    lab:
    	mov		al, [ebx]			// get a byte from str1
    	cmp		al, [ecx]			// compare it to str2
    	jne		out					// if not equal we are done
    	
		test	al, al				// and check for end of string
		je		out					// looping if not
		
    	inc		ebx					// bump the pointers
    	inc		ecx
    	
    	sub		edx,1				// decrement count
    	jne		lab					// and loop if not done
    	jmp		done
    	
    out:
    	movzx	ecx, byte ptr [ecx]
		movzx	eax, al
    	sub		eax, ecx
    	mov		result, eax
	done:
    }
    
    return result;
}


// Search for a char in a string

char * _MSL_CDECL strchr(const char *s, int c)
{
	register char *loc = 0;
	
    asm
    {
    	mov		esi, s				// get source
    	mov		ecx, c				// get char
		
    lab:
		lodsb						// load a byte
		cmp		al, cl				// check it
		je		out					// if equal we found it
		
		test	al, al				// check for zero terminator
		jne		lab
		jmp		done				// not found, return 0    	
		
		align
	out:
		lea		loc, -1[esi]		// found, return address
		
    done:
    }
    
    return loc;
}

_MSL_END_EXTERN_C

#endif /* _MSL_STRING_X86_H */

/* Change record:
 * hh  971206 Added include guards
 * hh  971206 Added namespace support
 * hh  980122 Added <cstring> for prototypes
 * bk  980806 Fixes bug in strcmp and strncmp in which string compares
 *            did not work properly  with bytes in the range of 128...255
 *            (-128...-1 signed)
 * blc 990309 Changed to use non-declspec(naked) implementation
 * blc 990310 Fixed code in strncmp and strlen (left in naked's pops)
 * ejs 010320 Add "register" keyword to build at opt 0
 * cc  010410 updated to new namespace macros
 * JWW 010618 Use cname headers exclusively to prevent namespace pollution in C++
 * cc  011203 Added _MSL_CDECL for new name mangling 
 * JWW 020917 Added _MSL_RESTRICT to get the restrict type specifier for certain C99 functions
 */