/*
* Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
*
*/
#include <boost/regex.hpp>
#include "symbolprocessunit.h"
#include "e32image.h"
#define MAX_LINE 65535

#if defined(__LINUX__)
#define PATH_SEPARATOR '/'
#else
#define PATH_SEPARATOR '\\'
#endif

void CommenSymbolProcessUnit::ProcessExecutableFile(const string& aFile)
{
	ResetContentLog();
	char str[MAX_LINE];
	string outString;
	outString = "\nFrom    ";
	outString += aFile + "\n\n";
	iSymbolContentLog.push_back(outString);
	string mapFile2 = aFile+".map";
	size_t dot = aFile.rfind('.');
	string mapFile = aFile.substr(0,dot)+".map";
	ifstream fMap;
	fMap.open(mapFile2.c_str());
	if(!fMap.is_open()) {
		fMap.open(mapFile.c_str());
	}

	if(!fMap.is_open()) {
		sprintf(str, "%s\nWarning: Can't open \"%s\" or \"%s\"\n",aFile.c_str(),mapFile2.c_str(),mapFile.c_str());
		iStdoutLog.push_back(str);
	    int binSize = GetSizeFromBinFile(aFile);
	    memset(str,0,sizeof(str));
	    sprintf(str,"%04x", binSize);
	    outString = "00000000    ";
	    outString += str;
	    outString += "    ";
	    outString += aFile.substr(aFile.rfind(PATH_SEPARATOR)+1)+"\n";
	    iSymbolContentLog.push_back(outString);
	}
	else {
		if(!fMap.good()) fMap.clear();
	    boost::regex regARMV5("ARMV5", boost::regex::icase);
	    boost::regex regGCCEoARMV4("(GCCE|ARMV4)", boost::regex::icase);
	    boost::cmatch what;
	    if(regex_search(aFile, what, regARMV5)) {
	        ProcessArmv5File(aFile, fMap);
	    }
	    else if(regex_search(aFile, what, regGCCEoARMV4)) {
	        ProcessGcceOrArm4File(aFile, fMap);
	    }
	    else {
	        sprintf(str, "\nWarning: cannot determine linker type used to create %s\n",aFile.c_str());
	        iStdoutLog.push_back(str);
	        outString = "00000000    0000    ";
	        outString += aFile.substr(aFile.rfind(PATH_SEPARATOR)+1)+"\n";
	        iSymbolContentLog.push_back(outString);
	        }
	    }
}
void CommenSymbolProcessUnit::ProcessDataFile(const string& aFile)
{
	ResetContentLog();
	string line = "\nFrom    "+aFile+"\n\n00000000    0000    "+aFile.substr(aFile.rfind(PATH_SEPARATOR)+1)+"\n";
	iSymbolContentLog.push_back(line);
}
void CommenSymbolProcessUnit::FlushStdOut(ostream& aOut)
{
	for(int i = 0; i < (int) iStdoutLog.size(); i++)
	{
		aOut << iStdoutLog[i];
	}
}
void CommenSymbolProcessUnit::FlushSymbolContent(ostream &aOut)
{
	for(int i = 0; i < (int) iSymbolContentLog.size(); i++)
	{
		aOut << iSymbolContentLog[i];
	}
}
void CommenSymbolProcessUnit::ResetContentLog()
{
	iStdoutLog.clear();
	iSymbolContentLog.clear();
}
void CommenSymbolProcessUnit::ProcessArmv5File( const string& fileName, ifstream& aMap ){
    aMap.seekg (0, ios::beg);
    char str[MAX_LINE];
    char outbuffer[MAX_LINE];
    string outString;
    aMap.getline(str,MAX_LINE);
    boost::cmatch what;
    boost::regex reg("^ARM Linker");
    if(!regex_search(str, what, reg)) {
        sprintf(outbuffer, "\nWarning: expecting %s to be generated by ARM linker\n", fileName.c_str());
        iStdoutLog.push_back(outbuffer);
        outString = "00000000    0000    "+fileName.substr(fileName.rfind(PATH_SEPARATOR)+1)+"\n";
        iSymbolContentLog.push_back(outString);
    }
    reg.assign("Global Symbols");
    while(aMap.getline(str,MAX_LINE)) {
        if(regex_search(str, what, reg)) {
            break;
        }
    }

    reg.assign("^\\s*(.+)\\s*0x(\\S+)\\s+[^\\d]*(\\d+)\\s+(.*)$");
    string sSym,sTmp,sSection;
    unsigned int addr,size,baseOffset = 0;
    map<unsigned int,string> syms;
    char symString[MAX_LINE];
    while(aMap.getline(str,MAX_LINE)) {
        if(regex_search(str, what, reg)) {
            sSym.assign(what[1].first,what[1].second-what[1].first);
            sTmp.assign(what[2].first,what[2].second-what[2].first);
            addr = strtol(sTmp.c_str(), NULL, 16);
            sTmp.assign(what[3].first,what[3].second-what[3].first);
            size = strtol(sTmp.c_str(), NULL, 10);
            sSection.assign(what[4].first,what[4].second-what[4].first);
            if(sSection.find("(StubCode)") != string::npos)
                size = 8;
            if(addr > 0) {
                memset(symString,0,sizeof(symString));
                sprintf(symString,"%04x    ",size);
                outString = symString;
                outString += sSym+" ";
                outString += sSection;
                if(baseOffset == 0)
                    baseOffset = addr;
                unsigned int k = addr - baseOffset;
                if( (syms.find(k) == syms.end()) || size != 0)
                    syms[k] = outString;
            }
            // end of addr>0
        }
        // end of regex_search
    }

    map<unsigned int,string>::iterator it;
    for(it = syms.begin(); it != syms.end(); it++) {
        memset(str,0,sizeof(str));
        sprintf(str,"%08x",it->first);
        outString = str;
        outString += "    ";
        outString += it->second+"\n";
        iSymbolContentLog.push_back(outString);
    }
}
void CommenSymbolProcessUnit::ProcessGcceOrArm4File( const string& fileName, ifstream& aMap ){
    aMap.seekg (0, ios_base::beg);
    char str[MAX_LINE];
    char outbuffer[MAX_LINE];
    aMap.getline(str,MAX_LINE);
    boost::cmatch what;
    boost::regex reg("^\\.text\\s+");
    while(aMap.getline(str,MAX_LINE)) {
        if(regex_search(str, what, reg)) {
            break;
        }
    }

    reg.assign("^\\.text\\s+(\\w+)\\s+\\w+");
    if(!regex_search(str, what, reg)) {
        sprintf(outbuffer, "ERROR: Can't get .text section info for \"%s\"\n",fileName.c_str());
        iStdoutLog.push_back(outbuffer);
    }
    else {
        string sTmp, sLibFile;
        sTmp.assign(what[1].first,what[1].second-what[1].first);
        unsigned int imgText = strtol(sTmp.c_str(), NULL, 16);

        reg.assign("^LONG 0x.*", boost::regex::icase);
        boost::cmatch what1;
        boost::regex reg1("^\\s(\\.text)?\\s+(0x\\w+)\\s+(0x\\w+)\\s+(.*)$", boost::regex::icase);
        boost::regex reg2("^\\s+(\\w+)\\s\\s+([a-zA-Z_].+)", boost::regex::icase);
        boost::regex reg3(".*lib\\(.*d\\d*s_?\\d{5}.o\\)$", boost::regex::icase);

        map<unsigned int,string> syms;
        unsigned int addr, len, stubhex;

        while(aMap.getline(str,MAX_LINE)) {
            if(strlen(str) == 0)
                break;
            else if(regex_search(str, what, reg1)) {
                sLibFile.assign(what[4].first,what[4].second-what[4].first);
                if(!regex_search(sLibFile, what1, reg)) {
                    sTmp.assign(what[2].first,what[2].second-what[2].first);
                    addr = strtol(sTmp.c_str(), NULL, 16);
                    sTmp.assign(what[3].first,what[3].second-what[3].first);
                    len = strtol(sTmp.c_str(), NULL, 16);
                    syms[addr+len] = "";
                    if(regex_search(sLibFile, what, reg3)) {
                        stubhex = addr;
                    }
                }
            }
            else if(regex_search(str, what, reg2)) {
                sTmp.assign(what[1].first,what[1].second-what[1].first);
                addr = strtol(sTmp.c_str(), NULL, 16);
                sTmp.assign(what[2].first,what[2].second-what[2].first);
                syms[addr] = (addr == stubhex)? ("stub "+sTmp) : sTmp;
            }
        }

        map<unsigned int,string>::iterator it = syms.begin();
        map<unsigned int,string>::iterator itp = it++;
        string outString;
        for(; it != syms.end(); itp = it++) {
            if(itp->second != "") {
                memset(str,0,sizeof(str));
                sprintf(str,"%08x    %04x    ",(itp->first-imgText), (it->first-itp->first));
                outString = str;
                outString += it->second+"\n";
                iSymbolContentLog.push_back(outString);
            }
        }
    }
}
int CommenSymbolProcessUnit::GetSizeFromBinFile( const string& fileName ){
    TInt ret = 0;
    char outbuffer[MAX_LINE];
    ifstream aIf(fileName.c_str(), ios_base::binary);
    if( !aIf.is_open() ) {
        printf(outbuffer, "Warning: Cannot open file \n");
        iStdoutLog.push_back(outbuffer);
    }
    else {
        E32ImageFile e32Image;
        TUint32 aSz;

        aIf.seekg(0,ios_base::end);
        aSz = aIf.tellg();

        e32Image.Adjust(aSz);
        e32Image.iFileSize = aSz;

        aIf.seekg(0,ios_base::beg);
        aIf >> e32Image;
        ret = e32Image.iOrigHdr->iCodeSize;
    }
    return ret;
}
