/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/01/26 21:32:33 $
 * $Revision: 1.21 $
 */

#ifndef _MSL_MATH_X87_INLINES_H
#define _MSL_MATH_X87_INLINES_H

#if !defined(_MSL_MATH_X87_H) && !defined(_MSL_MATH_K63D_H) && !defined(_MSL_MATH_SSE_H)
#error This header may only be included from <math_XXX.h>
#endif

#define __TO_INFINITY	word ptr 0x0b7f /* mask to change rounding mode to "toward +inf" on 16 bit control word*/
#define __TO_M_INFINITY	word ptr 0x077f	/* mask to change rounding mode to "toward -inf" on 16 bit control word*/
#define	__TRUNC			word ptr 0x0f7f	/* mask to change rounding mode to "toward zero" on 16 bit control word*/  

#define	__SIXTYFOURBITS	word ptr 0x037f	/* sets precision to "full" or 64 significant bits(full extended format) */
#define	__HALF			dword ptr 0.5f
#define	__ONE			dword ptr 1.0f 	/* used instead of instruction fld1 */

_MSL_BEGIN_EXTERN_C

	/*	Private functions */
	extern __declspec(naked) void __cdecl _reduce(void);

_MSL_END_EXTERN_C


_MSL_BEGIN_NAMESPACE_STD
_MSL_BEGIN_EXTERN_C

	#if _MSL_USE_INLINE

		#pragma only_std_keywords off 		/* allow asm */
		#pragma volatile_asm off			/* optimize inline assembly */

		_MSL_INLINE double _MSL_MATH_CDECL copysign(double x, double y) _MSL_CANT_THROW
		{
			double __ret;
			asm
			{
			fld qword ptr [x]       ; load x
			mov eax, dword ptr[y+4]	; just want the first byte of y
			mov ecx, dword ptr[x+4]  ; sign of x
			xor eax,ecx
			and eax,0x80000000
			jz __done
			fchs                       ; change sign of x only when 
			                        ; sign(x) != sign(y)
		__done:
			fstp __ret
			}
			return __ret;
		}
		
		_MSL_INLINE double _MSL_MATH_CDECL rint(double x) _MSL_CANT_THROW
		{
			asm
			{
			fld x
			frndint
			fstp x
			}
			return x ;
		}

		_MSL_INLINE long _MSL_MATH_CDECL lrint(double x) _MSL_CANT_THROW
		{
			long __ret;
			asm
			{
			fld x
			frndint
			fistp __ret 
			}
			return __ret ;
		}

		_MSL_INLINE long long _MSL_MATH_CDECL llrint(double x)
		{
			long long __ret;
			asm
			{
			fld x
			frndint
			fistp __ret 
			}
			return __ret ;
		}

		_MSL_INLINE double _MSL_MATH_CDECL trunc(double x)
		{
			short __TEMP ;

			asm
			{
			fld x
			fstcw __TEMP
			fldcw __TRUNC
			frndint
			fldcw __TEMP
			fstp x
			}
			return x ;
		}
		       
		_MSL_INLINE double _MSL_MATH_CDECL ceil(double x)
		{
			short __TEMP ;

			asm
			{
			fld x
			fstcw __TEMP		    ; save callers current mode
			fldcw __TO_INFINITY	; positive infldcw __SIXTYFOURBITSy(rc=2)
			frndint
			fldcw __TEMP		    ; restore callers mode
			fstp x
			}
			return x ;
		}

		_MSL_INLINE double _MSL_MATH_CDECL floor(double x)
		{
			short __TEMP ;

			asm
			{
			fld x
			fstcw __TEMP		; save callers current mode; save callers current mode
			fldcw __TO_M_INFINITY
			frndint
			fldcw __TEMP		; restore callers mode
			fstp x
			}
			return x ;
		}

		_MSL_INLINE double _MSL_MATH_CDECL round(double x)
		{
			short __TEMP ;
			asm
			{
			fld x
			ftst
			fstsw ax
			and ax,0x0100
			jne __SUBTRACT
			fadd __HALF
			jmp __G0
		__SUBTRACT:
			fsub __HALF
		__G0:
			fstcw __TEMP
			fldcw __TRUNC
			frndint
			fldcw __TEMP
			fstp x
			}
			return x;
		}

		_MSL_INLINE long _MSL_MATH_CDECL lround(double x)
		{
			short __TEMP ;
			long __ret;

			asm
			{
			fld x
			ftst
			fstsw ax
			and ax,0x0100
			jne __SUBTRACT    ; c0 gets set by ftst when x is a nan
			fadd __HALF     ; but whether you add or substract .5 from the nan is irrelevant
			jmp __GO
		__SUBTRACT:
			fsub __HALF
		__GO:    
			fstcw __TEMP
			fldcw __TRUNC
			frndint
			fistp dword ptr[__ret]
			fldcw __TEMP
			}
			return __ret;

		}

		_MSL_INLINE long long _MSL_MATH_CDECL llround(double x)
		{
			short __TEMP ;
			long long __ret;

			asm
			{
			fld x
			ftst
			fstsw ax
			and ax,0x0100
			jne __LL_SUBTRACT    ; c0 gets set by ftst when x is a nan
			fadd __HALF     ; but whether you add or substract .5 from the nan is irrelevant
			jmp __LL_GO
		__LL_SUBTRACT:
			fsub __HALF
		__LL_GO:    
			fstcw __TEMP
			fldcw __TRUNC
			frndint
			fistp qword ptr[__ret]
			fldcw __TEMP
			}
			return __ret;
		}

		/* nearbyint:

		This function is a bit of a hack since there is no way to tailor
		the frndint instruction to NOT set the inexact exception flag.  
		some may see the setting of the inexact flag as an unecessary side effect of floating
		point computation as it is well understood that we get rounded (inexact)
		results during computation.  If the user wishes to catch all meaningful floating point
		exceptions in an application I suggest the use of the fesetenv function from fenv.h 
		to mask just the inexact exception flag and unmask the rest.  nearbyint has the overhead
		of clearing the exception flag set on the status word.  This operation is computationally
		expensive.
		 */

		_MSL_INLINE double _MSL_MATH_CDECL nearbyint(double x)
		{
			char __temp[28] ; /* 28 bytes of storage for fp environment */
			asm
			{
			fld x
			frndint		 ; rounded according to current rounding mode
			fnstenv __temp
			and    word ptr[__temp+4],0xffcf ; clears inexact exception
			fldenv __temp
			fstp x         
			}
			return x;
		}


		/* scaling functions, algebraic functions (non transcendental):
		 ldexp(scalbn),frexp, modf, fmod , sqrt
		 */
		 
		_MSL_INLINE double _MSL_MATH_CDECL fmod(double x, double y)
		{
			asm
			{
			fld y
			fabs			; to insure sign of x is returned
			fld x
		__again:	
			fprem
			fnstsw ax		
			test ax, 0x0400	; check for C2 bit
			jnz __again
			fstp st(1)		; pop divisor
			fstp x
			}
			return x ;
		} 

		_MSL_INLINE double _MSL_MATH_CDECL frexp(double x, int *exponent)
		{
			asm
			{
			fld x
			ftst			; need to check if x=0 due to  behavior of fxtract
			fnstsw ax
			test ax, 0x4000	; check if C3 bit is set
			jz __normal_frexp
			mov ECX,exponent 	
			fist dword ptr[ECX]
			jmp __frexp_done
		__normal_frexp:	
			fxtract			; st=Sx st(1)=n
			fld __HALF		; 2^-1
			fmul			; scale significand,pop __HALF 
			fxch			; put exponent on top of stack
			fld1
			fadd     		; increase exponent by 1, overwrite exponent  pop 1
						
			mov ECX,exponent
			fistp dword ptr[ECX]
		__frexp_done:	
			fstp x
			}
			return x ;
		}	
			
		_MSL_INLINE double _MSL_MATH_CDECL ldexp(double x, int exponent)
		{
			asm
			{
			fild exponent	; esp=address of LDEXP
						    ; esp increments in bytes
			fld x	        ; load x
			fscale			; st=x st(1)=n
			fstp st(1)
			fstp x
			}
			return x ;
		}

	#if 0
	#pragma mark -
	#endif
	
	/*
		All of these functions are inlined intrinsically by the compiler.
		It is better for the compiler to generate them than to make them
		inline assembly functions.
	*/
	
	#if defined(__MWERKS__) && !__option(inline_intrinsics)
		
		_MSL_INLINE double _MSL_MATH_CDECL fabs(double x)
		{
			asm 
			{
			fld x
			fabs
			fstp x
			}
			return x ;
		}
		
		/* transcendentals:

		log, log10,log2, log1p, atan, atan2, atanh, acosh, asinh, exp2, expm1.
		Some of the other transcendentals such as pow are too large to inline
		and are therefore located in one of the .obj files math_wrapper.obj,
		math.obj, or fpce.obj
		*/
	
		_MSL_INLINE  double      _MSL_MATH_CDECL asin(double x)
		{
			asm
			{
			fld x
			fld st			; st=st(1)=x need to save x
			fmul st,st		; st=x^2,st(1)=x
			fld1			; st=1,st(1)=x^2,st(2)=x
			fsubr			; st=1-x^2,st(1)=x
						;      _______
			fsqrt			; st=\|x^2 - 1 ,st(1)=x
			fpatan
			fstp x
			}
			return x;
		}

		_MSL_INLINE double _MSL_MATH_CDECL atan(double x)
		 {
			asm
			{
			fld x
			fld1
			fpatan
			fstp x
			}
			return x ;
		 }

		_MSL_INLINE double      _MSL_MATH_CDECL cos(double x)
		{
			asm
			{
			fld x
			fcos
			fnstsw ax		; 16 bit register
			test ax, 0400h		; check for C2 bit
			jz  __finish
			;; argument reduction stuff
			call _reduce
			fcos
		__finish:
			fstp x
			}
		    return x;
		}

		_MSL_INLINE  double      _MSL_MATH_CDECL sin(double x)
		{
			asm
			{
			fld x
			fsin
			fnstsw ax		; 16 bit register
			test ax, 0400h		; check for C2 bit
			jz  __finish_sin
			call _reduce
			fsin
		__finish_sin:		
			fstp x
			}
			return x;
		}

		_MSL_INLINE  double      _MSL_MATH_CDECL tan(double x)
		{
			asm
			{
			fld x
			fptan           ; note st=1, st(1)=tan(x)
			fnstsw ax		; 16 bit register
			test ax, 0400h	; check for C2 bit
			jz __finish_tan
			;; ARGUMENT reduction stuff
			fstp st
			call _reduce
			fptan

		__finish_tan:
			fstp st
			fstp x
			}
			return x;
		}


		_MSL_INLINE double _MSL_MATH_CDECL atan2(double y, double x)
		  {
			asm
			{
			fld y	; load y
			fld x	; st= st(1)=y
			fpatan			; atan(y/x)
			fstp x
			}
			return x ;
		 }

	#endif /* !__option(inline_intrinsics) */

	/*
		The compiler does not set errno, so any functions
		that need to do this must be inlined here.
	*/
	
	#if defined(__MWERKS__) && ((_MSL_MATH_ERRHANDLING & MATH_ERRNO) || !__option(inline_intrinsics))

		_MSL_INLINE  double      _MSL_MATH_CDECL acos(double x)
		{
		#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO)
			if ((x < -1.0) || (x > 1.0))						/*- mm 030306 -*/							
			{
				errno = EDOM;
				
				return NAN;
			}
		#endif
			asm
			{
			fld x
			fld st			; st=st(1)=x need to save x
			fmul st,st		; st=x^2,st(1)=x
			fld1			; st=1,st(1)=x^2,st(2)=x
			fsubr			; st=1-x^2,st(1)=x
						;      _______
			fsqrt			; st=\|1 - x^2 ,st(1)=x
			fpatan
			fldpi
			fld __HALF
			fmul			; mult st by st(1) then pop
			fsubr			; pops stack(clears st(7)

			fstp x
			}
			return x;
		}

		_MSL_INLINE double _MSL_MATH_CDECL sqrt(double x)
		{
		#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO)
			if(x < 0.0)
			{
				errno = EDOM;
				return NAN;
			}
		#endif
			asm 
			{
			fld x
			fsqrt
			fstp x
			}
			return x ;
		}	

		_MSL_INLINE double _MSL_MATH_CDECL log(double x)
		{
		#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO)
			if(x < 0.0)
			{
				errno = EDOM;
				return NAN;
			}
		#endif
			asm
			{
			fld x
			fld1      	; st=1.0 st(1)=x
			fxch
			fyl2x	        ; st=1.0*log (x)
			fldln2	    ;           2
			fmul
			fstp x  
			}
			return x;
		}

		_MSL_INLINE double _MSL_MATH_CDECL log10(double  x)
		{
		#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO)
			if(x < 0.0)
			{
				errno=EDOM ;
				return NAN;
			}
		#endif 
			asm
			{
			fld x
			fld1    ; st=1.0 st(1)=x
			fxch
			fyl2x		           ; st=1.0*log (x)
			fldlg2               ;           2
			fmul
			fstp x 
			}
			return x ;
		} 

	#endif /* (_MSL_MATH_ERRHANDLING & MATH_ERRNO) || !__option(inline_intrinsics) */

	#if defined(__MWERKS__) &&  !__option(inline_intrinsics)

		_MSL_INLINE  double      _MSL_MATH_CDECL exp(double x)
		{	
			/* SPECIAL CASES */ 							/* pmk 020322 */
			if (x == -INFINITY) return 0;
			if (x == INFINITY) 	return INFINITY;
			/* ALGORITHM */
			else
			{
				asm
				{
				fld x	
				fldl2e		; st=log2(e) st(1)=x
				fmul		; st=y=x*log2(e) st(1)=log2(e)
							; works only if rounding mode is 0
							; TO_nearest or 3 TOWARD_zero(chop)
							; the default control word is usually 037fH which
							; is RC=0
				fld st		; backup I+f
				frndint		; st=I st(1)=y*log2(x)
				fxch
				fsub st,st(1)	; st(1)-st=st=f overwrites st(1)
				                ; BUG if x is infinity or very large, this subtraction will
				                ; produce a nan from inf-inf, so exp returns a nan in this case.
				f2xm1
				fld1 
				fadd		; st=2^f st(1)=2^f-1 st(2)=I
				fscale
				fstp st(1)
				fstp x
				}
			    return x;
			}
		}

	#endif /* !__option(inline_intrinsics) */

	#if 0
	#pragma mark -
	#endif
			                 
		_MSL_INLINE double _MSL_MATH_CDECL log2(double x)
		{
			asm
			{
		    fld1
			fld x
			fyl2x
			fstp x
			}
		    return x ;
		}

		_MSL_INLINE double _MSL_MATH_CDECL logb(double x)
		{
			asm
			{
			fld x
			fxtract
			fxch
			fstp x
			fstp st
			}	
			return x ;
		}
							
		_MSL_INLINE double _MSL_MATH_CDECL log1p(double x)
		{
		short __TEMP ; 
		#define	LOG_ONEPX_MAX qword ptr 0x3fda82795703f2d4ULL  	/*.4142135d   */
		#define	LOG_ONEPX_MIN qword ptr 0xbfd2bec333018866ULL	/*-.29d*/ 
			asm
			{
			fstcw __TEMP
			fldcw __SIXTYFOURBITS
			fld x           
			fcom LOG_ONEPX_MAX
			fnstsw ax
			sahf
			fld1
			fxch
			jnbe __logp1_big_num
			fcom LOG_ONEPX_MIN
			fnstsw ax
			sahf
			jc __logp1_big_num
			fyl2xp1             ; st=log
			 
			jmp __logp1_done
		__logp1_big_num:
		                        ;st=x st(1)=1
			fadd st,st(1)       ; st=1+x,st(1)=1
			fyl2x
		__logp1_done:	
		    fldln2
			fmul
			fldcw __TEMP
			fstp x
			}
			return x ;
		}

		_MSL_INLINE double _MSL_MATH_CDECL expm1(double x)
		{
			short __TEMP ;
			switch(__HI(x)&0x7ff00000)
			{
			default:
				asm
				{
				fstcw __TEMP
				fldcw __TRUNC        ; round x toward zero, if x*log2(e) < 1
				fld x                ; we use the fast algorithm
				fldl2e
				fmul
				fld  st              ; make a copy of x*log2(e)
				frndint              
				ftst
				fstsw ax            
				and ax, 0x4000       ; check if C3 bit is set, we've already filtered out nans above 
				jz __big_num         ; so we don't worry about the c0,c2 bits
				fxch
				f2xm1
				fstp st(1)
				jmp __expm1_done
			__big_num:

				fsub st(1),st       ; dangerous only if x is an infinity, these are also filtered out 
				fxch	            ; in switch(using the intel extended range makes it impossible for x*log2e 
				f2xm1               ; to overflow.
				fld1
				fadd
				fscale
				fld1
				fsub
				fstp st(1)
			__expm1_done:	
				fldcw __TEMP          ; restore the callers rounding mode
				fstp x

				} /* end of asm block	*/
				return x ; 
			 
			case 0x7ff00000:
				if(isnan(x))
				return x;

				/* x is an infinity */ 
				if(__HI(x)&0x80000000)
					return -1.0 ;  /*x=-INFINITY*/

				return (double)INFINITY ;
			}	/* end of switch*/
		}

		_MSL_INLINE double _MSL_MATH_CDECL exp2(double x)
		{
			short __TEMP ;
			asm
			{
			fstcw __TEMP
			fldcw __SIXTYFOURBITS
			fld x
		    fld st
			frndint             ;st=int part, st(1)=whole number
			fxch
			fsub st,st(1)       ;st=whole-int=frac st(1)=int part
			f2xm1
			fld1
			fadd                ; st=2^frac , st(1)=int part
			fscale
			fldcw __TEMP
			fstp st(1)
			fstp x
			}
			return x ;
		}

		_MSL_INLINE double _MSL_MATH_CDECL asinh(double x )
		{
			if(fabs(x) < 2.2204460492503131e-016) 
				return x; /* error is O(10^-48) */
			else
			{
				asm
				{
				fld x 
				fld st
				fmul st,st(1)		; st=x^2 ,  st(1)=x
				fadd __ONE	        ; st=x^2 +1 st(1)=x
				fsqrt
				fadd    			; st= x + sqrt(x^2 + 1)
				fld1
				fxch
				fyl2x
				fldln2
				fmul
				fstp x
				} 
			}
			return x;
		}

		_MSL_INLINE double _MSL_MATH_CDECL atanh(double x )
		{
		#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO) 
			if(fabs(x) > 1.0)
			{ 
			  errno=EDOM;
			  return NAN;
			} 
			else
		#endif
			if(fabs(x) < 2.2204460492503131e-016) 
				return x; /* error is O(10^-48) */

			asm
			{
			fld x
			fld st
			fsub __ONE  ; st=(x-1), st(1)=x, st2=1
			fchs
			fxch
			fadd __ONE
			fld  __ONE
			fxch
			fyl2x                ; st1*ln(st)
			fldln2
			fmul                 
			fxch
			fld  __ONE
			fxch
			fyl2x
			fldln2
			fmul
			fsubr st,st(1)
			fmul __HALF
			fstp st(1)
			fstp x
			}
			return x;
		}

		_MSL_INLINE double _MSL_MATH_CDECL acosh(double x)
		{
			asm
			{
			fld  x ; x
			fcom __ONE
			fnstsw ax
			sahf
			jb seterrno
			fld st
			fmul st,st(1)
			fsub __ONE
			fsqrt
			fadd
			fld1
			fxch
			fyl2x
			fldln2
			fmul
			fstp x
			}
			return x;
			asm
			{
			seterrno:   
			fstp st
			}
		#if (_MSL_MATH_ERRHANDLING & MATH_ERRNO) 
			errno=EDOM ;
		#endif
			return NAN ;
		}

		_MSL_INLINE double 		_MSL_MATH_CDECL tanh(double x)
		{
			return 1.0 - 2.0/(exp(2.0*x)+1.0);
		}

		/*
			from C9X(7.7.12.3): NaN arguments are treated as missing data: 
			if one argument is a NaN and the other numeric, then
			fmax chooses the numeric value. 
			fmin is analogous to fmax in its treatment of NaNs.
		*/                         

		_MSL_INLINE double _MSL_MATH_CDECL fmin(double x, double y)
		 {
			asm
			{
			fld x
			fcom y              ; st=x, y never loaded onto stack simply returned if x > y
			fstsw ax
			and ax,0x4700       ; sift out condition codes C0-C3(bits 14,8-10)          
			jz _done_fmin        ; ax=0 -> c0-c3=0 or x > y

			and ax,0x0400       ; C2 bit set iff at least one of x or y is unordered
			                 
			jz _copy_xtoy       ; copy x into y
			                 ; one or more operand at this point is 
			                 ; guaranteed to be a nan	          
			fxam                ; checking if x is a nan, if x is numeric we move x to y
			fstsw ax 
			and ax,0x4400       ; at least one of C3,C2 is set -> x is not a nan->y=nan
			jz _done_fmin        ; x is a nan when ax=0 so we return original y(skip copy)
		_copy_xtoy: 	       
			fst y               ; copy x into y(i.e. x is numeric so the and is false)     	                         
		_done_fmin:
			fstp st     
			}
			return y ;
		}

		_MSL_INLINE double _MSL_MATH_CDECL fmax ( double x, double y)
		{
			asm
			{
			fld y
			fcom x              ; st=y, x never loaded onto stack simply returned if x > y
			fstsw ax
			and ax,0x4700       ; sift out condition codes C0-C3(bits 14,8-10)          
			jz _copy_ytox       ; ax=0 -> c0-c3=0 or y > x

			and ax,0x0400       ; C2 bit set iff at least one of x or y is unordered	                     
			jz _done_fmax       
			                 ; one or more operand at this point is 
			                 ; guaranteed to be a nan	          
			fxam                ; checking if y is a nan, if y is numeric we move y to x
			fstsw ax 
			and ax,0x4400       ; if at least one of C3,C2 is set y is not a nan->x=nan
			jz _done_fmax        ; so copy y -> x
		_copy_ytox: 	 
			fst x               ; copy x into y(i.e. x is numeric so the and is false)     	                         
		_done_fmax:
			fstp st
			}

			return x;
		}

		/*
			The fdim function determines the positive difference between its arguments:
			x - y if x > y
			+0 if x <= y
			A range error may occur.

			note : unlike fmin/max the standard says nothing about how fdim 
			behaves when one or more operand is a nan(we just return x-y=nan 
			in this case), since the coding is simplified.
		*/

#if _MSL_C99											/*- mm 030528 -*/

		_MSL_INLINE double _MSL_MATH_CDECL fdim ( double x, double y )
		 {
			double z=0.0;
			asm
			{
			fld y
			fcom x              ; st=y, x never loaded onto stack simply return 0 if y > x
			fstsw ax
			and ax,0x4700       ; sift out condition codes C0-C3(bits 14,8-10)          
			jz _done_fdim        ; ax=0 -> c0-c3=0 or x > y
			fsubr x             ; st=x-y
			fst  z              ; z=x-y
		_done_fdim:                       
			fstp st
			}
			return z ;
		}

		_MSL_INLINE double _MSL_MATH_CDECL hypot( double x, double y )
		{
			short _TEMP;
			double z;
			asm
			{
			fstcw _TEMP
			fldcw __SIXTYFOURBITS
			fld x
			fld st
			fld y
			fld st			; st=st1=real  st2=st3=imag
			fxch st(2)		; st=imag, st1=real=st2=real st3=imag
			fadd			; st=real+imag st(1)=real st(2)=imag
			fmul st,st
			fxch st(2)		; st=imag, st1=real st2=(real+imag)^2
			fmul			; st=imag*real st(1)=real+imag
			fadd st,st
			fsub
			fsqrt
			fldcw _TEMP
			fstp z
			}
			return z ;
		}

		/* remainder functions
		   double_t remquo    ( double_t x, double_t y, int *quo );
		   this only works for values of x and y such that the difference
		   in binary exponents of x and y is no more than 64. Have a solution using
		   array of long longs , but haven't thought about how to do efficiently.
		*/
		_MSL_INLINE	double _MSL_MATH_CDECL remquo ( double x, double y, int *quo )
		 {
		  double z;
	   /* short __TEMP;	*/
	   /* pmk 020709    
		  Don't chng the rounding mode, leave in default.
		  In a better version of the remquo function I will 
		  take advantage the fprem1 instruction.  fprem calculates 
		  3 least significant digits of quotient (these bits are located in fp status word).
		  Only difficulty is calculating the sign of quo */
		  long long __dummy;

		  asm
		  {
			fld  y
			fld  x
		_remquo_compute_again:		
			fprem1
			fnstsw ax        ; check C2 bit
			and   ah,0x04
			jnz _remquo_compute_again
			fstp z
			; now compute the quotient and store in _big_int 
			; lea ECX,dummy ;_big_int 
			
			fld x
			fld y
			fdiv
		 /* fstcw __TEMP	*/
		 /* fldcw __TRUNC	*/
			fistp qword ptr[__dummy]  ; st=quo, st(1)=x
		 /* fldcw __TEMP	*/
			fstp st
		  }	
		  
		  *quo=(int)((unsigned int)(__dummy&0x000000007fffffff) +
		            ((((unsigned int)(__dummy>>32))&0x80000000)));
		  return z ;
		}

		_MSL_INLINE double  _MSL_MATH_CDECL remainder ( double x, double y)
		 {
			double z;
			asm
			{
			fld y
			fld x
		__compute_again:		
			fprem1
			fnstsw ax
			and   ah,0x04
			jnz __compute_again
			fstp z
			fstp st  ; clear stack
			}	

			return z;  
		}

		/*	nan can take an implementation-defined string in the form "xxxx" 
			where the bytes in "xxxx" are packed into the lower-order bits of
			the NAN constant returned.
		*/
		_MSL_INLINE double      _MSL_MATH_CDECL nan(const char* x)
		{
			double 	ret;
			asm	
			{
			mov       ecx, x
			mov       eax,dword ptr [ecx]
			test      al,0xff		/* anything besides null string? */
			jne       __skip
			xor       eax,eax
		__skip:
			mov       dword ptr [ret+0x4],0x7ff80000
			mov       dword ptr [ret],eax
			}
			return ret;
		}
#endif /* _MSL_C99 */									/*- mm 030528 -*/

		#pragma volatile_asm reset			/* reset inline assembly optimization */
		#pragma only_std_keywords reset		/* no more asm */

	#endif /*_MSL_USE_INLINE*/

#undef __TO_INFINITY
#undef __TO_M_INFINITY	
#undef	__TRUNC			

#undef	__SIXTYFOURBITS	
#undef	__HALF			
#undef	__ONE			

_MSL_END_EXTERN_C
_MSL_END_NAMESPACE_STD

#endif	/*_MSL_MATH_X87_INLINES_H*/

/* Change record:
 * ejs 020124 Created
 * ejs 020222 Exclude intrinsic functions from inlining.
 * ejs 020314 Change __SET_ERRNO__ to check _MSL_MATH_ERRHANDLING
 * pmk 020322 Added special cases - BUG still exists but won't occur when x = -inf 
 * ejs 020329 Removed unnecessary checks from previous fix
 * ejs 020402 Rename locals/labels to have underscores to avoid conflict with macros
 * pmk 020709 Made remquo more accurate by removing code that changes rounding mode  
 * pmk ?????? Need to write a version of remquo that calculates quo from fprem1 instruction
 * mm  030306 Corrected test for setting errno to EDOM in acos
 * ejs 030424 Fixed instruction in nan() that has iffy semantics 
 * mm  030528 Added _MSL_C99 wrappers
 * ejs 030800 Force inline acos() when errno result needed
 */