#include <boost/regex.hpp>
#define MAX_LINE 65535
#include "symbolgenerator.h"
#include "e32image.h"

#if defined(__LINUX__)
#define PATH_SEPARATOR '/'
#else
#define PATH_SEPARATOR '\\'
#endif

boost::mutex SymbolGenerator::iMutexSingleton;
SymbolGenerator* SymbolGenerator::iInst = NULL;
SymbolGenerator* SymbolGenerator::GetInstance(){
    iMutexSingleton.lock();
    if(iInst == NULL) {
        iInst = new SymbolGenerator();
    }
    iMutexSingleton.unlock();
    return iInst;
}
void SymbolGenerator::Release() {
    iMutexSingleton.lock();
    if(iInst != NULL) {
        delete iInst;
        iInst = NULL;
    }
    iMutexSingleton.unlock();
}
void SymbolGenerator::SetSymbolFileName( const string& fileName ){
    if(iSymFile.is_open())
        iSymFile.close();
    string s = fileName.substr(0,fileName.rfind('.'))+".symbol";
    printf("* Writing %s - ROFS symbol file\n", s.c_str());
    iSymFile.open(s.c_str());
}
void SymbolGenerator::AddFile( const string& fileName, bool isExecutable ){
    iMutex.lock();
    iQueueFiles.push(TPlacedEntry(fileName,isExecutable));
    iMutex.unlock();
    iCond.notify_all();
}
void SymbolGenerator::ProcessExecutable( const string& fileName ){
    char str[MAX_LINE];
    string outString;
    outString = "\nFrom    ";
    outString += fileName + "\n\n";
    iSymFile.write(outString.c_str(),outString.length());
    string mapFile2 = fileName+".map";
    size_t dot = fileName.rfind('.');
    string mapFile = fileName.substr(0,dot)+".map";
    ifstream fMap;
    fMap.open(mapFile2.c_str());
    if(!fMap.is_open()) {
        fMap.open(mapFile.c_str());
    }

    if(!fMap.is_open()) {
        printf("%s\nWarning: Can't open \"%s\" or \"%s\"\n",fileName.c_str(),mapFile2.c_str(),mapFile.c_str());
        int binSize = GetSizeFromBinFile(fileName);
        memset(str,0,sizeof(str));
        sprintf(str,"%04x", binSize);
        outString = "00000000    ";
        outString += str;
        outString += "    ";
        outString += fileName.substr(fileName.rfind(PATH_SEPARATOR)+1)+"\n";
        iSymFile.write(outString.c_str(),outString.length());
    }
    else {
		if(!fMap.good()) fMap.clear();
        boost::regex regARMV5("ARMV5", boost::regex::icase);
        boost::regex regGCCEoARMV4("(GCCE|ARMV4)", boost::regex::icase);
        boost::cmatch what;
        if(regex_search(fileName, what, regARMV5)) {
            ProcessArmv5File(fileName, fMap);
        }
        else if(regex_search(fileName, what, regGCCEoARMV4)) {
            ProcessGcceOrArm4File(fileName, fMap);
        }
        else {
            printf("\nWarning: cannot determine linker type used to create %s\n",fileName.c_str());
            outString = "00000000    0000    ";
            outString += fileName.substr(fileName.rfind(PATH_SEPARATOR)+1)+"\n";
            iSymFile.write(outString.c_str(),outString.length());
        }
    }
}
void SymbolGenerator::ProcessDatafile( const string& fileName ){
    string line = "\nFrom    "+fileName+"\n\n00000000    0000    "+fileName.substr(fileName.rfind(PATH_SEPARATOR)+1)+"\n";
    iSymFile.write(line.c_str(),line.length());
}
void SymbolGenerator::ProcessArmv5File( const string& fileName, ifstream& aMap ){
    aMap.seekg (0, ios::beg);
    char str[MAX_LINE];
    string outString;
    aMap.getline(str,MAX_LINE);
    boost::cmatch what;
    boost::regex reg("^ARM Linker");
    if(!regex_search(str, what, reg)) {
        printf("\nWarning: expecting %s to be generated by ARM linker\n", fileName.c_str());
        outString = "00000000    0000    "+fileName.substr(fileName.rfind(PATH_SEPARATOR)+1)+"\n";
        iSymFile.write(outString.c_str(),outString.length());
    }
    reg.assign("Global Symbols");
    while(aMap.getline(str,MAX_LINE)) {
        if(regex_search(str, what, reg)) {
            break;
        }
    }

    reg.assign("^\\s*(.+)\\s*0x(\\S+)\\s+[^\\d]*(\\d+)\\s+(.*)$");
    string sSym,sTmp,sSection;
    unsigned int addr,size,baseOffset = 0;
    map<unsigned int,string> syms;
    char symString[MAX_LINE];
    while(aMap.getline(str,MAX_LINE)) {
        if(regex_search(str, what, reg)) {
            sSym.assign(what[1].first,what[1].second-what[1].first);
            sTmp.assign(what[2].first,what[2].second-what[2].first);
            addr = strtol(sTmp.c_str(), NULL, 16);
            sTmp.assign(what[3].first,what[3].second-what[3].first);
            size = strtol(sTmp.c_str(), NULL, 10);
            sSection.assign(what[4].first,what[4].second-what[4].first);
            if(sSection.find("(StubCode)") != string::npos)
                size = 8;
            if(addr > 0) {
                memset(symString,0,sizeof(symString));
                sprintf(symString,"%04x    ",size);
                outString = symString;
                outString += sSym+" ";
                outString += sSection;
                if(baseOffset == 0) 
                    baseOffset = addr;
                unsigned int k = addr - baseOffset;
                if( (syms.find(k) == syms.end()) || size != 0) 
                    syms[k] = outString;
            }
            // end of addr>0
        }
        // end of regex_search
    }

    map<unsigned int,string>::iterator it;
    for(it = syms.begin(); it != syms.end(); it++) {
        memset(str,0,sizeof(str));
        sprintf(str,"%08x",it->first);
        outString = str;
        outString += "    ";
        outString += it->second+"\n";
        iSymFile.write(outString.c_str(),outString.length());
    }
}
void SymbolGenerator::ProcessGcceOrArm4File( const string& fileName, ifstream& aMap ){
    aMap.seekg (0, ios_base::beg);
    char str[MAX_LINE];
    aMap.getline(str,MAX_LINE);
    boost::cmatch what;
    boost::regex reg("^\\.text\\s+");
    while(aMap.getline(str,MAX_LINE)) {
        if(regex_search(str, what, reg)) {
            break;
        }
    }

    reg.assign("^\\.text\\s+(\\w+)\\s+\\w+");
    if(!regex_search(str, what, reg)) {
        printf("ERROR: Can't get .text section info for \"%s\"\n",fileName.c_str());
    }
    else {
        string sTmp, sLibFile;
        sTmp.assign(what[1].first,what[1].second-what[1].first);
        unsigned int imgText = strtol(sTmp.c_str(), NULL, 16);

        reg.assign("^LONG 0x.*", boost::regex::icase);
        boost::cmatch what1;
        boost::regex reg1("^\\s(\\.text)?\\s+(0x\\w+)\\s+(0x\\w+)\\s+(.*)$", boost::regex::icase);
        boost::regex reg2("^\\s+(\\w+)\\s\\s+([a-zA-Z_].+)", boost::regex::icase);
        boost::regex reg3(".*lib\\(.*d\\d*s_?\\d{5}.o\\)$", boost::regex::icase);

        map<unsigned int,string> syms;
        unsigned int addr, len, stubhex;

        while(aMap.getline(str,MAX_LINE)) {
            if(strlen(str) == 0)
                break;
            else if(regex_search(str, what, reg1)) {
                sLibFile.assign(what[4].first,what[4].second-what[4].first);
                if(!regex_search(sLibFile, what1, reg)) {
                    sTmp.assign(what[2].first,what[2].second-what[2].first);
                    addr = strtol(sTmp.c_str(), NULL, 16);
                    sTmp.assign(what[3].first,what[3].second-what[3].first);
                    len = strtol(sTmp.c_str(), NULL, 16);
                    syms[addr+len] = "";
                    if(regex_search(sLibFile, what, reg3)) {
                        stubhex = addr;
                    }
                }
            }
            else if(regex_search(str, what, reg2)) {
                sTmp.assign(what[1].first,what[1].second-what[1].first);
                addr = strtol(sTmp.c_str(), NULL, 16);
                sTmp.assign(what[2].first,what[2].second-what[2].first);
                syms[addr] = (addr == stubhex)? ("stub "+sTmp) : sTmp;
            }
        }

        map<unsigned int,string>::iterator it = syms.begin();
        map<unsigned int,string>::iterator itp = it++;
        string outString;
        for(; it != syms.end(); itp = it++) {
            if(itp->second != "") {
                memset(str,0,sizeof(str));
                sprintf(str,"%08x    %04x    ",(itp->first-imgText), (it->first-itp->first));
                outString = str;
                outString += it->second+"\n";
                iSymFile.write(outString.c_str(),outString.length());
            }
        }
    }
}
int SymbolGenerator::GetSizeFromBinFile( const string& fileName ){
    TInt ret = 0;
    ifstream aIf(fileName.c_str(), ios_base::binary);
    if( !aIf.is_open() ) {
        printf("Warning: Cannot open file \n");
    }
    else {
        E32ImageFile e32Image;
        TUint32 aSz;

        aIf.seekg(0,ios_base::end);
        aSz = aIf.tellg();

        e32Image.Adjust(aSz);
        e32Image.iFileSize = aSz;

        aIf.seekg(0,ios_base::beg);
        aIf >> e32Image;
        ret = e32Image.iOrigHdr->iCodeSize;
    }
    return ret;
}
void SymbolGenerator::thrd_func(){
    SymbolGenerator* me = GetInstance();

    TPlacedEntry pe("",false);
    while(1) {
        if(1) {
            //scope the code block with if(1) for lock
            boost::mutex::scoped_lock lock(me->iMutex);
            while(me->iQueueFiles.empty())
                me->iCond.wait(lock);
            /*
            if(me->iQueueFiles.empty()) {
                boost::this_thread::sleep(boost::posix_time::milliseconds(10));
                continue;
            }
            */

            pe = me->iQueueFiles.front();
            me->iQueueFiles.pop();
        }

        if(pe.iFileName == "")
            break;
        else if(pe.iExecutable) 
            me->ProcessExecutable(pe.iFileName);
        else
            me->ProcessDatafile(pe.iFileName);
    }
}
SymbolGenerator::SymbolGenerator() : boost::thread(thrd_func) {
}
SymbolGenerator::~SymbolGenerator(){
    if(joinable())
        join();
    iSymFile.flush();
    iSymFile.close();
}
