/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/

package com.nokia.s60tools.remotecontrol.keyboard.ui.view;

import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.widgets.Button;

import com.nokia.s60tools.remotecontrol.keyboard.IKeyboardMediator;

/**
 * Key listener for keyboard buttons. Listen space button of PC keyboard.
 */
class KeyboardKeyListener implements KeyListener {
	
	/**
	 * Timestamp when space button is pressed
	 */
	private int keyDownTimestamp = 0;
	
	/**
	 * True if space is pressed down but not released yet
	 */
	private boolean keyDown = false;
	
	/**
	 * ASCII code for space character
	 */
	static final char SPACE = 32;
	
	/**
	 * Key data
	 */
	private KbKey key;
	
	/**
	 * Instance of the keyboard which is registered the keyboard mediator
	 */
	private Object keyboard;
	
	/**
	 * Keyboard mediator to delegate events to.
	 */
	private IKeyboardMediator keyboardMediator;
	
	/**
	 * Keeping  button down more than 500 ms is handled as long press
	 */
	private static final long LONG_PRESS_TIME = 500;

	/**
	 * Constructor
	 * @param key Key data
	 * @param keyboard Instance of the keyboard which is registered the keyboard mediator
	 * @param keyboardMediator Keyboard mediator to delegate events to
	 */
	public KeyboardKeyListener(KbKey key, Object keyboard, IKeyboardMediator keyboardMediator) {
		super();
		this.key = key;
		this.keyboard = keyboard;
		this.keyboardMediator = keyboardMediator;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) {
		// Several events are received when key is hold down.
		// Only first event is handled
		if (e.character == SPACE && !keyDown) {
			if (key.getScancode() >= 0) {
				// Scancode is mapped to button. Scancodes are sent after button is released.
				// Wait for button release.
				keyDown = true;
				keyDownTimestamp = e.time;
			} else {
				// Shift key(on QWERTY keyboard) press is not send to target. 
				// It only changes button labels to shift labels.
				if (key.getKeyFunc() != KbKey.KbKeyFunc.SHIFT) {
					// Character is mapped to button. Send character event.
					keyboardMediator.characterKeyPressed(keyboard, ((Button)e.widget).getText().charAt(0));
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) {
		if (e.character == SPACE) {
			// Space released
			if (key.getScancode() >= 0) {
				// Calculate time that button is kept down
				int holdTime = e.time - keyDownTimestamp;
				if (holdTime > LONG_PRESS_TIME) {
					// Button is hold. Send press and hold key event.
					keyboardMediator.scanCodeKeyPressedAndHold(keyboard, key.getScancode(), holdTime);
				} else {
					// Button is pressed. Send press key event.
					keyboardMediator.scanCodeKeyPressed(keyboard, key.getScancode());
				}
			}
			keyDown = false;
		}
	}
}
