/*
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of the License "Symbian Foundation License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description:
 *
 */
package com.nokia.svg2svgt.converter;

import java.util.ArrayList;
import java.util.Hashtable;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Element;

import com.nokia.svg2svgt.log.Logger;
import com.nokia.svg2svgt.util.XMLFileParser;

/**
 * <code>GradientDefinitionMerger</code> merges the gradient paint definitions
 * that are same (in the input SVG file).
 * 
 */
public class GradientDefinitionMerger implements ConversionConstants {

	// PUBLIC METHODS

	/**
	 * Merges the gradient paint definitions that are same.
	 * 
	 * @param svgtDoc
	 *            SVCT document reference.
	 * @param logger
	 *            Logger reference for logging.
	 */
	public static void mergeGradientDefinitions(Document svgtDoc, Logger logger) {
		// PrintDOMTree.printDOMTree( svgtDoc );
		NodeList nodeList = svgtDoc.getElementsByTagName(TAG_LINEAR_GRADIENT);
		// merge linear gradient paint definitions
		mergeSimilarGradientDefinitions(nodeList, svgtDoc, logger);

		nodeList = svgtDoc.getElementsByTagName(TAG_RADIAL_GRADIENT);
		// merge linear gradient paint definitions
		mergeSimilarGradientDefinitions(nodeList, svgtDoc, logger);
		// PrintDOMTree.printDOMTree( svgtDoc );
	}

	// PRIVATE METHODS

	/**
	 * Merges the gradient paint definitions that are same.
	 * 
	 * @param nodeLis
	 *            List of nodes with gradient paint definitions.
	 * @param svgtDoc
	 *            SVCT document reference.
	 * @param logger
	 *            Logger reference for logging.
	 */
	private static void mergeSimilarGradientDefinitions(NodeList nodeList,
			Document svgtDoc, Logger logger) {
		// merging happens only after non-referred ids have been removed.
		// NodeList nodeList = svgtDoc.getElementsByTagName( TAG_LINEAR_GRADIENT
		// );
		ArrayList nodes = null;
		if (null != nodeList) {
			// copy all the return gradient definitions in an ArrayList
			nodes = new ArrayList(nodeList.getLength());
			for (int i = 0; i < nodeList.getLength(); i++) {
				nodes.add(nodeList.item(i));
			}

			if (1 == nodes.size()) {
				Node node = (Node) nodes.get(0);// node reference containing the
												// attribute
				createDefsParentElement(node, svgtDoc);
			}

			int currentIndex = 0; // current index of parent node
			int maximumIndex = nodes.size(); // maximum nodes present
			int tempIndex = currentIndex + 1; // temporary index
			while (currentIndex != maximumIndex) {
				if (currentIndex == tempIndex) {
					tempIndex++;
					continue;
				}
				Node node1 = (Node) nodes.get(currentIndex);
				if ((currentIndex < maximumIndex)
						&& (tempIndex <= nodes.size() - 1))
				// ( ( currentIndex + 1 ) < nodes.size() ) )
				{
					// System.out.println( nodes.size() + "  " + currentIndex );
					// Node node2 = ( Node) nodes.get( currentIndex + 1 );
					Node node2 = (Node) nodes.get(tempIndex);
					boolean areEqual = compareNodes(node1, node2);
					if (false == areEqual) {
						tempIndex++;
					} else {

						Attr attr1 = ((Element) node1)
								.getAttributeNode(ID_ATTRIBUTE);
						Attr attr2 = ((Element) node2)
								.getAttributeNode(ID_ATTRIBUTE);
						if ((null == attr1) && (null == attr2)) {
							// both the ids are null
							// delete the second reference
							Node parentNode = node2.getParentNode();// node
																	// reference
																	// containing
																	// the
																	// attribute
							parentNode.removeChild(node2);
							nodes.remove(node2);
							createDefsParentElement(node1, svgtDoc);

							maximumIndex--;
							if (maximumIndex > tempIndex)
								tempIndex++;
							// continue;
						} else {
							if ((null == attr1) || (null == attr2)) {
								// if either one of them is null
								if (null == attr1) {
									// remove attr1
									Node parentNode = node1.getParentNode();
									parentNode.removeChild(node1);
									nodes.remove(node1);
									createDefsParentElement(node2, svgtDoc);
									// make the second node as home node
									currentIndex++;
									// node1 = node2;
								} else {
									// for sure that attr2 is null
									// remove attr2
									Node parentNode = node2.getParentNode();
									parentNode.removeChild(node2);
									nodes.remove(node2);
									createDefsParentElement(node1, svgtDoc);
								}
								maximumIndex--;
								if (maximumIndex > tempIndex)
									tempIndex++;
								// continue;
							} else {
								// it comes here only if both the references are
								// not null
								// now delete the second node and update the
								// reference id as node1's id
								updateIdReferences(node2, attr1.getNodeValue(),
										svgtDoc);
								Node parentNode = node2.getParentNode();
								parentNode.removeChild(node2);
								createDefsParentElement(node1, svgtDoc);
								nodes.remove(node2);
								maximumIndex--;
								if (maximumIndex > tempIndex)
									tempIndex++;
							}
						}
					}
				}
				if (tempIndex < maximumIndex) {
					continue;
				} else if (tempIndex == maximumIndex) {
					currentIndex++;
					tempIndex = currentIndex + 1;
				}
			}
		}

	}

	/**
	 * Checks if the gardient definition is defined under defs element tag or
	 * not. If encapsulating defs element is not present, it creates the defs
	 * element and appends this node to it.
	 * 
	 * @param node
	 *            Gradient paint node.
	 * @param svgtDoc
	 *            SVGT Document reference.
	 */
	private static void createDefsParentElement(Node node, Document svgtDoc) {
		// get the node's parent node
		Node parentNode = node.getParentNode();
		// Node parentNode = node.getPreviousSibling();
		// System.out.println( "node = " + node.getNodeName());
		// System.out.println( "Parent node = " + parentNode );
		if ((null != parentNode)
				&& (false == parentNode.getNodeName().equals(TAG_DEFS))) {
			Element defsElem = svgtDoc.createElement(TAG_DEFS);
			defsElem.appendChild(node);
			// parentNode.removeChild( node );
			parentNode.appendChild(defsElem);
		}
	}

	/**
	 * Updates the referenced identifiets to the deleted nodes with a new
	 * identifier of the retained node.
	 * 
	 * @param node
	 *            Node for which the id reference shoudl be updated.
	 * @param newIdRef
	 *            New reference identifier.
	 * @param svgtDoc
	 *            SVGT Document reference.
	 */
	private static void updateIdReferences(Node node, String newIdRef,
			Document svgtDoc) {
		Hashtable referredIdData = NonReferredIdRemover
				.getReferencedIdentifiers();
		Element elementNode = (Element) node;
		Attr attr = elementNode.getAttributeNode(ID_ATTRIBUTE);
		String idValue = attr.getNodeValue();

		// this entry should for sure exist in the hashtable
		// get the node name referring to this identifier
		String nodeName = null;

		// though hashtable cannot be null if id is being referred
		if (null != referredIdData) {
			nodeName = (String) referredIdData.get(idValue);
			// node name cannot be null, play safe ;)
			NodeList nodeList = null;
			if (null != nodeName) {
				nodeList = svgtDoc.getElementsByTagName(nodeName);
			}
			if (null != nodeList) {
				// traverse through the list
				Node tempNode = null;
				Attr attrNode = null;
				NamedNodeMap attrList = null;
				String attrValue = null;
				for (int i = 0; i < nodeList.getLength(); i++) {
					tempNode = nodeList.item(i);
					if (Node.ELEMENT_NODE != tempNode.getNodeType()) {
						continue;
					}

					// attrNode = ( ( Element ) tempNode).getAttributeNode()
					if (false == tempNode.hasAttributes()) {
						continue;
					}
					attrList = tempNode.getAttributes();
					for (int j = 0; j < attrList.getLength(); j++) {
						attrNode = (Attr) attrList.item(j);
						attrValue = attrNode.getValue();
						if ((-1 != attrValue.indexOf(ID_REFERENCE_IDENTIFIER))
								&& (-1 != attrValue.indexOf(idValue))) {
							int startIndex = attrValue.indexOf(idValue);
							StringBuffer tempStr = new StringBuffer();
							tempStr.append(attrValue.substring(0, startIndex));
							tempStr.append(newIdRef);
							tempStr.append(attrValue.substring(startIndex
									+ idValue.length(), attrValue.length()));
							attrNode.setNodeValue(tempStr.toString());
						} else if ((-1 != attrValue
								.indexOf(ID_XPOINTER_REFERENCE_IDENTIFIER))
								&& (-1 != attrValue.indexOf(idValue))) {
							int startIndex = attrValue.indexOf(idValue);
							StringBuffer tempStr = new StringBuffer();
							tempStr.append(attrValue.substring(0, startIndex));
							tempStr.append(newIdRef);
							tempStr.append(attrValue.substring(startIndex
									+ idValue.length(), attrValue.length()));
							attrNode.setNodeValue(tempStr.toString());
						}
					}// end attribute list iteration
				}// end node list iteration
			}
		}

	}

	/**
	 * Does a deep comparison of the two nodes for equality.
	 * 
	 * @param node1
	 *            Element node to be compared.
	 * @param node2
	 *            Element node to be compared.
	 * @return True if both the nodes are equal, else false.
	 */
	private static boolean compareNodes(Node node1, Node node2) {

		// if both the nodes have different names
		if (!(node1.getNodeName()).equalsIgnoreCase(node2.getNodeName())) {
			return false;
		}

		// compare the attribute for each element
		boolean haveSameAttr = compareAttributes(node1, node2);
		if (false == haveSameAttr) {
			return false;
		}
		NodeList nodeList1 = node1.getChildNodes();
		NodeList nodeList2 = node2.getChildNodes();

		// compare the child elements
		ArrayList childNodes1 = new ArrayList();
		ArrayList childNodes2 = new ArrayList();

		// read valid items from node list 1
		for (int i = 0; i < nodeList1.getLength(); i++) {
			if (Node.ELEMENT_NODE != nodeList1.item(i).getNodeType()) {
				continue;
			}
			childNodes1.add(nodeList1.item(i));
		}

		// read valid items from node list 2
		for (int i = 0; i < nodeList2.getLength(); i++) {
			if (Node.ELEMENT_NODE != nodeList2.item(i).getNodeType()) {
				continue;
			}
			childNodes2.add(nodeList2.item(i));
		}

		if (childNodes1.size() != childNodes2.size()) {
			return false;
		}

		// if same number of child elements, compare each child element
		for (int i = 0; i < childNodes1.size(); i++) {
			// recursive loop for comparing child nodes
			boolean areEqual = compareNodes((Node) childNodes1.get(i),
					(Node) childNodes2.get(i));
			if (true == areEqual) {
				continue;
			} else {
				return false;
			}
		}

		return true;
	}

	/**
	 * Does a comparison for the attribute for two nodes for equality.
	 * 
	 * @param node1
	 *            Attribute node to be compared.
	 * @param node2
	 *            Attribute node to be compared.
	 * @return True if both the nodes are equal, else false.
	 */
	private static boolean compareAttributes(Node node1, Node node2) {
		NamedNodeMap attr1 = null;
		NamedNodeMap attr2 = null;
		// String n1 = node1.getNodeName();
		// if both the nodes have attributes, compare each attribute
		if ((true == node1.hasAttributes()) && (true == node2.hasAttributes())) {
			attr1 = node1.getAttributes();
			attr2 = node2.getAttributes();
			int attrCount1 = attr1.getLength();
			String id = ((Element) node1).getAttribute(ID_ATTRIBUTE);
			if ((null != id) && (id.trim().length() > 0)) {
				attrCount1--;
			}
			int attrCount2 = attr2.getLength();
			id = ((Element) node2).getAttribute(ID_ATTRIBUTE);
			if ((null != id) && (id.trim().length() > 0)) {
				attrCount2--;
			}
			Attr attribute = null;
			String attributeName = null;
			String attributeValue = null;
			// have similar number of attributes
			if (attrCount1 != attrCount2) {
				return false;
			}
			// compare each attribute
			// boolean flag = false;
			for (int i = 0; i < attr1.getLength(); i++) {
				// check if node2 has this attribute with the specified value
				attribute = (Attr) attr1.item(i);
				attributeName = attribute.getName();
				attributeValue = attribute.getValue();
				attribute = ((Element) node2).getAttributeNode(attributeName);
				if (null == attribute) {
					return false;
				} else {
					// id will always be unique, so ignore the id identifier
					if (attributeName.equalsIgnoreCase(ID_ATTRIBUTE)) {
						continue;
					}
					if (attributeValue.equalsIgnoreCase(attribute.getValue())) {
						continue;
					} else {
						return false;
					}
				}
			}
		}
		// if both the nodes dont have any attributes
		if ((false == node1.hasAttributes())
				&& (false == node2.hasAttributes())) {
			return true;
		}
		return true;
	}

	/**
	 * Main method for testing purpose.
	 * 
	 * @param args
	 *            Command line arguments.
	 */
	public static void main(String[] args) {
		try {
			Document svgtDoc = XMLFileParser.parseFile(
					"C:\\SVG\\samples\\Mine.svg", false);
			NonReferredIdRemover.processNonReferencedIDs(svgtDoc, null);
			// System.out.println("Merge similar definitions");
			GradientDefinitionMerger.mergeGradientDefinitions(svgtDoc, null);
		} catch (Exception ex) {
			ex.printStackTrace();
		}

	}

}
// if( attributeName.equalsIgnoreCase( attribute.getNodeName() ) )
// {
// flag = true;
// }
// if( attributeName.equalsIgnoreCase( attribute.getNodeName() ) )