/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.gui;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.nokia.svg2svgt.util.XMLFileParser;
import com.nokia.svg2svgt.converter.ConversionConstants;
import com.nokia.svg2svgt.converter.SVGAttribute;
import com.nokia.svg2svgt.converter.SVGElement;
import com.nokia.svg2svgt.converter.SVGModule;
import com.nokia.svg2svgt.log.Logger;
import com.nokia.svg2svgt.SVG2SVGTConstants;

/**
 * <code>BlackListAnalyzer</code> removes non SVGT elements from the input SVG
 * file. Additional elements to be removed can also be defined in the black list
 * file. It also uses the grey list file and issues a warning for the elements
 * specified in grey list file.
 * 
 */
public class SVGDTDAnalyzer implements ConversionConstants {
	/**
	 * Parsed black list document.
	 */
	private Document myDocument = null;

	/**
	 * List containg <code>SVGModules</code> in the black list file.
	 */
	private ArrayList myModuleList = null;

	/**
	 * Logger instance for logging events
	 */
	private Logger myLogger = null;

	// PUBLIC METHODS

	/**
	 * Creates an BlackFileAnalyser that parses the black list file.
	 * 
	 * @param fileName
	 *            Complete name of the black list file.
	 */
	public SVGDTDAnalyzer(String fileName, Logger logger)
			throws FactoryConfigurationError, ParserConfigurationException,
			IOException, SAXException, IllegalArgumentException {
		myLogger = logger;
		this.logEvents(SVG2SVGTConstants.LOADING_FILE,
				new String[] { fileName });
		myDocument = XMLFileParser.parseFile(fileName, false);
		// load the black list file elements into a datastructure
		loadSVGDTDFile();
		// printBlackList( myBlackList );
		// System.out.println("Black list loaded successfully...");
	}

	/**
	 * Checks if an element or attribute by the tag name of the input node is
	 * present in the black list file or not.
	 * 
	 * @param node
	 *            Node to be searched.
	 * @return True if found, else false.
	 */
	public boolean isSVGTag(Node node) {
		String nodeName = node.getNodeName();
		// System.out.println(nodeName);
		if (Node.ELEMENT_NODE == node.getNodeType()) {
			SVGElement svgElem = this.fetchElementByTagName(nodeName);
			if (null != svgElem) {
				return true;
			}
		} else if (Node.ATTRIBUTE_NODE == node.getNodeType()) {
			SVGAttribute svgAttr = this.fetchAttributeByTagName(nodeName);
			if (null != svgAttr) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Checks if an element by the specified name is present in the black list
	 * or not.
	 * 
	 * @param elementName
	 *            Name of the element to be searched.
	 * @return Index of the element in the module list.
	 */
	public int containsElement(String elementName) {
		if (null == myModuleList) {
			return -1;
		}
		Iterator iter = myModuleList.iterator();
		int indexCount = 0;
		SVGModule svgModule = null;
		while (true == iter.hasNext()) {
			svgModule = (SVGModule) iter.next();
			if (-1 != svgModule.containsElement(elementName)) {
				return indexCount;
			}
			indexCount++;
		}
		return -1;
	}

	/**
	 * Fetches the <code>SVGElement</code> instance containing the specified
	 * attribute name.
	 * 
	 * @param attrName
	 *            Attribute name to be searched.
	 * @return <code>SVGElement</code> instance if found, else null.
	 */
	public SVGElement fetchElementContainingAttribute(String attrName) {
		if (null == myModuleList) {
			return null;
		}
		Iterator iter = myModuleList.iterator();
		SVGModule svgModule = null;
		SVGElement svgElement = null;
		while (true == iter.hasNext()) {
			svgModule = (SVGModule) iter.next();
			svgElement = svgModule.getElementContainingAttribute(attrName);
			if (null != svgElement) {
				return svgElement;
			}
		}
		return null;
	}

	/**
	 * Fetches the <code>SVGElement</code> instance with the specified element
	 * name.
	 * 
	 * @param elementName
	 *            Element name to be searched.
	 * @return <code>SVGElement</code> instance if found, else null.
	 */
	public SVGElement fetchElementByTagName(String elementName) {
		if (null == myModuleList) {
			return null;
		}
		Iterator iter = myModuleList.iterator();
		SVGModule svgModule = null;
		SVGElement svgElement = null;
		// boolean isPresent = false;
		while (true == iter.hasNext()) {
			svgModule = (SVGModule) iter.next();
			svgElement = svgModule.getElementByTagName(elementName);
			if (null != svgElement) {
				return svgElement;
			}
		}
		return null;
	}

	/**
	 * Fetches the <code>SVGAttribute</code> instance with the specified
	 * attribute name.
	 * 
	 * @param attrName
	 *            Attribute name to be searched.
	 * @return <code>SVGAttribute</code> instance if found, else null.
	 */
	public SVGAttribute fetchAttributeByTagName(String attrName) {
		if (null == myModuleList) {
			return null;
		}
		Iterator iter = myModuleList.iterator();
		SVGModule svgModule = null;
		SVGAttribute svgAttribute = null;
		int elemIndex = 0;
		while (true == iter.hasNext()) {
			svgModule = (SVGModule) iter.next();
			// return svgModule.getElementByTagName( attrName );
			svgAttribute = svgModule.getAttributeByTagName(attrName);
			if (null != svgAttribute) {
				return svgAttribute;
			}
		}
		return null;
	}

	/**
	 * Returns the list of all the black list modules in the black list file.
	 * 
	 * @return List of modules in black list file.
	 */
	public ArrayList getSVGModules() {
		return myModuleList;
	}

	// PRIVATE METHODS

	/**
	 * Parses the black list file and stores the contents of the file as
	 * <code>SVGModule</code> elements.
	 * 
	 */
	private void loadSVGDTDFile() {
		Element rootElement = myDocument.getDocumentElement();

		// if not a black list file, return without loading or throw an
		// exception
		if (!(rootElement.getTagName()).equalsIgnoreCase(TAG_SVG_ROOT)) {
			return;
		}

		// get the module elements
		NodeList moduleNodes = rootElement.getElementsByTagName(TAG_MODULE);

		Element module = null;
		// ArrayList moduleNode = new ArrayList();
		myModuleList = new ArrayList();// collection of modules
		SVGModule svgModule = null;
		String moduleName = null;

		// iterate through the available module nodes
		for (int i = 0; i < moduleNodes.getLength(); i++) {
			// get the module node
			module = (Element) moduleNodes.item(i);

			// get the attribute for this node
			moduleName = module.getAttribute(TAG_NAME);
			// System.out.println( moduleName);
			if (null != moduleName) {
				ArrayList elementNodes = resolveElementsInModule(module,
						moduleName);
				if ((null != elementNodes) || (0 < elementNodes.size())) {
					svgModule = new SVGModule(moduleName);
					svgModule.addElements(elementNodes);
					myModuleList.add(svgModule);
				}
			} // end not a null module name
		}// end for loop
	}

	/**
	 * Resolves the element tags present in a single module element tag.
	 * 
	 * @param module
	 *            Module element to be searched.
	 * @param moduleName
	 *            Name of this module.
	 * @return List of <code>SVGElements</code>
	 */
	private ArrayList resolveElementsInModule(Element module, String moduleName) {
		Element element = null;
		String elementName = null;
		String isMarked = null;
		SVGElement svgElement = null;
		ArrayList elementList = new ArrayList();

		// get all the elements by tag name element
		NodeList elementNodes = module.getElementsByTagName(TAG_ELEMENT);

		// iterate through the available list
		for (int i = 0; i < elementNodes.getLength(); i++) {
			element = (Element) elementNodes.item(i);

			// get the name for this element
			elementName = element.getAttribute(TAG_NAME);
			// System.out.println("Elements=" + elementName );

			// if the attribute needs to be removed
			if (null != elementName) {
				svgElement = new SVGElement(elementName, moduleName);
				ArrayList attrList = resolveAttributesInElement(element,
						moduleName, elementName);
				if ((null != attrList) || (0 < attrList.size())) {
					svgElement.addAttributes(attrList);
				}
				elementList.add(svgElement);
				continue;

			} // end if null elementName

		} // end for - element nodes
		return elementList;
	}

	/**
	 * Resolves the attribute tag present in the element tag. If the attribute
	 * tag contains a reference to an attribute set, it resolves the attributes
	 * present in that attribute set also.
	 * 
	 * @param element
	 *            Element for which attributes are to be searched.
	 * @param moduleName
	 *            Name of the module to which this element belongs.
	 * @param elementName
	 *            Name of the element.
	 * @return List of <code>SVGAttributes<code> for this element.
	 */
	private ArrayList resolveAttributesInElement(Element element,
			String moduleName, String elementName) {
		Attr attr = null;
		Element attribute = null;
		String attributeName = null;
		String attributeValue = null;
		SVGAttribute svgAttribute = null;
		ArrayList attributesList = new ArrayList();

		// get all the tags with name as attribute
		NodeList attributeNodes = element.getElementsByTagName(TAG_ATTRIBUTE);

		// iterate through the retreived list
		for (int i = 0; i < attributeNodes.getLength(); i++) {
			attribute = (Element) attributeNodes.item(i);
			attr = attribute.getAttributeNode(TAG_NAME);
			if (null != attr) {
				// read the attribute name and value
				attributeName = attr.getValue();

				// construct the SVGAttribute instance
				svgAttribute = new SVGAttribute(attributeName, elementName,
						moduleName, null, "");
				// svgElement.addAttribute( svgAttribute );
				attributesList.add(svgAttribute);
			}
		} // end for - attribute nodes
		return attributesList;
	}

	/**
	 * Prints the black list. Just used for testing.
	 * 
	 * @param moduleNode
	 *            SVGModules in black list
	 */
	private void printBlackList(ArrayList moduleNode) {
		for (int i = 0; i < moduleNode.size(); i++) {
			SVGModule module = (SVGModule) moduleNode.get(i);
			System.out.println(module.toString());
		}
	}

	private void logEvents(long code, String[] desc) {

	}

	/**
	 * Main method for testing purposes.
	 * 
	 * @param args
	 *            Command line arguments.
	 */
	public static void main(String[] args) {
		try {
			SVGDTDAnalyzer bla = new SVGDTDAnalyzer(
					"D:\\JavaFiles\\BlackList.xml", null);
		} catch (Exception ex) {
			ex.printStackTrace();
		}

	}

}
