/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.gui;

import java.util.Hashtable;

import com.nokia.svg2svgt.SVG2SVGTConstants;
import com.nokia.svg2svgt.configuration.ConfigurationManager;
import com.nokia.svg2svgt.converter.SVG2SVGTConverter;
import com.nokia.svg2svgt.localiser.Localiser;
import com.nokia.svg2svgt.log.LogListener;
import com.nokia.svg2svgt.log.LogWriter;

/**
 * <code>SVGTConverterController</code> implements <code>LogListener</code>
 * interface and acts the controller the converter tools GUI. It also creates
 * the model and the view and sets this model to the view. The model and view
 * are created only if the converter tool is launched in the stand-alone mode.
 * 
 * <code>SVGTConverterController</code> uses <code>ConfigurationManager</code>
 * for managing the converter tools configuration and uses
 * <code>SVG2SVGTConverter</code> to carry out the conversion operation. In
 * batch processing mode (if multiple files are specified), the conversion is
 * done in order in which they are specified on the command line. If any error
 * occurs while conversion, the current conversion process for this file is
 * stopped and the <code>SVG2SVGTConverter</code> continues with the next file
 * in the list.
 * 
 * <code>SVGTConverterController</code> implements <code>LogListener</code>
 * interface to listen to the log events invoked during the conversion process.
 * There events are passed further to the <code>SVGTConverterView</code> to be
 * displayed on the UI and the <code>FileLogger</code> to be written into the
 * file.
 */
public class SVGTConverterController implements LogListener {

	/**
	 * The view for this controller.
	 */
	private SVGTConverterView myView;

	/**
	 * Model to be associated with the view.
	 */
	private SVGTConverterModel myModel;

	/**
	 * Configuration manager for reading/modifying the configuration.
	 */
	private ConfigurationManager myConfigMgr;

	/**
	 * Hashtable containing the command line parameters.
	 */
	private Hashtable myCmdLineParams;

	private LogWriter myLogWriter;

	private SVG2SVGTConverter myConverter;

	private String myModeOfOperation;

	/**
	 * 
	 * @param modeOfOperation
	 * @param logger
	 */
	public SVGTConverterController(String modeOfOperation, LogWriter logger) {
		this(null, modeOfOperation, logger);
	}

	/**
	 * 
	 * @param cmdLineParams
	 * @param modeOfOperation
	 * @param logger
	 */
	public SVGTConverterController(Hashtable cmdLineParams,
			String modeOfOperation, LogWriter logger) {

		// System.out.println( "In OCnst " + myLogWriter );
		myLogWriter = logger;
		myModeOfOperation = modeOfOperation;
		myCmdLineParams = cmdLineParams;

		if (true == modeOfOperation
				.equalsIgnoreCase(SVG2SVGTConstants.GUI_MODE)) {
			// GUI mode - create the UI
			// String region = ( String ) cmdLineParams.get("-region");
			String region = "GB";

			// localiser required only if GUI is used....
			Localiser localiser = null;
			if (null != region) {
				try {
					if (false == isRegionSupported(region)) {
						localiser = new Localiser("GB");
					} else {
						localiser = new Localiser(region);
					}

				} catch (Exception ex) {
					localiser = null;
				}
			}

			myView = new SVGTConverterView(this, localiser);
			myModel = new SVGTConverterModel();
			myView.setModel(myModel);
		} else {
			// batch mode, go ahead with the conversion
			doConvert(myCmdLineParams);
		}
	}

	/**
	 * Checks if a region settings are supported by the tool or not.
	 * 
	 * @param region
	 *            Region settings to be checked.
	 * @return True if region is supported, else false.
	 */
	private boolean isRegionSupported(String region) {
		boolean isSupported = false;
		for (int i = 0; i < (SVG2SVGTConstants.SUPPORTED_REGIONS).length; i++) {
			if (region.equalsIgnoreCase(SVG2SVGTConstants.SUPPORTED_REGIONS[i])) {
				isSupported = true;
				break;
			}
		}
		return isSupported;
	}

	/**
	 * Facilitates SVG to SVGT conversion.
	 */
	public void doConvert(Hashtable confParams) {
		// GUI Mode, view should prepare this hashtable
		// BATCH Mode, argument is command line params
		myConverter = new SVG2SVGTConverter(confParams);

		// registers itself as listener for the log events
		myConverter.registerLogListener(this);
		myConverter.convert();

	}

	/**
	 * Returns the configuration manager instance used by this controller.
	 * 
	 * @return <code>ConfigurationManager</code> instance.
	 */
	public ConfigurationManager getConfigurationManager() {
		return myConfigMgr;
	}

	/**
	 * Forwards the log event to the <code>SVGTConverterView</code> and
	 * <code>FileLogger</code>.
	 * 
	 * @param desc
	 *            Log event description
	 */
	public void log(String desc) {
		// check the mode of operation
		// if GUI mode, update the model with the log event and send events to
		// file
		// if batch mode, write to screen
		// System.out.println( " MyLogger = " + myLogWriter );
		myLogWriter.log(this.myModeOfOperation, desc);

		// for this event to the view
		if (SVG2SVGTConstants.GUI_MODE.equals(this.myModeOfOperation)) {
			myView.log(desc);
		}

	}

	/**
	 * Forwards the warning event to the <code>SVGTConverterView</code> and
	 * <code>FileLogger</code>.
	 * 
	 * @param desc
	 *            Warning event description.
	 */
	public void logWarning(String desc) {
		myLogWriter.log(this.myModeOfOperation, desc);
	}
}
