/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.localiser;

import java.io.File;
import java.util.ResourceBundle;
import com.nokia.svg2svgt.configuration.*;

/**
 * Localises the data to be displayed on the UI based on the region specified by
 * the user.
 * 
 */
public class Localiser {

	/**
	 * File extension for the String bundle and Icon bundle files.
	 */
	private static final String FILE_EXTENSION = ".properties";

	/**
	 * Identifier for a components Tool tip text in the String bundle.
	 */
	private static final String TOOLTIPTEXT = "_ToolTipText";

	/**
	 * Identifier for the menumonic key for a component on the UI.
	 */
	private static final String MNEUMONIC = "_Mneumonic";

	/**
	 * Identifier for the components text in the String bundle.
	 */
	private static final String COMPONENTTEXT = "_Text";

	/**
	 * ConfigurationManager for the String bundle specifying the text data for
	 * the components displayed on the UI.
	 */
	private ConfigurationManager myStringBundleMgr = null;

	/**
	 * Configuration instance wrapping the string bundle strings.
	 */
	private Configuration myStringBundle = null;

	/**
	 * ConfigurationManager for the Icon bundle specifying the path for the
	 * icons to be displayed on the UI.
	 */
	private ConfigurationManager myIconBundleMgr = null;

	/**
	 * Configuration instance wrapping the icon paths.
	 */
	private Configuration myIconBundle = null;

	/**
	 * Creates a localiser instance for a specific region.
	 * 
	 * @param region
	 *            Region for this localiser. Default is "GB".
	 * @throws Exception
	 *             If creation of instance fails.
	 */
	public Localiser(String region) throws Exception {
		try {
			if ((null != region) && (0 < region.length())) {
				myStringBundleMgr = new ConfigurationManager(ResourceBundle
						.getBundle("SVG2SVGTProperties").getString(
								"SVGT_STRING_BUNDLE_FILE")
						+ File.separator
						+ "StringBundle_"
						+ region
						+ FILE_EXTENSION);
				myStringBundle = myStringBundleMgr.getConfiguration();

				myIconBundleMgr = new ConfigurationManager(ResourceBundle
						.getBundle("SVG2SVGTProperties").getString(
								"SVGT_STRING_BUNDLE_FILE")
						+ File.separator
						+ "IconBundle_"
						+ region
						+ FILE_EXTENSION);
				myIconBundle = myIconBundleMgr.getConfiguration();
			}
		} catch (Exception ex) {
			// throw an exception
			throw new Exception("Error creating localiser. Reason: "
					+ ex.toString());
		}

	}

	/**
	 * Returns the localised text for the component specified by componentName.
	 * 
	 * @param componentName
	 *            Component name for which the localised text is to be fetched.
	 * @return Localised component name.
	 */
	public String getText(String componentName) {
		Parameter param = null;
		String text = componentName + COMPONENTTEXT;
		if (null != myStringBundle) {
			param = myStringBundle.getParameter(text);
			if (null != param) {
				return param.getValue();
			}
		}

		return null;
	}

	/**
	 * Return the Menumonic to be used for this component.
	 * 
	 * @param componentName
	 *            Component name.
	 * @return Mneumonic key.
	 */
	public String getMneumonic(String componentName) {
		Parameter param = null;
		String mneumonic = componentName + MNEUMONIC;
		if (null != myStringBundle) {
			param = myStringBundle.getParameter(mneumonic);
			if (null != param) {
				return param.getValue();
			}
		}
		return null;
	}

	/**
	 * Returns the localised tool tip text for the component specified by
	 * componentName.
	 * 
	 * @param componentName
	 *            Component name.
	 * @return Localised tool tip text.
	 */
	public String getToolTipText(String componentName) {
		Parameter param = null;
		String toolTip = componentName + TOOLTIPTEXT;
		if (null != myStringBundle) {
			param = myStringBundle.getParameter(toolTip);
			if (null != param) {
				return param.getValue();
			}
		}
		return null;
	}

	/**
	 * Returns the localised icon to be used for the specified component.
	 * 
	 * @param componentName
	 *            Component name.
	 * @return Local path from where icon is to be fetched.
	 */
	public String getIconPath(String componentName) {
		Parameter param = null;
		if (null != myIconBundle) {
			param = myIconBundle.getParameter(componentName);
			if (null != param) {
				return param.getValue();
			}
		}
		return null;
	}

	/**
	 * Main method for testing purposes
	 * 
	 * @param args
	 *            Command line arguments.
	 */
	public static void main(String[] args) {
		try {
			Localiser l = new Localiser("GB");
			System.out.println(l.getText("JMenu.fileMenuItem"));
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}
}
