/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.util;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Print the input DOM tree to the standard output. This is a utility class used
 * for testing purposes.
 */
public class PrintDOMTree {

	/**
	 * Node types in a DOM tree.
	 */
	private static short[] nodeTypes = { Node.ATTRIBUTE_NODE,
			Node.CDATA_SECTION_NODE, Node.COMMENT_NODE,
			Node.DOCUMENT_FRAGMENT_NODE, Node.DOCUMENT_NODE,
			Node.DOCUMENT_TYPE_NODE, Node.ELEMENT_NODE, Node.ENTITY_NODE,
			Node.ENTITY_REFERENCE_NODE, Node.NOTATION_NODE,
			Node.PROCESSING_INSTRUCTION_NODE, Node.TEXT_NODE };

	/**
	 * Node type names in a DOM tree.
	 */
	private static String[] nodeTypeNames = { "ATTRIBUTE", "CDATA_SECTION",
			"COMMENT_NODE", "DOCUMENT_FRAGMENT", "DOCUMENT", "DOCUMENT_TYPE",
			"ELEMENT", "ENTITY", "ENTITY_REFERENCE", "NOTATION",
			"PROCESSING_INSTRUCTION", "TEXT" };

	/**
	 * Prints the DOM tree to the standard output.
	 * 
	 * @param document
	 *            In document
	 */
	public static void printDOMTree(Document document) {
		Element root = document.getDocumentElement();
		showTree(root, 2, 0);
	}

	/**
	 * Obtains the node type name for the given node type.
	 * 
	 * @param type
	 *            Node type.
	 * @return Node type name.
	 */
	public static String getType(short type) {
		for (int i = 0; i < nodeTypes.length; i++) {
			if (type == nodeTypes[i]) {
				return nodeTypeNames[i];
			}
		}
		return "UNKNOWN";
	}

	/**
	 * Returns the number of black spaces for proper indentation.
	 * 
	 * @param count
	 *            Number of blank spaces.
	 * @return String containing blank spaces.
	 */
	private static String getSpaces(int count) {
		String spaces = "";
		for (int i = 0; i < count; i++) {
			spaces += " ";
		}

		return spaces;
	}

	/**
	 * Displays the node of the standard output.
	 * 
	 * @param node
	 *            Node to be displayed.
	 * @param indent
	 *            Identations.
	 * @param size
	 *            Size
	 */
	public static void showTree(Node node, int indent, int size) {
		showTree(node, indent, size, -1);
	}

	/**
	 * Displays the node of the standard output.
	 * 
	 * @param node
	 *            Node to be displayed.
	 * @param indent
	 *            Indentation.
	 * @param size
	 *            Size
	 * @param depth
	 *            Depth
	 */
	public static void showTree(Node node, int indent, int size, int depth) {
		if (null == node) {
			return;
		}
		String name = node.getNodeName();
		short type = node.getNodeType();
		String nodeInfo = "";
		nodeInfo += getSpaces(size * indent) + "<" + name;
		NamedNodeMap attributes = node.getAttributes();
		if (null != attributes) {
			int count = attributes.getLength();
			for (int i = 0; i < count; i++) {
				Node attribute = attributes.item(i);
				nodeInfo += " " + attribute.getNodeName() + "=\""
						+ attribute.getNodeValue() + "\"";
			}
		}
		if (Node.TEXT_NODE == type) {
			nodeInfo += " " + node.getNodeValue();
		}
		nodeInfo += ">";

		NodeList list = node.getChildNodes();
		boolean hasText = false, hasNodes = false;
		String text;

		for (int i = 0; i < list.getLength(); i++) {
			if (Node.TEXT_NODE == list.item(i).getNodeType()) {
				text = list.item(i).getNodeValue();
				if (true == !text.trim().equals("")) {
					nodeInfo += text;
					hasText = true;
				}
			}
		}

		if (true == hasText) {
			nodeInfo += "</" + name + ">";
		}
		System.out.println(nodeInfo);

		if (node.hasChildNodes() && (depth == -1 || depth > 0)) {
			for (int i = 0; i < list.getLength(); i++) {
				if (list.item(i).getNodeType() != Node.TEXT_NODE) {
					showTree(list.item(i), indent, size + 1,
							depth != -1 ? depth - 1 : -1);
				}
			}
		}
		if (false == hasText) {
			System.out.println(getSpaces(size * indent) + "</" + name + ">");
		}
	}

}
