/*
* Copyright (c) 2006-2010 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.tools.theme.core;

import java.awt.image.RenderedImage;
import java.io.File;

import org.eclipse.gef.EditPart;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;

import com.nokia.tools.editing.core.EditingUtil;
import com.nokia.tools.editing.model.EditObject;
import com.nokia.tools.media.image.CoreImage;
import com.nokia.tools.media.image.IRenderedImageProvider;
import com.nokia.tools.media.image.RenderedImageDescriptor;
import com.nokia.tools.media.utils.layers.IImage;
import com.nokia.tools.media.utils.layers.IImageAdapter;
import com.nokia.tools.platform.theme.ColourGraphic;
import com.nokia.tools.platform.theme.ComponentGroup;
import com.nokia.tools.platform.theme.SkinnableEntity;
import com.nokia.tools.platform.theme.Task;
import com.nokia.tools.platform.theme.Theme;
import com.nokia.tools.platform.theme.ThemeBasicData;
import com.nokia.tools.platform.theme.ThemeConstants;
import com.nokia.tools.platform.theme.ThemeTag;
import com.nokia.tools.platform.theme.preview.PreviewElement;
import com.nokia.tools.platform.theme.preview.PreviewRefer;
import com.nokia.tools.platform.theme.preview.ThemePreview;
import com.nokia.tools.resource.util.SimpleCache;
import com.nokia.tools.screen.ui.ISkinnableEntityAdapter;
import com.nokia.tools.theme.content.ThemeData;
import com.nokia.tools.theme.content.ThemeUtil;
import com.nokia.tools.theme.editing.IEntityImageFactory;


public class ThemeLabelProvider extends LabelProvider implements
		IContentLabelProvider {
	private static final int DEFAULT_SIZE = 16;

	private ThemeData data;
	private IEntityImageFactory factory;
	private Image image;

	public static String ICONS_LOCATION = "config/model/";

	public ThemeLabelProvider(ThemeData data) {
		this(data, data.getProvider().getEntityImageFactory());
	}

	public ThemeLabelProvider(IEntityImageFactory factory) {
		this(null, factory);
	}

	public ThemeLabelProvider(ThemeData data, IEntityImageFactory factory) {
		this.data = data;
		this.factory = factory;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.LabelProvider#getImage(java.lang.Object)
	 */
	@Override
	public Image getImage(Object element) {
		if (image != null) {
			image.dispose();
			image = null;
		}

		boolean useLocId = true;
		if (element instanceof ThemeData) {
			useLocId = ((ThemeData) element).useLocId();
		}
		
		ImageDescriptor desc = getIconImageDescriptor(element, useLocId , DEFAULT_SIZE,
				DEFAULT_SIZE);

		if (desc != null) {
			image = desc.createImage();
		}

		return image;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.LabelProvider#getText(java.lang.Object)
	 */
	@Override
	public String getText(Object element) {
		element = getData(element);
		if (element instanceof Task) {
			String name = ThemeUtil
					.toTitleCase(((Task) element).getThemeName());
			return (name == null) ? "" : name;
		}
		if (element instanceof PreviewElement) {
			return ((PreviewElement) element).getCompName();
		}
		if (element instanceof ThemePreview) {
			return "Preview";
		}
		if (element instanceof PreviewRefer) {
			return "Refer: " + ((PreviewRefer) element).getReferedScreenName();
		}
		if (element instanceof ThemeBasicData) {
			String name = ((ThemeBasicData) element).getThemeName();
			if (name == null) {
				return "";
			}
			return name;
		}
		return super.getText(element);
	}

	private IImage getHint() {
		if (data != null) {
			ISkinnableEntityAdapter ska = (ISkinnableEntityAdapter) data
					.getAdapter(ISkinnableEntityAdapter.class);

			// 9-piece needs must be generated this way, defect fix.
			// (look at list/grid higlight, generated by getImageDescriptor(w,
			// h); -
			// defect)

			if (ska != null && (ska.isBitmap() || ska.isMultiPiece())) {
				// find out dimension
				if (ska.isMultiPiece() || ska.isConvertedFromMultipiece()) {
					/*
					 * for nine piece image, we must pass IImage aware of
					 * correct layout information, otherwise visual defects may
					 * occur
					 */				
					return ((IImageAdapter) data
							.getAdapter(IImageAdapter.class)).getImage(true);
				}				
			}
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.nokia.tools.theme.adapter.IThemeLabelProvider#getIconImageDescriptor(java.lang.Object,
	 *      int, int)
	 */
	public ImageDescriptor getIconImageDescriptor(Object element, int width,
			int height) {
		return getIconImageDescriptor(element, true, width, height);
	}

	public ImageDescriptor getIconImageDescriptor(Object element, boolean useLocId, int width,
			int height) {
		boolean isIcon = true;
		IImage hint = getHint();
		if (hint != null) {
			isIcon = (hint.getWidth() > width || hint.getHeight() > height);
		}
		return getImageDescriptorWithOptionalLocId(element, width, height,
				isIcon, useLocId);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see com.nokia.tools.theme.adapter.IThemeLabelProvider#getImageDescriptor(java.lang.Object,
	 *      int, int)
	 */
	public ImageDescriptor getImageDescriptor(Object element, int width,
			int height) {
		return getImageDescriptor(element, width, height, false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.nokia.tools.theme.adapter.IThemeLabelProvider#getImageDescriptor(java.lang.Object,
	 *      int, int)
	 */
	public ImageDescriptor getImageDescriptor(Object element, final int width,
			final int height, final boolean isIcon) {
		return getImageDescriptorWithOptionalLocId(element, width, height,
				isIcon, true);
	}

	private ImageDescriptor getImageDescriptorWithOptionalLocId(Object element,
			final int width, final int height, final boolean isIcon, final boolean useLocId) {
		
		element = getData(element);
		if (!(element instanceof ThemeBasicData)) {
			return null;
		}
		if (element instanceof ComponentGroup) {
			ImageDescriptor desc = getComponentGroupImageDescriptor(
					(ComponentGroup) element, width, height);
			if (desc != null) {
				return desc;
			}
		}
		if (element instanceof Task) {
			Task task = (Task) element;
			ImageDescriptor desc = null;
			if (task.getFixedPicture() != null) {
				CoreImage img = new CoreImage();
				try {
					String picture = task.getFixedPicture();

					File imgFile = new File(picture);
					if (imgFile.exists()) {
						img
								.load(imgFile, width, height,
										CoreImage.SCALE_TO_FIT);
						desc = ImageDescriptor.createFromImage(img.getSwt());
					}
					img.dispose();
				} catch (Exception e) {
					e.getMessage();
				}

			}
			// if image not found set default
			if (desc == null) {
				desc = getTaskImageDescriptor((Task) element, width, height);
			}
			if (desc != null) {
				return desc;
			}
		}
		PreviewElement preview = null;
		if (element instanceof PreviewElement) {
			preview = (PreviewElement) element;
		}
		element = ((ThemeBasicData) element).getSkinnableEntity();
		//if (!useLocId)  ((SkinnableEntity) element).setAttribute("USELOCID", "FALSE");
		final PreviewElement previewElement = preview;
		final ThemeBasicData themeData = (ThemeBasicData) element;

		Object cacheGroup = null;
		String key = null;
		if (themeData != null) {
			cacheGroup = themeData.getImageCacheGroup();
			key = themeData.getImageCacheKey(width, height);
			RenderedImageDescriptor image = (RenderedImageDescriptor) SimpleCache
					.getData(cacheGroup, key);
			if ((image != null)&&(useLocId)) {
				return image;
			}
		}

		RenderedImageDescriptor desc = new RenderedImageDescriptor(
				new IRenderedImageProvider() {

					/*
					 * (non-Javadoc)
					 * 
					 * @see com.nokia.tools.media.utils.layers.IRenderedImageProvider#getHeight()
					 */
					public int getHeight() {
						return height;
					}

					/*
					 * (non-Javadoc)
					 * 
					 * @see com.nokia.tools.media.utils.layers.IRenderedImageProvider#getRenderedImage()
					 */
					public RenderedImage getRenderedImage() {
						if (themeData instanceof Theme) {
							return ((Theme) themeData).getBackGround();
						}
						if (themeData instanceof SkinnableEntity) {
							if (themeData.isEntityType().equals(
									ThemeTag.ELEMENT_COLOUR)) {
								try {
									ColourGraphic graphic = (ColourGraphic) ((SkinnableEntity) themeData)
											.getThemeGraphic();
									if (graphic != null)
										return graphic.generateIcon(width,
												height);
								} catch (Exception e) {
									Activator.error(e);
									return null;
								}
							}
							IImage hint = getHint();
							if (hint != null) {
								if (!useLocId) {
									if ((themeData!=null)&&(themeData.getSkinnableEntity() !=null)) themeData.getSkinnableEntity().setAttribute(ThemeConstants.ATTR_USE_LOCID, "false");
								} else{
									if ((themeData!=null)&&(themeData.getSkinnableEntity() !=null)) themeData.getSkinnableEntity().setAttribute(ThemeConstants.ATTR_USE_LOCID, null);
								}

								RenderedImage renderedImg = hint.getAggregateImage(width, height);
								if (!useLocId) {
									if ((themeData!=null)&&(themeData.getSkinnableEntity() !=null)) themeData.getSkinnableEntity().setAttribute(ThemeConstants.ATTR_USE_LOCID, null);
								}								
								return renderedImg;
							}
							if (isIcon) {
								return factory.createEntityImage(
										(SkinnableEntity) themeData,
										previewElement, 0, 0)
										.getAggregateImage(width, height);
							}
							return factory.createEntityImage(
									(SkinnableEntity) themeData, null, width,
									height).getAggregateImage();
						}
						return null;
					}

					/*
					 * (non-Javadoc)
					 * 
					 * @see com.nokia.tools.media.utils.layers.IRenderedImageProvider#getWidth()
					 */
					public int getWidth() {
						return width;
					}

				});

		if (themeData != null) {
			// we don't want to cache too big objects hard way
			// cache group using the content root so it will guarantee the
			// cached data are freed when the content is disposed
			SimpleCache.cache(cacheGroup, key, desc, true);
		}
		//if (!useLocId)  ((SkinnableEntity) element).setAttribute("USELOCID", null);
		return desc;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.BaseLabelProvider#dispose()
	 */
	@Override
	public void dispose() {
		if (image != null) {
			image.dispose();
		}
		super.dispose();
	}

	protected ThemeBasicData getData(Object element) {
		if (element instanceof ThemeBasicData) {
			return (ThemeBasicData) element;
		}
		if (element instanceof EditPart) {
			return getData(((EditPart) element).getModel());
		}
		if (element instanceof EditObject) {
			return getData(EditingUtil.getBean((EditObject) element));
		}
		if (element instanceof ThemeData) {
			return getData(((ThemeData) element).getData());
		}
		return null;
	}

	protected ImageDescriptor getComponentGroupImageDescriptor(
			ComponentGroup component, int width, int height) {
		return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
				"icons/icons16/bullet.gif");
	}

	protected ImageDescriptor getTaskImageDescriptor(Task task, int width,
			int height) {
		String name = task.getThemeName();
		if ("BACKGROUND".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("background.png", width, height));
		}

		if ("AREAS".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("areas.png", width, height));
		}

		if ("ICONS".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("icons.png", width, height));
		}

		if ("HIGHLIGHTS".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("highlight.png", width, height));
		}

		if ("SCROLL BARS AND LINES".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("lines.png", width, height));
		}

		if ("POP-UP WINDOWS".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("popup.png", width, height));
		}

		if ("THIS AND THAT".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("misc.png", width, height));
		}

		if ("INDICATORS".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("misc.png", width, height));
		}

		if ("APPLICATIONS".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("applications.png", width, height));
		}

		if ("MULTIMEDIA".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("sound.png", width, height));
		}

		if ("COLOURS".equalsIgnoreCase(name)) {
			return Activator.imageDescriptorFromPlugin(Activator.PLUGIN_ID,
					getImagePath("color.png", width, height));
		}
		return null;
	}

	private String getImagePath(String name, int width, int height) {
		if (width == 16 && height == 16) {
			return "icons/icons16/" + name;
		}
		return "icons/icons24/" + name;
	}

	/**
	 * Don't use loc id for populating icon view page
	 */
	public ImageDescriptor getImageDescriptor(Object element, boolean useLocId,
			int width, int height) {
		return getImageDescriptorWithOptionalLocId(element, width, height,
				false, useLocId);
	}

}
