/*
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of the License "Symbian Foundation License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description:
 *
 */
package com.nokia.svg2svgt;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.ResourceBundle;
import java.util.StringTokenizer;

import com.nokia.svg2svgt.configuration.Configuration;
import com.nokia.svg2svgt.configuration.ConfigurationManager;
import com.nokia.svg2svgt.log.LogMessages;
import com.nokia.svg2svgt.log.LogWriter;
import com.nokia.svg2svgt.util.WCMatcher;

/**
 * Parses the command line argument specified by the user while launching the
 * Converter tool. Returns the parsed arguments as a Hashtable. The possible
 * command line options are:
 * <p>
 * -svgfile or -svglist : This argument specifies the SVG file paths or text
 * file containing the SVG file paths. -svgfile option is used for specifying
 * the SVG file whereas, -svglist option is used to specify the file containing
 * the list of SVG files.
 * </p>
 * <p>
 * -outdir : This argument specifies the output directory where the SVGT files
 * are to be placed. If this parameter is not specified, a new directory SVGT
 * files is created in the current working directory.
 * </p>
 * <p>
 * -g : This argument specifies the grey list file name. This file contains the
 * elements and attributes that should generate a warning message during the
 * conversion process.
 * </p>
 * <p>
 * -b : This argument specifies the black list file name. This file contains the
 * elements and attributes that should be removed during the conversion process.
 * The black list file has a preference over the grey list file i.e. if the same
 * element/attribute is defined in both black and grey list files, the
 * element/attribute will be removed.
 * </p>
 */

public class CommandLineParser implements SVG2SVGTConstants {

	private static LogWriter myLogWriter = null;

	/**
	 * Parses the commmand line arguments and returns the command line data as a
	 * Hashtable.
	 * 
	 * @param args
	 *            String array contaning the command line arguments.
	 * @return Hashtable Hashtable containing parsed arguments.
	 * @throws Exception
	 *             If invalid arguments are specified.
	 */
	public static Hashtable parseCommandLineArguments(String[] args)
			throws Exception {
		myLogWriter = (LogWriter) ServiceRegistry
				.getService("com.nokia.svg2svgt.log.Logger");
		if (null == myLogWriter) {
			myLogWriter = new LogWriter(SVG2SVGTConstants.DEFAULT_REGION);
			ServiceRegistry.registerService(myLogWriter);
		}

		// parse the command line arguments, if invalid arguments, return error
		// null args has already been checked, no need to do the check again
		int argsCount = args.length;
		Hashtable parsedCmdLineParams = new Hashtable();
		parsedCmdLineParams.put(SVG2SVGTConstants.CONVERSIONSFILE_OPTION,
				ResourceBundle.getBundle("SVG2SVGTProperties").getString(
						"SVGT_CONVERSION_FILEPATH"));

		// just a check....too pessimistic about the user ;)
		if (0 > argsCount) {
			return parsedCmdLineParams;
		}

		String argsValue = null;
		ArrayList cmdLineParams = new ArrayList(argsCount);

		// copy the array contents into a collection
		for (int i = 0; i < argsCount; i++) {
			cmdLineParams.add(args[i]);
		}
		// create a iterator for iterating the list
		Iterator iter = cmdLineParams.iterator();
		String optionValue = null;

		while (true == iter.hasNext()) {
			argsValue = (String) iter.next();

			// if -svglist option encountered
			if ((SVG2SVGTConstants.SVGFILELIST_OPTION)
					.equalsIgnoreCase(argsValue)) {

				if (true == iter.hasNext()) {
					// this is the value for this option
					optionValue = (String) iter.next();
					raiseEvent(SVG2SVGTConstants.CMDLINE_OPTION_ENCOUNTERED,
							new String[] { argsValue, optionValue });
					if (true == isAnOption(optionValue)) {
						// invalid option value
						raiseException(SVG2SVGTConstants.INVALID_OPTION_ERROR,
								new String[] { optionValue, argsValue });
						// break;
					}
					resolveSVGFileListOption(parsedCmdLineParams, optionValue);
					continue;
				} else {
					// No option value
					raiseException(SVG2SVGTConstants.NO_OPTION_ERROR,
							new String[] { argsValue });
					// break;
				}
			}
			// -o option
			else if ((SVG2SVGTConstants.OUTDIR_OPTION)
					.equalsIgnoreCase(argsValue)) {
				if (true == iter.hasNext()) {
					// this is the value for this option
					optionValue = (String) iter.next();
					raiseEvent(SVG2SVGTConstants.CMDLINE_OPTION_ENCOUNTERED,
							new String[] { argsValue, optionValue });
					if (true == isAnOption(optionValue)) {
						// invalid command line argument
						raiseException(SVG2SVGTConstants.INVALID_OPTION_ERROR,
								new String[] { optionValue, argsValue });
						// break;
					}
					parsedCmdLineParams.put(SVG2SVGTConstants.OUTDIR_OPTION,
							optionValue);
					continue;
				} else {
					// no option value specified
					raiseException(SVG2SVGTConstants.NO_OPTION_ERROR,
							new String[] { argsValue });
					// break;
				}
			}
			// -g option
			/*
			 * else if ( ( SVG2SVGTConstants.GREYFILE_OPTION ).equalsIgnoreCase(
			 * argsValue ) ) { if ( true == iter.hasNext() ) { // this is the
			 * value for this option optionValue = ( String ) iter.next();
			 * raiseEvent( SVG2SVGTConstants.CMDLINE_OPTION_ENCOUNTERED, new
			 * String[]{ argsValue, optionValue }); if ( true == isAnOption(
			 * optionValue ) ) { // invalid command line argument
			 * raiseException( SVG2SVGTConstants.INVALID_OPTION_ERROR, new
			 * String[]{optionValue,argsValue } ); //break; }
			 * parsedCmdLineParams.put( SVG2SVGTConstants.GREYFILE_OPTION,
			 * optionValue ); continue; } else { // no option value
			 * raiseException( SVG2SVGTConstants.NO_OPTION_ERROR, new String[]{
			 * argsValue } ); //break; } } // -b option else if ( (
			 * SVG2SVGTConstants.BLACKFILE_OPTION ).equalsIgnoreCase( argsValue
			 * ) ) { if ( true == iter.hasNext() ) { // this is the value for
			 * this option optionValue = ( String ) iter.next(); raiseEvent(
			 * SVG2SVGTConstants.CMDLINE_OPTION_ENCOUNTERED, new String[]{
			 * argsValue, optionValue }); if ( true == isAnOption( optionValue )
			 * ) { // invalid command line argument raiseException(
			 * SVG2SVGTConstants.INVALID_OPTION_ERROR, new
			 * String[]{optionValue,argsValue } ); //break; }
			 * parsedCmdLineParams.put( SVG2SVGTConstants.BLACKFILE_OPTION,
			 * optionValue ); continue; } else { // no option value
			 * raiseException( SVG2SVGTConstants.NO_OPTION_ERROR, new String[]{
			 * argsValue } ); //break; } }
			 */
			// -b option
			else if ((SVG2SVGTConstants.PASSTHROUGHFILE_OPTION)
					.equalsIgnoreCase(argsValue)) {
				if (true == iter.hasNext()) {
					// this is the value for this option
					optionValue = (String) iter.next();
					raiseEvent(SVG2SVGTConstants.CMDLINE_OPTION_ENCOUNTERED,
							new String[] { argsValue, optionValue });
					if (true == isAnOption(optionValue)) {
						// invalid command line argument
						raiseException(SVG2SVGTConstants.INVALID_OPTION_ERROR,
								new String[] { optionValue, argsValue });
						// break;
					}
					parsedCmdLineParams.put(
							SVG2SVGTConstants.PASSTHROUGHFILE_OPTION,
							optionValue);
					continue;
				} else {
					// no option value
					raiseException(SVG2SVGTConstants.NO_OPTION_ERROR,
							new String[] { argsValue });
					// break;
				}
			}
			// -file option
			else if ((SVG2SVGTConstants.CMDLINEFILE_OPTION)
					.equalsIgnoreCase(argsValue)) {
				if (true == iter.hasNext()) {
					// this is the value for this option
					optionValue = (String) iter.next();
					raiseEvent(SVG2SVGTConstants.CMDLINE_OPTION_ENCOUNTERED,
							new String[] { argsValue, optionValue });
					if (true == isAnOption(optionValue)) {
						// invalid command line argument
						raiseException(SVG2SVGTConstants.INVALID_OPTION_ERROR,
								new String[] { optionValue, argsValue });
						// break;
					}
					// parse the file
					resolveFileOption(parsedCmdLineParams, optionValue);
					continue;
				} else {
					// no option value
					raiseException(SVG2SVGTConstants.NO_OPTION_ERROR,
							new String[] { argsValue });
					// break;
				}
			}
			// -help option
			else if ((SVG2SVGTConstants.HELP_OPTION)
					.equalsIgnoreCase(argsValue)) {
				// show command line tool usage to the user
				displayCommandLineUsage();
			} else {
				// the files have been specified manually
				String[] fileArray = null;
				ArrayList fileList = new ArrayList();
				fileArray = checkWildCharacters(argsValue);
				if (null != fileArray) {
					for (int i = 0; i < fileArray.length; i++) {
						fileList.add(fileArray[i]);
					}
				}

				while (true == iter.hasNext()) {
					optionValue = (String) iter.next();
					if (true == isAnOption(optionValue)) {
						// invalid argument
						raiseException(SVG2SVGTConstants.INVALID_OPTION_ERROR,
								new String[] { optionValue, argsValue });
						// break;
					}
					// resolve the wild char characters if any
					fileArray = checkWildCharacters(optionValue);
					if (null != fileArray) {
						for (int i = 0; i < fileArray.length; i++) {
							fileList.add(fileArray[i]);
						}
						// fileList.add( fileArray );
					}
				}
				parsedCmdLineParams.put(SVG2SVGTConstants.SVGFILE_OPTION,
						fileList);
			}
		} // end of while loop

		validateCommandLineArguments(parsedCmdLineParams);
		return parsedCmdLineParams;
	}

	/**
	 * Validates the command line arguments. If absent, uses the default values
	 * for them.
	 * 
	 * @param parsedCmdLineParams
	 *            Command line arguments.
	 */
	private static void validateCommandLineArguments(
			Hashtable parsedCmdLineParams) throws Exception {
		// no conversions file specified.
		if (null == parsedCmdLineParams
				.get(SVG2SVGTConstants.CONVERSIONSFILE_OPTION)) {
			parsedCmdLineParams.put(SVG2SVGTConstants.CONVERSIONSFILE_OPTION,
					ResourceBundle.getBundle("SVG2SVGTProperties").getString(
							"SVGT_CONVERSION_FILEPATH"));
			// "..\\config\\Conversions.properties");
		}
		/*
		 * // no black file specified if ( null == parsedCmdLineParams.get(
		 * SVG2SVGTConstants.BLACKFILE_OPTION ) ) { parsedCmdLineParams.put(
		 * SVG2SVGTConstants.BLACKFILE_OPTION, ".\\config\\BlackList.xml" ); }
		 * 
		 * // no grey file specified if ( null == parsedCmdLineParams.get(
		 * SVG2SVGTConstants.GREYFILE_OPTION ) ) { parsedCmdLineParams.put(
		 * SVG2SVGTConstants.GREYFILE_OPTION, ".\\config\\GreyList.xml" ); }
		 */

		// no grey file specified
		if (null == parsedCmdLineParams
				.get(SVG2SVGTConstants.PASSTHROUGHFILE_OPTION)) {
			parsedCmdLineParams.put(SVG2SVGTConstants.PASSTHROUGHFILE_OPTION,
					ResourceBundle.getBundle("SVG2SVGTProperties").getString(
							"SVGT_DEFAULT_PASSTHROUGH"));
			// "..\\config\\passThrough.xml" );
		}

		// no output directory specified
		if (null == parsedCmdLineParams.get(SVG2SVGTConstants.OUTDIR_OPTION)) {
			parsedCmdLineParams.put(SVG2SVGTConstants.OUTDIR_OPTION,
					ResourceBundle.getBundle("SVG2SVGTProperties").getString(
							"SVGT_DEFAULT_OUTPUTDIR"));
			// "..\\svgtfiles" );
		}

		// no input files specified
		if (null == parsedCmdLineParams.get(SVG2SVGTConstants.SVGFILE_OPTION)) {
			raiseException(SVG2SVGTConstants.MISSING_INPUT_FILES, null);
		}
	}

	/**
	 * Reads the string specified in the method arguments and checks if any wild
	 * characters are specified for file selection. If such characters are
	 * specified, it searches the specified directory or the current directory
	 * for the specified pattern and returns the file names as an array of
	 * Strings.
	 * 
	 * @param argsValue
	 *            Pattern to be searched.
	 * @return String[] String array containing absolute file names.
	 */
	private static String[] checkWildCharacters(String argsValue)
			throws Exception {
		String fileSeparator = System.getProperty("file.separator");
		int fileSeparatorIndex = argsValue.lastIndexOf(fileSeparator);
		String directoryName = null;
		String fileName = null;
		if (-1 != fileSeparatorIndex) {
			// check the specified directory
			directoryName = argsValue.substring(0, fileSeparatorIndex);
		} else {
			// check the current directory
			directoryName = ".";
		}
		fileName = argsValue.substring(fileSeparatorIndex + 1, argsValue
				.length());
		if ((null == fileName) || (0 >= fileName.length())) {
			// error no filename specified
			raiseException(SVG2SVGTConstants.NO_FILENAME_ERROR, null);
		}

		File parent = new File(directoryName);
		if (false == parent.isDirectory()) {
			// should return from here

		}

		// String[] fileList = parent.list( new SVGFilenameFilter( fileName ) );
		// if ( null != fileList || fileName != null)
		// {
		// for ( int i = 0; i < fileList.length; i++ )
		// {
		// // form the absolute path for the files
		// fileList[i] = directoryName + System.getProperty("file.separator") +
		// fileList[i];
		// }
		// }
		String[] fileList = new String[1];
		fileList[0] = directoryName + System.getProperty("file.separator")
				+ fileName;

		return fileList;
	}

	/**
	 * Resolves -svglist option. Reads the file containing the file list. Each
	 * file is specified in the input file on a new line.
	 * 
	 * @param parsedCmdLineParams
	 *            Hashtable containing the parsed command line options.
	 * @param paramValue
	 *            The file name containing the list of files.
	 */
	private static void resolveSVGFileListOption(Hashtable parsedCmdLineParams,
			String paramValue) throws Exception {
		BufferedReader reader = null;
		String fileName = null;
		try {
			ArrayList fileList = new ArrayList();
			reader = new BufferedReader(new FileReader(paramValue));
			while (null != (fileName = reader.readLine())) {
				String[] files = checkWildCharacters(fileName);
				if (null != files) {
					for (int i = 0; i < files.length; i++) {
						fileList.add(files[i]);
					}
				}
				// fileList.add( fileName );
			}
			reader.close();
			if (0 < fileList.size()) {
				parsedCmdLineParams.put(SVG2SVGTConstants.SVGFILE_OPTION,
						fileList);
			}
		} catch (FileNotFoundException fnfex) {
			raiseException(SVG2SVGTConstants.FILE_READING_ERROR, new String[] {
					fileName, fnfex.getMessage() });
		} catch (IOException iex) {
			raiseException(SVG2SVGTConstants.FILE_READING_ERROR, new String[] {
					fileName, iex.getMessage() });
		}

	}

	/**
	 * Resolves -file option i.e. it reads the file containing the command line
	 * parameters and populates the parameters in the Hashtable. If -svgfile
	 * option is used with wild characters, it resolves these characters to the
	 * equivalent file names. If -svglist option is used, it reads the file and
	 * populates the Hahstable accordingly.
	 * 
	 * @param parsedCmdLineParams
	 *            Hashtable containing parsed command line parameters.
	 */
	private static void resolveFileOption(Hashtable parsedCmdLineParams,
			String fileName) throws Exception {
		// get the value for this option
		// String fileName = ( String ) parsedCmdLineParams.get(
		// SVG2SVGTConstants.CMDLINEFILE_OPTION );

		// create a configuration manager for reading this file
		ConfigurationManager configMgr = new ConfigurationManager(fileName);
		Configuration config = configMgr.getConfiguration();

		// read the configuration as a Hashtable
		Hashtable params = config.toHash();
		if (null != params) {
			parsedCmdLineParams.putAll(params);
			// System.out.println(parsedCmdLineParams);

			if (parsedCmdLineParams
					.containsKey(SVG2SVGTConstants.CMDLINEFILE_OPTION)) {
				raiseException(
						SVG2SVGTConstants.WRONG_FILE_SPECIFICATION_ERROR, null);
			} else if (parsedCmdLineParams
					.containsKey(SVG2SVGTConstants.SVGFILELIST_OPTION)) {
				// resolve -svglist option if present in the file
				fileName = (String) parsedCmdLineParams
						.get(SVG2SVGTConstants.SVGFILELIST_OPTION);
				resolveSVGFileListOption(parsedCmdLineParams, fileName);
			} else if (parsedCmdLineParams
					.containsKey(SVG2SVGTConstants.SVGFILE_OPTION)) {
				// edit the -svgfile options if it exists
				fileName = (String) parsedCmdLineParams
						.get(SVG2SVGTConstants.SVGFILE_OPTION);

				// all the file names are specified as blank separated names
				StringTokenizer tokenizer = new StringTokenizer(fileName, " ");
				ArrayList fileList = new ArrayList();
				while (true == tokenizer.hasMoreTokens()) {
					fileName = tokenizer.nextToken();
					String[] nameList = checkWildCharacters(fileName);
					if (null != nameList) {
						for (int i = 0; i < nameList.length; i++) {
							// System.out.println( nameList[i] );
							fileList.add(nameList[i]);
						}

					}
				}

				// if valid files present, add to the list
				if (0 < fileList.size()) {
					parsedCmdLineParams.put(SVG2SVGTConstants.SVGFILE_OPTION,
							fileList);
				} else {
					raiseException(SVG2SVGTConstants.EMPTY_FILE_ERROR,
							new String[] { fileName });
				}
			} // end else-if ladder

		} // end if null params
	}

	/**
	 * Checks if this string matches any of the options.
	 * 
	 * @param value
	 *            String to be verfied
	 * @return true if an option, else false
	 */
	private static boolean isAnOption(String value) {
		// if ( value.equalsIgnoreCase( SVG2SVGTConstants.BLACKFILE_OPTION ) ||
		// value.equalsIgnoreCase( SVG2SVGTConstants.GREYFILE_OPTION ) ||
		if (value.equalsIgnoreCase(SVG2SVGTConstants.PASSTHROUGHFILE_OPTION)
				|| value.equalsIgnoreCase(SVG2SVGTConstants.CMDLINEFILE_OPTION)
				|| value.equalsIgnoreCase(SVG2SVGTConstants.HELP_OPTION)
				|| value.equalsIgnoreCase(SVG2SVGTConstants.OUTDIR_OPTION)
				|| value.equalsIgnoreCase(SVG2SVGTConstants.SVGFILE_OPTION)
				|| value
						.equalsIgnoreCase(SVG2SVGTConstants.CONVERSIONSFILE_OPTION)
				|| value.equalsIgnoreCase(SVG2SVGTConstants.SVGFILELIST_OPTION)) {
			return true;
		}
		return false;
	}

	/**
	 * Displays the command line tool usage to the user. Can be invoked by
	 * typing -help option on the command line.
	 */
	private static void displayCommandLineUsage() {
		System.out
				.println("===============================================================");
		System.out
				.println("=============== Usage of command line tool ====================");
		System.out
				.println("===============================================================");
		System.out.println("The available options are:");
		System.out
				.println("-svglist :  Specifies the file name containing the list of ");
		System.out.println("            the input SVG file(s)");
		System.out
				.println("-o       :  Specifies the ouptput directory where SVGT files");
		System.out.println("            are saved.");
		System.out.println("-p       :  Specifies the passthrough file name.");
		System.out
				.println("-help    :  Displays the help on command line usage.");
		System.out
				.println("-file    :  Specifies the file containing the commandline args.");
		System.out
				.println("===============================================================");
		System.out.println("\n\n");

	}

	/**
	 * Uses the <code>LogWriter</code> to log the exceptions to the standard
	 * output.
	 * 
	 * @param code
	 *            Code for retreiving the exception message.
	 * @param params
	 *            Options for the message.
	 */
	private static void raiseException(long code, Object[] params)
			throws Exception {
		if (null != myLogWriter) {

			String desc = LogMessages.getLogMessage(code, params);
			// throw new Exception( desc );

		}
	}

	/**
	 * Uses the <>LogWriter</code> to log the events to the standard output.
	 * 
	 * @param code
	 * @param params
	 */
	private static void raiseEvent(long code, Object[] params) {
		if (null != myLogWriter) {

			String desc = LogMessages.getLogMessage(code, params);
			myLogWriter.log(SVG2SVGTConstants.BATCH_MODE, desc);
		}
	}

	// ------ Internal Class - SVGFilenameFilter --------

	/**
	 * <code>SVGFilenameFilter</code> implements <code>FilenameFilter</code>. It
	 * provides the functionality for filtering the file present in a directory
	 * based on the criteria specified by the user.
	 * 
	 */
	private static class SVGFilenameFilter implements FilenameFilter {
		/**
		 * Specifies the filtering criteria set by the user.
		 */
		private String myFilteringCriteria = null;

		/**
		 * Creates a filter based on the criteria specified by the user.
		 * 
		 * @param filter
		 *            Filtering criteria specified by the user.
		 */
		public SVGFilenameFilter(String filter) {
			myFilteringCriteria = filter;
		}

		/**
		 * Tests if a specified file should be included in a file list.
		 * 
		 * @param dir
		 *            The directory in which the file was found
		 * @param name
		 *            The name of the file
		 * @return True if and only if the name should be included in the file
		 *         list; false otherwise
		 */
		public boolean accept(File dir, String name) {
			File temp = new File(name);
			if (true == temp.isDirectory()) {
				return false;
			}

			// apply wild char pattern
			return WCMatcher.match(myFilteringCriteria, name);
		}

	}

	/**
	 * Main method for testing purposes only
	 * 
	 * @param args
	 *            Command line arguments
	 */
	public static void main(String[] args) {
		// just for unit testing....
		try {
			Hashtable ht = CommandLineParser.parseCommandLineArguments(args);
			// System.out.println( ht );
			// String[] fileList = checkWildCharacters(
			// );
			// for ( int i=0; i< fileList.length; i++)
			// {
			// System.out.println( fileList[i] );
			// }

		} catch (Exception ex) {
			ex.printStackTrace();
		}

	}

}