/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.configuration;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Properties;

/**
 * This class provides access to the configuration storage. The
 * configuration files are normal Java property files with the
 * parameters specified in normal key=value pairs. This class
 * also provides methods to read and save the configuration file.
 */
public class ConfigurationStorage
{

    /**
     * Current configuration file name for which the configuration is to
     * be read/modified.
     */
    private String myCurrentConfigFileName = null;

    /**
     * File name used for saving the previous configuration for this file.
     */
    private String myPreviousConfigFileName = null;

    /**
     * The suffix used for saving the previuos configuration.
     */
    public final String PREVIOUS_SUFFIX = ".previous";


   /**
    * Creates a cofiguration storage for a given file name.
    *
    * @param fileName   Configuration file to be read.
    */
   public ConfigurationStorage(String fileName)
   {
        myPreviousConfigFileName = fileName + PREVIOUS_SUFFIX;
        myCurrentConfigFileName  = fileName;
   }



   /**
    * Reads the given configuration file.
    *
    * @return <code>Configuration</code> for the given file.
    */
   public Configuration readConfiguration()
   {
        return readFile( myCurrentConfigFileName );
   }


   /**
    * Saves the given configuration file.
    *
    * @param config <code>Configuration</code> to be saved.
    * @return       true if saving is successful, else false.
    */
   public boolean saveConfiguration( Configuration newConfig )
   {

       // 1# Make sure configuration files are existing
       // ------------------------------------------------
       // post condition:
       //  configuration   storage
       //  -------------   ----------
       //  current         .cfg
       //  previous        .cfg.previous
       //  new             <not existing>

       // 2# Write new configuration to a .cfg.new file
       // -------------------------------------------
       // post condition:
       //  configuration   storage
       //  -------------   ----------
       //  current         .cfg
       //  previous        .cfg.previous
       //  new             .cfg.new

       // 3# move current to current.temp
       // -------------------------------------------
       // post condition:
       //  configuration   storage
       //  -------------   ----------
       //  current         .cfg.temp
       //  previous        .cfg.previous
       //  new             .cfg.new


       // 4# move .previous to .previous.temp
       // -------------------------------------------
       // post condition:
       //  configuration   storage
       //  -------------   ----------
       //  current         .cfg.temp
       //  previous        .cfg.previous.temp
       //  new             .cfg.new

       // 5# move new to current
       // -------------------------------------------
       // post condition:
       //  configuration   storage
       //  -------------   ----------
       //  current         .cfg.temp
       //  previous        .cfg.previous.temp
       //  new             .cfg

       // 6# move current.temp to previous
       // ----------------------------------------
       // post condition:
       //  configuration   storage
       //  -------------   ----------
       //  current         .cfg.previous
       //  previous        .cfg.previous.temp
       //  new             .cfg

       // 7# delete previous
       // -----------------------------
       // post condition:
       //  configuration   storage
       //  -------------   ----------
       //  current         .cfg.previous
       //  previous        <not existing>
       //  new             .cfg




       String newConfFileName = myCurrentConfigFileName + ".new";
       String tempCurrFileName = myCurrentConfigFileName + ".temp";
       String tempPrevFileName = myPreviousConfigFileName + ".temp";
       File currentF = new File( myCurrentConfigFileName);
       File previousF = new File( myPreviousConfigFileName);
       File newF = new File( newConfFileName );
       File tempCurrentF = new File( tempCurrFileName );
       File tempPreviousF = new File( tempPrevFileName );

       // 1#
       //Check the existence of the files
       try
       {
           //Current
           if ( !( currentF.exists() ) )
           {
               FileOutputStream currentFOut = new FileOutputStream( myCurrentConfigFileName);
               currentFOut.write(0);
               currentFOut.close();
           }

           //previous
           if( !( previousF.exists() ) )
           {
               FileOutputStream previousFOut = new FileOutputStream( myPreviousConfigFileName );
               previousFOut.write(0);
               previousFOut.close();
           }
       }
       catch( IOException e )
       {
           System.out.println("Configuration Storage " + e);
           return false;
       }

       // 2#
       //write new configuration to a .cfg.new file
       try
       {
           FileOutputStream temporaryNewPropFile = new FileOutputStream(newConfFileName);
           Properties newProp = confToProp( newConfig );
           newProp.save(temporaryNewPropFile, newConfFileName);
           temporaryNewPropFile.close();
       }
       catch (IOException e)
       {
           // remove temporary file
           File f = new File(newConfFileName);
           f.delete();
           //System.out.println("ConfigurationStorage.saveConfiguration: " + e);
           return false;
       }


       boolean renameOk = false;

       // 3#
       //rename current to current.tmp
       renameOk  = currentF.renameTo(tempCurrentF);

       if ( !renameOk )
       {
           // failed to rename .cfg to .cfg.temp
           newF.delete();
           return false;
       }


       // 4#
       //rename previous to previous.temp
       renameOk = previousF.renameTo( tempPreviousF );
       if (!renameOk)
       {
           // failed to rename .cfg.previous to .cfg.previous.temp
           tempCurrentF.renameTo(currentF);
           newF.delete();
           return false;
       }

       // 5#
       //rename new to current
       renameOk = newF.renameTo(currentF);
       if ( !renameOk )
       {
           // failed to rename .new to .cfg
           tempPreviousF.renameTo(previousF);
           tempCurrentF.renameTo(currentF);
           newF.delete();
           return false;
       }


       // 6#
       //rename current.temp to previous
       renameOk = tempCurrentF.renameTo(previousF);
       if ( !renameOk )
       {
           // failed to rename .cfg.temp to .cfg.previous
           tempPreviousF.renameTo(previousF);
           tempCurrentF.renameTo(currentF);
           newF.delete();
           return false;
       }

       // 7#
       //delete previous
       tempPreviousF.delete();

       return true;
   }


   /**
    * Reads the configuration file from the absolute path specified in the
    * message argument.
    * @param fileName   Absoulte path of the file to be read.
    * @return           <code>Configuration</code> read from file.
    */
   private Configuration readFile( String fileName )
   {
       Configuration conf = new Configuration();
       Properties prop = new Properties();
       FileInputStream file = null;

       // read the file
       try
       {
           file = new FileInputStream( fileName );
           prop.load( file );
           file.close();
       }
       catch( FileNotFoundException e )
       {
          //System.out.println("File not found " + e);
          // return empty Configuration
          return conf;
       }
       catch(IOException e)
       {
           //System.out.println("IO Exception " + e);
           // return empty Configuration
           return conf;
       }
       propToConf(conf, prop );
       return conf;
   }



   /**
    * Populates the <code>Properties</code> contents into the
    * <code>Configuration</code> object.
    * @param conf   <code>Configuration</code> object to be populated
    * @param prop   <code>Properties</code> containing the configuration.
    */
   private void propToConf(Configuration conf, Properties prop )
   {
       // fill up Parameters
       Enumeration propNames = prop.propertyNames();
       while( true == propNames.hasMoreElements( ) )
       {
           String key = ( String )propNames.nextElement();
           //if (isAttribute(key))
           //{
           //    continue;
           //}
           String value = prop.getProperty( key );
           //int type = getType( key, prop );
           // construct a new parameter and put it into configuration
           Parameter par = null;
           par = new Parameter( key, value ); //, type );
           conf.put( par );

       }// while
   }



   /**
    * Converts the given configuration into <code>Properties</code>.
    * @param conf Input configuration.
    * @return     Properties instance.
    */
   private Properties confToProp(Configuration conf)
   {
       Properties prop = new Properties();

       //loop the Parameters
       Enumeration params = conf.elements();
       while(params.hasMoreElements())
       {
           Parameter par = (Parameter)params.nextElement();

           // Parameter itself
           String key = par.getKey();
           String value = par.getValue();
           prop.put(par.getKey(), par.getValue());
       }

       return prop;
   }


}





