/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.converter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.nokia.svg2svgt.log.Logger;
import com.nokia.svg2svgt.util.XMLFileParser;

/**
 * <code>GreyListAnalyzer</code> uses the grey list file and issues a warning for 
 * the attributes specified in grey list file.
 */
public class GreyListAnalyzer implements ConversionConstants 
{
	/**
	 * Parsed grey list document.
	 */
	private Document myDocument = null;
	
	/**
	 * List containg <code>SVGModules</code> in the grey list file.
	 */
	private ArrayList myGreyList = null;

    /**
     * Logger instance for logging.
     */
    private Logger myLogger      = null;


// PUBLIC METHODS


   /**
    * Creates an GreyListAnalyser that parses the grey list file.
    * @param fileName	Complete name for the gery file.
    */
   public GreyListAnalyzer( String fileName, Logger logger ) throws
   												FactoryConfigurationError,
   												ParserConfigurationException,
   												IOException,
   												SAXException,
   												IllegalArgumentException
   {
   	   myLogger = logger;
  	   myDocument = XMLFileParser.parseFile( fileName, false );
	   loadGreyListFile();
   }


   /**
    * Returns all the list of modules in the grey list file. 
    * @return	List of grey list modules.
    */
   public ArrayList getGreyListModules()
   {
   	   return myGreyList;
   }


   /**
    * Checks if an element or attribute by the tag name of the input node
    * is present in the grey list file or not.
    * @param node	Node to be searched.
    * @return		True if found, else false.
    */
   public boolean isGreyListedTag( Node node )
   {
	   String nodeName = node.getNodeName();
	   //System.out.println("In grey list...");
	   if ( Node.ELEMENT_NODE == node.getNodeType() )
	   {
		   SVGElement svgElem = this.fetchElementByTagName( nodeName );
		   if ( null != svgElem )
		   {
			   return svgElem.isWarned();
		   }
	   }
	   else if ( Node.ATTRIBUTE_NODE == node.getNodeType() )
	   {
		   SVGAttribute svgAttr = this.fetchAttributeByTagName( nodeName );
		   if ( null != svgAttr )
		   {
			   return true;
		   }   	   	   
	   }
	   return false;
   }


   /**
    * Checks if an element by the specified name is present in the grey
    * list or not.
    * @param elementName	Name of the element to be searched.
    * @return				Index of the element in the module list.
    */
   public int containsElement( String elementName )
   {
	   if ( null == myGreyList )
	   {
		   return -1;
	   }
	   Iterator iter = myGreyList.iterator();
	   int indexCount = 0;
	   SVGModule svgModule = null;
	   while( true == iter.hasNext() )
	   {
		  svgModule = ( SVGModule ) iter.next();
		  if ( -1 != svgModule.containsElement( elementName ) )
		  {
			return indexCount;
		  }
		  indexCount++;
	   }
	   return -1;
   }


   /**
    * Fetches the <code>SVGElement</code> instance containing the specified
    * attribute name. 
    * @param attrName	Attribute name to be searched.
    * @return			<code>SVGElement</code> instance if found, else null.
    */ 
   public SVGElement fetchElementContainingAttribute( String attrName )
   {
	   if ( null == myGreyList )
	   {
		   return null;
	   }
	   Iterator iter = myGreyList.iterator();
	   SVGModule svgModule = null;
	   while( true == iter.hasNext() )
	   {
		  svgModule = ( SVGModule ) iter.next();
		  return svgModule.getElementContainingAttribute( attrName ); 
	   }
	   return null;
   }


   /**
    * Fetches the <code>SVGElement</code> instance with the specified
    * element name.
    * @param elementName	Element name to be searched.
    * @return				<code>SVGElement</code> instance if found, else null.
    */
   public SVGElement fetchElementByTagName( String elementName )
   {
	   if ( null == myGreyList )
	   {
		   return null;
	   } 
	   Iterator iter = myGreyList.iterator();
	   SVGModule svgModule = null;
	   while( true == iter.hasNext() )
	   {
		  svgModule = ( SVGModule ) iter.next();
		  return svgModule.getElementByTagName( elementName ); 
	   }
	   return null;
   }


   /**
    * Fetches the <code>SVGAttribute</code> instance with the specified
    * attribute name.
    * @param attrName	Attribute name to be searched.
    * @return			<code>SVGAttribute</code> instance if found, else null.
    */
   public SVGAttribute fetchAttributeByTagName( String attrName )
   {
	   if ( null == myGreyList )
	   {
		   return null;
	   }
	   Iterator iter = myGreyList.iterator();
	   SVGModule svgModule = null;
	   int elemIndex = 0;
	   while( true == iter.hasNext() )
	   {
		  svgModule = ( SVGModule ) iter.next();
		  //return svgModule.getElementByTagName( attrName );
		  return svgModule.getAttributeByTagName( attrName );
	   }
	   return null;   	
   }

// PRIVATE METHODS


   /**
	* Parses the grey list file and stores the contents of the file
	* as <code>SVGModule</code> elements. 
	*
	*/
   private void loadGreyListFile()
   {
	   Element rootElement = myDocument.getDocumentElement();
	   String name = rootElement.getTagName();
	   // if not a black list file, return without loading or throw an exception
	   if ( !( rootElement.getTagName() ).equalsIgnoreCase( TAG_GREYLIST_ROOT ) )
	   {
		   return;
	   }

	   // get the module elements
	   NodeList moduleNodes = rootElement.getElementsByTagName( TAG_MODULE );
	   
	   Element module = null;
	   //ArrayList moduleNode = new ArrayList();
	   myGreyList = new ArrayList();
	   SVGModule svgModule = null;
	   String moduleName = null;
	   
	   // iterate through the available module nodes
	   for ( int i = 0; i < moduleNodes.getLength(); i++ )
	   {
		   // get the module node
		   module = ( Element ) moduleNodes.item( i );

		   // get the attribute for this node
		   moduleName = module.getAttribute( TAG_NAME );
		   if ( null != moduleName )
		   {
			   ArrayList elementNodes = resolveElementsInModule( module, moduleName );
			   if ( ( null != elementNodes ) || ( 0 < elementNodes.size() ) )
			   {
				   svgModule = new SVGModule( moduleName );
				   svgModule.addElements( elementNodes );
				   //moduleNode.add( svgModule );
				   myGreyList.add( svgModule );
			   }
		   } // end not a null module name
	   }// end for loop
   }



   /**
	* Resolves the element tags present in a single module element tag.
	* @param module		Module element to be searched.
	* @param moduleName	Name of this module.
	* @return			List of <code>SVGElements</code>
	*/
   private ArrayList resolveElementsInModule( Element module, 
											  String moduleName )
   {
	   Element element = null;
	   String elementName = null;
	   String isMarked = null;
	   SVGElement svgElement = null;
	   ArrayList elementList = new ArrayList();

	   // get all the elements by tag name element
	   NodeList elementNodes = module.getElementsByTagName( TAG_ELEMENT );

	   // iterate through the available list
	   for ( int i = 0; i < elementNodes.getLength(); i++ )
	   {
		   element = ( Element ) elementNodes.item( i );
		   
		   // get the name for this element
		   elementName = element.getAttribute( TAG_NAME );

		   // is this element marked for deletion
		   isMarked = element.getAttribute( TAG_REMOVE );
//		   if ( isMarked.equalsIgnoreCase( REMOVE_ELEMENT ) )
//		   {
//			   // no need to load the attributes, can do with element itself
//			   svgElement = new SVGElement( elementName, moduleName );
//			   svgElement.setMark( true );
//			   elementList.add( svgElement );
//			   continue;
//		   }
		   // if the attribute needs to be removed		   
		   if ( null != elementName )
		   {
			   svgElement = new SVGElement( elementName, moduleName );
			   svgElement.setWarn( false );
			   ArrayList attrList = resolveAttributesInElement( element, moduleName, elementName );
			   if ( ( null != attrList ) || ( 0 < attrList.size() ) )
			   {
				   svgElement.addAttributes( attrList );
			   }
			   elementList.add( svgElement );
			   continue;

		   } // end if null elementName
			   	   
	   } // end for - element nodes
	   return elementList;
   }



   /**
	* Resolves the attribute tag present in the element tag. If the 
	* attribute tag contains a reference to an attribute set, it resolves 
	* the attributes present in that attribute set also.
	* @param element	Element for which attributes are to be searched.
	* @param moduleName	Name of the module to which this element belongs.
	* @param elementName	Name of the element.
	* @return			List of <code>SVGAttributes<code> for this element.
	*/
   private ArrayList resolveAttributesInElement( Element element, 
												 String moduleName, 
												 String elementName )
   {
	   Attr attr = null;
	   Element attribute = null;
	   String attributeName = null;
	   String attributeValue = null;
	   SVGAttribute svgAttribute = null;
	   ArrayList attributesList = new ArrayList();

	   // get all the tags with name as attribute
	   NodeList attributeNodes = element.getElementsByTagName( TAG_ATTRIBUTE );
	   
	   // iterate through the retreived list
	   for ( int i = 0; i < attributeNodes.getLength(); i++ )
	   {
		   attribute = ( Element ) attributeNodes.item( i );
		   attr = attribute.getAttributeNode(TAG_NAME );
		   if ( null != attr )
		   {
			   // read the attribute name and value
			   attributeName = attr.getValue();
		       
			   // read the attribute value for this element
			   attr = attribute.getAttributeNode(TAG_VALUE );
			   if ( null != attr )
			   {
					attributeValue = attr.getValue();
			   }
			   // construct the SVGAttribute instance
			   svgAttribute = new SVGAttribute( attributeName, 
												elementName, 
												moduleName, 
												null, 
												attributeValue );
			   //svgElement.addAttribute( svgAttribute );
			   attributesList.add( svgAttribute );
		   }
		   else
		   {
			   attr = attribute.getAttributeNode( TAG_ATTRSET );
			   if ( null == attr )
			   {
				   // not attrSet element found, get next
				   continue;
			   }
               
			   String attrSetName = attr.getValue();
			   if ( null == attrSetName )
			   {
				   continue;
			   }

			   NodeList attrSet = myDocument.getElementsByTagName( attrSetName );
			   if ( null == attrSet )
			   {
					continue;
			   }
			   for ( int j = 0; j < attrSet.getLength(); j++ )
			   {
				   Element attrSetElement = ( Element ) attrSet.item( j );
				   ArrayList attrSetAttrs = resolveAttributesInElement( attrSetElement, moduleName, elementName );
				   if ( ( null != attrSetAttrs ) || ( 0 < attrSetAttrs.size() ) )
				   {
					   attributesList.addAll( attrSetAttrs );
				   }
			   }//end for attrSet
		   }
	   } // end for - attribute nodes
	   return attributesList;
   }

   
   /**
    * Removes non SVGT attributes from the input SVG file.
    */
   private void raiseWarningForTags() 
   {
   }
}
