/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.converter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.nokia.svg2svgt.util.XMLFileParser;
import com.nokia.svg2svgt.log.Logger;
import com.nokia.svg2svgt.SVG2SVGTConstants;

/**
 * <code>NameSpaceAnalyzer</code> removes non SVGT elements from the input SVG 
 * file. Additional elements to be removed can also be defined in the black 
 * list file. It also uses the grey list file and issues a warning for the 
 * elements specified in grey list file.
 * 
 */
public class NameSpaceAnalyzer implements ConversionConstants
{
   /**
    * Parsed black list document.
    */
   private Document myDocument = null;

   /**
	* List containg <code>SVGModules</code> in the black list file.
	*/
   private ArrayList myNameSpaceList = null;

   /**
    * Logger instance for logging events
    */
   private Logger myLogger = null;
   

// PUBLIC METHODS

   /**
	* Creates an NameSpaceAnalyser that parses the pass through file.
	* @param fileName	Complete name of the pass through file.
	*/   
   public NameSpaceAnalyzer( String fileName, Logger logger ) throws 
                                      FactoryConfigurationError,
	                                  ParserConfigurationException,
	                                  IOException,
	                                  SAXException,
	                                  IllegalArgumentException
   {
	   myLogger = logger;
	   this.logEvents( SVG2SVGTConstants.LOADING_FILE, 
					   new String[]{ fileName } );
	   myDocument = XMLFileParser.parseFile( fileName, false );
       // load the pass through  elements into a datastructure
       loadPassThroughFile();
       printPassThroughFile( myNameSpaceList );
       //System.out.println("Black list loaded successfully...");
   }

   /**
    * 
    * @param url
    * @return
    */    
   private SVGNameSpace getSVGNameSpacesByURL( String url ) 
    {
    	Iterator iter = myNameSpaceList.iterator();
    	SVGNameSpace nameSpace = null;
    	if ( null == url )
    	{
    		return null;
    	}
    	while (  iter.hasNext() )
    	{
    		nameSpace =( SVGNameSpace )iter.next();
    		if ( true == nameSpace.isURI() )
    		{
				if( nameSpace.getMyUri().equals( url ) )
				{
					return nameSpace;
				}
    		}
    		else
    		{
				if( ( null != nameSpace.getMyUri() ) && 
				    ( -1 != url.indexOf( nameSpace.getMyUri() ) ) )
				{
					return nameSpace;
				}
    		}
    	}
    	return null;
   }

   /**
    * Checks if an element or attribute by the tag name of the input node
    * is present in the black list file or not.
    * @param node	Node to be searched.
    * @return		True if found, else false.
    */
   public boolean isNodeAllowed( String nsName, String nsURL, String nodeName, short nodeType )
   {
   	   //System.out.println(nodeName);
   	   if ( ( null == nsName ) && ( null == nsURL ) )
   	   {
   	   	 return true;
   	   }
   	   Hashtable ht = SVG2SVGTConverter.getNSInCurrentDoc();
   	   Enumeration enum1 = ht.keys();
	   String reqdUrl = null;
       if ( null != nsName )
       {
   	       while( true == enum1.hasMoreElements())
   	       {
   	   	       String url = ( String )enum1.nextElement();
   	   	       ArrayList variables = ( ArrayList ) ht.get( url );
   	   	       Iterator iter = variables.iterator();

   	   	       while ( iter.hasNext() )
   	   	       {
   	   	  	      if( (( String) iter.next() ). equalsIgnoreCase( nsName )  )
   	   	  	      {
   	   	  		      reqdUrl = url;
   	   	  		      break;	
   	   	  	      }
   	   	       }
   	       }
       }
       else
       {
       	   reqdUrl = nsURL;
       }
	   SVGNameSpace nameSpace = null;
	   nameSpace = getSVGNameSpacesByURL( reqdUrl );
	
   	   
   	   if( null == nameSpace )
   	   {
   	   	 return false;
   	   }
   	   
   	   if ( true == nameSpace.isIsallowed() )
   	   {
   	   	  return true;
   	   }
   	   
   	   if ( Node.ELEMENT_NODE == nodeType )
   	   {
   	   	   SVGElement svgElem = nameSpace.getElementByTagName( nodeName );
   	   	   if ( null != svgElem )
   	   	   {
   	   	   	   return true;
   	   	   }
   	   }
   	   else if ( Node.ATTRIBUTE_NODE == nodeType )
   	   {
   	   	   SVGAttribute svgAttr = nameSpace.getAttributeByTagName( nodeName );
		   if ( null != svgAttr )
		   {
			   return true;
		   }   	   	   
   	   }
   	   return false;
   }


   /**
	* Checks if an element by the specified name is present in the pass through
	* file or not.
	* @param elementName	Name of the element to be searched.
	* @return				Index of the element in the module list.
	*/
   public int containsElement( String elementName )
   {
	   if ( null == myNameSpaceList )
	   {
		   return -1;
	   }
   	   Iterator iter = myNameSpaceList.iterator();
   	   int indexCount = 0;
   	   SVGNameSpace svgNameSpace = null;
   	   while( true == iter.hasNext() )
   	   {
   	   	  svgNameSpace = ( SVGNameSpace ) iter.next();
   	   	  if ( -1 != svgNameSpace.containsElement( elementName ) )
   	   	  {
   	   	  	return indexCount;
   	   	  }
   	   	  indexCount++;
   	   }
   	   return -1;
   }



   /**
	* Fetches the <code>SVGElement</code> instance containing the specified
	* attribute name. 
	* @param attrName	Attribute name to be searched.
	* @return			<code>SVGElement</code> instance if found, else null.
	*/ 
   public SVGElement fetchElementContainingAttribute( String attrName )
   {
	   if ( null == myNameSpaceList )
	   {
		   return null;
	   }
	   Iterator iter = myNameSpaceList.iterator();
	   SVGNameSpace svgNameSpace = null;
	   SVGElement svgElement = null;
	   while( true == iter.hasNext() )
	   {
		  svgNameSpace = ( SVGNameSpace ) iter.next();
	      svgElement = svgNameSpace.getElementContainingAttribute( attrName );
	      if ( null != svgElement )
	      {
	      	  return svgElement; 
	      }
	   }
	   return null;
   }


   /**
	* Fetches the <code>SVGElement</code> instance with the specified
	* element name.
	* @param elementName	Element name to be searched.
	* @return				<code>SVGElement</code> instance if found, else null.
	*/
   public SVGElement fetchElementByTagName( String elementName )
   {
	   if ( null == myNameSpaceList )
	   {
		   return null;
	   }
	   Iterator iter = myNameSpaceList.iterator();
	   SVGNameSpace svgNameSpace = null;
	   SVGElement svgElement = null;
	   //boolean isPresent = false;
	   while( true == iter.hasNext() )
	   {
		  svgNameSpace = ( SVGNameSpace ) iter.next();
		  svgElement = svgNameSpace.getElementByTagName( elementName );
		  if ( null != svgElement )
		  {
		  	  return svgElement; 
		  }
	   }
	   return null;
   }


   /**
	* Fetches the <code>SVGAttribute</code> instance with the specified
	* attribute name.
	* @param attrName	Attribute name to be searched.
	* @return			<code>SVGAttribute</code> instance if found, else null.
	*/
   public SVGAttribute fetchAttributeByTagName( String attrName )
   {
	   if ( null == myNameSpaceList )
	   {
	   	   return null;
	   }
	   Iterator iter = myNameSpaceList.iterator();
	   SVGNameSpace svgNameSpace = null;
	   SVGAttribute svgAttribute = null;
	   int elemIndex = 0;
	   while( true == iter.hasNext() )
	   {
		  svgNameSpace = ( SVGNameSpace ) iter.next();
	      //return svgModule.getElementByTagName( attrName );
          svgAttribute = svgNameSpace.getAttributeByTagName( attrName );
          if ( null != svgAttribute )
          {
          	  return svgAttribute;
          }
	   }
	   return null;   	
   }


   /**
    * Returns the list of all the namespaces in the pass through file.
    * @return	List of modules in pass through file.
    */
   public ArrayList getNSInPassThrough()
   {
   	   return ( ArrayList ) myNameSpaceList.clone();
   }

// PRIVATE METHODS


   /**
	* Parses the black list file and stores the contents of the file
	* as <code>SVGModule</code> elements. 
	*
	*/
   private void loadPassThroughFile()
   {
	   Element rootElement = myDocument.getDocumentElement();

	   // if not a black list file, return without loading or throw an exception
	   if ( !( rootElement.getTagName() ).equalsIgnoreCase( TAG_PASSTHROUGH_ROOT ) )
	   {
		   return;
	   }

	   // get the module elements
	   NodeList nameSpaceNodes = rootElement.getElementsByTagName( TAG_NAMESPACE );
	   
	   Element nameSpace = null;
	   //ArrayList moduleNode = new ArrayList();
	   myNameSpaceList = new ArrayList();// collection of modules
	   SVGNameSpace svgNameSpace = null;
	   String nameSpaceName = null;	
	   String url = null;
	   String urlContains = null;
	   boolean  isCompleteURI = false;
	   String  isWarningReqd = null;
	   // iterate through the available module nodes
	   for ( int i = 0; i < nameSpaceNodes.getLength(); i++ )
	   {
		   // get the module node
		   nameSpace = ( Element ) nameSpaceNodes.item( i );
           
		   // get the attribute for this node
		   nameSpaceName = nameSpace.getAttribute( TAG_NAME );		   		    
		   isWarningReqd = nameSpace.getAttribute( TAG_WARN );
		   //System.out.println( moduleName);
		   NodeList childNodes = null;
		   Node node = null;
		   String isAllAllowed = null;
		   if ( null != nameSpaceName )
		   {
			   childNodes = rootElement.getElementsByTagName( nameSpaceName );
			   node = childNodes.item( 0 ); //only one entry will be present
			   
			   isAllAllowed = ( ( Element )node ).getAttribute( TAG_ALLOWALL );
			   
			   svgNameSpace = new SVGNameSpace( nameSpaceName );
			   if( ( null != isWarningReqd) && 
			       ( isWarningReqd.equalsIgnoreCase( TAG_TRUE ) ) )
			   {
				  svgNameSpace.setWarn( true );
			   }
			   			   
			   if ( ( null != nameSpace.getAttribute( TAG_URL ) ) &&
			        ( 0 < ( nameSpace.getAttribute( TAG_URL ) ).length() ) ) 
			   {
				  svgNameSpace.setURI( true );
				  svgNameSpace.setMyUri( nameSpace.getAttribute( TAG_URL ) );
		       }
		       else if ( ( null != nameSpace.getAttribute( TAG_URLCONTAINS ) ) && 
			             ( 0 < ( nameSpace.getAttribute( TAG_URLCONTAINS ) ).length() ) )
			   {
				 svgNameSpace.setMyUri( nameSpace.getAttribute( TAG_URLCONTAINS ) );
			   }

			   if( ( null != isAllAllowed ) && ( isAllAllowed.equals( TAG_TRUE ) ) )
			   {
				  myNameSpaceList.add( svgNameSpace );
				  svgNameSpace.setIsallowed( true );
			   	  continue;		   	
			   }
			   else
			   {			   
				   ArrayList elementNodes = resolveElementsInNameSpaces( ( Element ) node , nameSpaceName );
				   svgNameSpace.addElements( elementNodes );
				   myNameSpaceList.add( svgNameSpace );				   
			  
			  }//end of the allow all 
		   
		   } // end not a null nameSpace name
	   }// end for loop
   }



   /**
	* Resolves the element tags present in a single module element tag.
	* @param module		Module element to be searched.
	* @param moduleName	Name of this module.
	* @return			List of <code>SVGElements</code>
	*/
   private ArrayList resolveElementsInNameSpaces( Element module, 
											  String moduleName )
   {
	   Element element = null;
	   String elementName = null;
	   String isWarn = null;
	   SVGElement svgElement = null;
	   ArrayList elementList = new ArrayList();

	   // get all the elements by tag name element
	   NodeList elementNodes = module.getElementsByTagName( TAG_ELEMENT );

	   // iterate through the available list
	   for ( int i = 0; i < elementNodes.getLength(); i++ )
	   {
		   element = ( Element ) elementNodes.item( i );
		   
		   // get the name for this element
		   elementName = element.getAttribute( TAG_NAME );
		   //System.out.println("Elements=" + elementName );

		   // is this element marked for deletion
		   isWarn = element.getAttribute( TAG_WARN );
		   //System.out.println( isMarked );
		   if ( isWarn.equalsIgnoreCase( TAG_TRUE ) )
		   {
			   // no need to load the attributes, can do with element itself
			   svgElement = new SVGElement( elementName, moduleName );
			   svgElement.setWarn( true );
				ArrayList attrList = resolveAttributesInElement( element, moduleName, elementName );
				if ( ( null != attrList ) || ( 0 < attrList.size() ) )
				{
					svgElement.addAttributes( attrList );
				}
			   elementList.add( svgElement );
			   continue;
		   }
		   // if the attribute needs to be removed		   
		   if ( null != elementName )
		   {
			   svgElement = new SVGElement( elementName, moduleName );
			   svgElement.setWarn( false );
			   ArrayList attrList = resolveAttributesInElement( element, moduleName, elementName );
			   if ( ( null != attrList ) || ( 0 < attrList.size() ) )
			   {
				   svgElement.addAttributes( attrList );
			   }
			   elementList.add( svgElement );
			   continue;

		   } // end if null elementName
			   	   
	   } // end for - element nodes
	   return elementList;
   }



   /**
	* Resolves the attribute tag present in the element tag. If the 
	* attribute tag contains a reference to an attribute set, it resolves 
	* the attributes present in that attribute set also.
	* @param element	Element for which attributes are to be searched.
	* @param moduleName	Name of the module to which this element belongs.
	* @param elementName	Name of the element.
	* @return			List of <code>SVGAttributes<code> for this element.
	*/
   private ArrayList resolveAttributesInElement( Element element, 
												 String moduleName, 
												 String elementName )
   {
	   Attr attr = null;
	   Element attribute = null;
	   String attributeName = null;
	   String attributeValue = null;
	   SVGAttribute svgAttribute = null;
	   ArrayList attributesList = new ArrayList();

	   // get all the tags with name as attribute
	   NodeList attributeNodes = element.getElementsByTagName( TAG_ATTRIBUTE );
	   
	   NamedNodeMap attrList = null;
	   // iterate through the retreived list
	   for ( int i = 0; i < attributeNodes.getLength(); i++ )
	   {
		   attribute = ( Element ) attributeNodes.item( i );
		   attrList = attribute.getAttributes();
		   if ( null != attrList )
		{ 
		  SVGAttribute svgNameTag = null;
		  SVGAttribute svgValueTag = null;
		  ArrayList tempList = new ArrayList();
		  for ( int j = 0; j < attrList.getLength(); j++ )
		   	  {
		   	  	attr = ( Attr )attrList.item( j );
		   	    attributeName = attr.getName();
		   	    attributeValue = attr.getValue();
				svgAttribute = new SVGAttribute( attributeName, 
								 elementName, 
								 moduleName, 
								 null, 
								 attributeValue );
				if( attributeName.equals(TAG_NAME)) 
				{
					svgNameTag = svgAttribute; 
				}
				else if(attributeName.equalsIgnoreCase(TAG_VALUE)) 
				{
					svgValueTag = svgAttribute;
				}
				if(attributeName.equalsIgnoreCase(TAG_WARN))
				{
					if(attributeValue.equalsIgnoreCase(TAG_TRUE))
					{
					svgAttribute.setWarned(true);
					}
				}												 
											 
				// add this attribuet to the SVGElement
				tempList.add( svgAttribute );
			  }
			  if( ( svgNameTag != null ) ) 
			  { 
				String name = null;
				String value = null;
				boolean warn = false;		   	  	 
				for( int k =0; k< tempList.size(); k++ )
				{
					SVGAttribute temp = ( SVGAttribute )tempList.get( k );
					if( temp.getAttributeName().equalsIgnoreCase( TAG_NAME ) )
					{
						name = ( String ) temp.getAttributeValue();		   	  								
					}
					if( temp.getAttributeName().equalsIgnoreCase( TAG_VALUE ) )
					{
						value = ( String ) temp.getAttributeValue();		   	  								
					}
					if( temp.getAttributeName().equalsIgnoreCase( TAG_WARN ) )
					{
						warn = temp.isWarned();				
					}
								   	  		
				}
				svgAttribute = new SVGAttribute( name ,	elementName, moduleName, null, value );
				if( warn )
				{
					svgAttribute.setWarned( warn );		   	   
				}				
				attributesList.add( svgAttribute );
		   	  }
		   }
	   } // end for - attribute nodes
	   return attributesList;
   }


   /**
    * Prints the black list. Just used for testing.
    * @param moduleNode	SVGModules in black list
    */
   private void printPassThroughFile( ArrayList nameSpaceNode )
   {
	   for ( int i=0;i< nameSpaceNode.size(); i++)
	   {
		   SVGNameSpace nameSpace = (SVGNameSpace) nameSpaceNode.get(i);
	   }
   }


   private void logEvents( long code, String[] desc )
   {
   	
   }


   /**
    * Main method for testing purposes.
    * 
    * @param args	Command line arguments.
    */
   public static void main( String[] args )
   {
   	  try
   	  {
		  NameSpaceAnalyzer bla = new NameSpaceAnalyzer(".\\config\\passThrough.xml", null);
		  	  }
   	  catch( Exception ex )
   	  {
   	  	ex.printStackTrace();
   	  }
   	  
   }
   


  public boolean isNameSpacePresent( String nsName )
  {
  	 SVGNameSpace svgNS = null;
  	 for( int i = 0; i < myNameSpaceList.size(); i++ )
  	 {
  	 	svgNS = ( SVGNameSpace ) myNameSpaceList.get( i );
  	 	if ( ( svgNS.getnameSpaceName().equalsIgnoreCase( nsName ) ) && 
  	 	     ( null == svgNS.getMyUri() ) )
  	 	{
  	 		return true;
  	 	}
  	 	
  	 }
  	 return false;
  }



  public boolean isWarningTag( String nsName, String nsURL, String nodeName, short nodeType )
  {
	//System.out.println(nodeName);
	if ( ( null == nsName ) && ( null == nsURL ) )
	{
	  return false;
	}
	Hashtable ht = SVG2SVGTConverter.getNSInCurrentDoc();
	Enumeration enum1 = ht.keys();
	String reqdUrl = null;
	if ( null != nsName )
     {
	   	while( true == enum1.hasMoreElements())
	    {			   	   	
	      String url = ( String )enum1.nextElement();
	      ArrayList variables = ( ArrayList ) ht.get( url );
	      Iterator iter = variables.iterator();  	   	  
	      while ( iter.hasNext() )
	      {
		    if( (( String) iter.next() ). equalsIgnoreCase( nsName )  )
		    {
		    	reqdUrl = url;
		 	    break;	
		    }
	     }
	   }
     }  
	 else 
	 {
		reqdUrl = nsURL;
	 }
	   SVGNameSpace nameSpace = null;
	   nameSpace = getSVGNameSpacesByURL( reqdUrl );
	      	   
	if( null == nameSpace )
	{
	  return false;
	}
   	   
	if ( true == nameSpace.isWarn() )
	{
	   return true;
	}
   	   
	if ( Node.ELEMENT_NODE == nodeType )
	{
		SVGElement svgElem = nameSpace.getElementByTagName( nodeName );
		if ( ( null != svgElem ) && ( true == svgElem.isWarned() ) )
		{
			return true;
		}
	}
	else if ( Node.ATTRIBUTE_NODE == nodeType )
	{
		SVGAttribute svgAttr = nameSpace.getAttributeByTagName( nodeName );
		if ( ( null != svgAttr ) && ( true == svgAttr.isWarned() ) )
		{
			return true;
		}   	   	   
	}
	return false;
  }
 
}
