/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.converter;

import java.util.Enumeration;
import java.util.Hashtable;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.nokia.svg2svgt.log.Logger;
import com.nokia.svg2svgt.util.XMLFileParser;

/**
 * <code>NonReferredIdRemover</code> removes all the definitions (elements 
 * with id attributes) that are not referred to in the SVG file.
 * 
 */
public class NonReferredIdRemover implements ConversionConstants
{

   /**
    * Array list containing referenced ids.
    */
   //private static ArrayList myIds = new ArrayList();
   
   /**
    * Hashtable containing only the referenced identifier as keys and the
    * node name containing this identifier as value.
    */
   private static Hashtable myReferencedIdData = new Hashtable();
   
   /**
    * Hashtable containing all the identifier references as keys and the
    * node name containing this identifier as value.
    */
   private static Hashtable myAllIdData        = new Hashtable();
   
   /**
    * Hashtable containing the referenced identifiers for which the corresponding
    * identifiers have been deleted.
    */
   private static Hashtable myOrphanReferenceData = new Hashtable();


// PUBLIC METHODS

  
   /**
    * Removes all the definitions that are not referred to in the SVG file.
    * 
    * @param svgtDoc SVGT Document reference.
    */
   public static void processNonReferencedIDs( Document svgtDoc, Logger logger ) 
   {
   	   Element rootElem = svgtDoc.getDocumentElement();
	   //PrintDOMTree.printDOMTree( svgtDoc );
       if ( null == rootElem )
       {
       	  return;
       }

       // read all the referred ids in the document
   	   trackReferredIds( rootElem );
       trackOrphanReferences( );
   	   // removed ids not referenced
   	   removeNonReferencedIDs( rootElem );
   	   
   	   //System.out.println( myAllIdData );
	   //System.out.println( myReferencedIdData );
	   //System.out.println( myOrphanReferenceData );
   	   //PrintDOMTree.printDOMTree( svgtDoc );
   }


 
   /**
	* Gets the referenced identifiers in the SVGT document.
	* @return	Hashtable of referenced identifiers.
	*/
   //public static ArrayList getReferencedIdentifiers()
   public static Hashtable getReferencedIdentifiers()
   {
		//return myIds;
		return myReferencedIdData;
   }

// PRIVATE METHODS


   /**
    * Tracks the orpahn references i.e. attributes which refer to some nodes
    * but those nodes have been deleted in the SVGT conversion process.
    */
   private static void trackOrphanReferences()
   {
	   //Enumeration enum = myReferencedIdData.keys();
	   Enumeration enum1 = myAllIdData.keys();
	   while( enum1.hasMoreElements() )
	   {
		   String orphanRef = ( String ) enum1.nextElement();
		   //if ( false == myAllIdData.containsKey( orphanRef ) )
		   if ( false == myReferencedIdData.containsKey( orphanRef ) )
		   {
		   	    //myOrphanReferenceData.put( orphanRef, myReferencedIdData.get(orphanRef) );
			    myOrphanReferenceData.put( orphanRef, myAllIdData.get( orphanRef ) );
		   }
	   }
   }


   /**
    * Traverses through the SVGT document and removes all the non reference
    * identifiers from the svgt document tree.
    * @param node	Root element of the svgt document.
    */
   private static void removeNonReferencedIDs( Node node )
   {
	   NamedNodeMap attrList = null;
	   // get the attributes for the root element
	   if ( true == node.hasAttributes() )
	   {
	   	   attrList = node.getAttributes();
	   	   removeOrphanReferenceNodes( attrList, node );
	   	   // does it contain an "id" attribute?? 
	   	   Attr idAttr = ( ( Element ) node).getAttributeNode("id");
	   	   if ( null != idAttr )
	   	   {
			  // validate the identifier and remove it if not referenced.
			  checkAndRemoveNonReferredId( idAttr, node );
	   	   }
	   }
	   
	   // get the child nodes for the root element
	   NodeList nodeList = node.getChildNodes();
	   Node childNode = null;
	   for ( int i = 0; i < nodeList.getLength(); i++ )
	   {
		   childNode = nodeList.item( i );
		   if ( Node.ELEMENT_NODE != childNode.getNodeType() )
		   {
		   	   continue;
		   }
		   //System.out.println(childNode.getNodeName());
           // get the attributes for the root element
		   if ( true == childNode.hasAttributes() )
		   {
               attrList = childNode.getAttributes();
               removeOrphanReferenceNodes( attrList, childNode );
               // does it contain an "id" attribute??
			   Attr idAttr = ( ( Element ) node).getAttributeNode("id");
			   if ( null != idAttr )
			   {
				  // validate the identifier and remove it if not referenced.
				  checkAndRemoveNonReferredId( idAttr, childNode );
			   }
		   }
		   // recursive method call for traversing the entire tree
		   removeNonReferencedIDs( childNode );

	   } // end for node list
   }


   /**
    * Removes the oprhan reference nodes from the SVGT DOM tree.
    * @param attrList	Attribute list to be searched for orphan references.
    * @param parentNode	Node containing the attribute with orphan reference.
    */
   private static void removeOrphanReferenceNodes( NamedNodeMap attrList, Node parentNode )
   {
        if ( 0 == myOrphanReferenceData.size() )
        {
        	// all references are valid
        	return;
        }
        
        Enumeration enum1 = null;
        String orphanRef = null;
        for ( int i = 0; i < attrList.getLength(); i++ )
        {
			enum1 = myOrphanReferenceData.keys();
        	Attr attribute = ( Attr ) attrList.item( i );
        	String attrValue = attribute.getNodeValue();
        	while( true == enum1.hasMoreElements() )
        	{
        		 orphanRef = ( String ) enum1.nextElement();
        		 if ( -1 != attrValue.indexOf( orphanRef ) )
        		 {
        		 	NamedNodeMap map =  parentNode.getAttributes();
        		 	Node tempNode = map.getNamedItem(attribute.getName());
        		 	if(tempNode != null){
						( ( Element ) parentNode).removeAttributeNode( attribute );        		 		
        		 	}
        		 }
        	}
        } // end for loop
   }



   /**
    * Checks if the given id attribute has been referenced or not. If it is
    * not referenced, it removes it from the parent element that contains it.
    * @param attribute	Id attribute.
    * @param parentNode	Parent node containing this attribute.
    */
   private static void checkAndRemoveNonReferredId( Attr attribute, Node parentNode )
   {
   	   String attrValue = attribute.getValue();
   	   // check if the referenced identifier list contains this id or not
   	   //if ( false == myIds.contains( attrValue ) )
	   if ( false == myReferencedIdData.containsKey( attrValue ) )
   	   {
   	   	    ( ( Element )parentNode ).removeAttributeNode( attribute );
   	   }
   }


   /**
    * Searches the entire SVGT DOM tree for the referenced identifiers. The
    * reference is provided using "url(#<<identifier>>)" or through 
    * "#xpointer("<<identifier>>")" values.
    * @param node	Node in which reference is to be searched.
    */
   private static void trackReferredIds( Node node )
   {
	  //NodeList nodeList = node.getChildNodes();
	  String id = null;
	  //System.out.println(node.getNodeName());
	  Node attribute = null;
	  // get the attributes for this node.
	  if ( true == node.hasAttributes( ) )
	  {
		   NamedNodeMap attrList = node.getAttributes();
		   for ( int i = 0; i < attrList.getLength(); i++ )
		   {
			   attribute = attrList.item( i );
			   // check each attribute for reference
			   checkIdReference( ( Attr ) attribute, node );
		   }
		   
	  }
	  
	  // get the child elements,if any for this node
	  NodeList nodeList = node.getChildNodes();
	  Node childNode = null;
	  for ( int i = 0; i < nodeList.getLength(); i++ )
	  {
	  	   
	  	   childNode = nodeList.item( i );
		   if ( Node.ELEMENT_NODE != childNode.getNodeType() )
		   {
		   	   continue;
		   }
		   //System.out.println(childNode.getNodeName());
		   // for each element, get its attributes
		   if ( true == childNode.hasAttributes( ) )
		   {
			   NamedNodeMap attrList = childNode.getAttributes();
			   for( int j = 0; j < attrList.getLength(); j++ )
			   {
				  attribute = attrList.item( j );
				  // check each attribute for reference
				  checkIdReference( ( Attr ) attribute, childNode );
			   }
			   
		   }
		   // recursive method call for traversing the tree.
		   trackReferredIds( childNode );
	  }
	  //System.out.println( myIds);
	  //System.out.println( myReferencedIdData );
   }



   /**
    * Analyzes an attribute to check if its value is referencing to some
    * identifier. If a reference is found, it registers the identifier in
    * the list of used ids.
    * @param attribute	Attribute to be analyzed.
    */
   private static void checkIdReference( Attr attribute, Node parentNode )
   {
	   String attrName = null;
	   String attrValue = null;
	   //Node parentNode = null;
	   attrName = attribute.getNodeName();
	   attrValue = attribute.getNodeValue();
	   //System.out.println( attrName );
	   //System.out.println( attrValue );
	   
	   if ( attrName.equals( ID_ATTRIBUTE ) )
	   {
	   	   myAllIdData.put( attrValue, parentNode.getNodeName() );
	   }
	   
	   // if attribute reference is through "url(#<<identifier>>)"
	   if ( -1 != attrValue.indexOf( ID_REFERENCE_IDENTIFIER ) )
	   {
		   int startIndex = attrValue.indexOf( ID_REFERENCE_IDENTIFIER );
		   String tempStr = attrValue.substring( startIndex + ID_REFERENCE_IDENTIFIER.length() );
		   tempStr = tempStr.substring( 0, tempStr.indexOf(")") );
		   //if ( false == myIds.contains( tempStr ) )
		   if ( false == myReferencedIdData.containsKey( tempStr ) )
		   {
			   //myIds.add( tempStr );
			   //parentNode = attribute.getParentNode();
			   myReferencedIdData.put( tempStr, parentNode.getNodeName() );
		   }
		   
	   }
	   // if attribute reference is through "#xpointer("<<identifier>>")"
	   else if ( -1 != attrValue.indexOf( ID_XPOINTER_REFERENCE_IDENTIFIER ))
	   {
		   int startIndex = attrValue.indexOf( ID_XPOINTER_REFERENCE_IDENTIFIER );
		   String tempStr = attrValue.substring( startIndex + ID_XPOINTER_REFERENCE_IDENTIFIER.length() );
		   tempStr = tempStr.substring( 0, tempStr.indexOf("\")") );
		   //if ( false == myIds.contains( tempStr ) )
		   if ( false == myReferencedIdData.containsKey( tempStr ) )
		   {
			   //myIds.add( tempStr );
			   //parentNode = attribute.getParentNode();
			   myReferencedIdData.put( tempStr, parentNode.getNodeName() );
		   }
	   }
	   else if ( -1 != attrValue.indexOf( HASH_REFERENCE_IDENTIFIER ) )
	   {
		  int startIndex = attrValue.indexOf( HASH_REFERENCE_IDENTIFIER );
		  String tempStr = attrValue.substring( startIndex + 1  );
		  //tempStr = tempStr.substring( 0, tempStr.indexOf("\")") );
		  //if ( false == myIds.contains( tempStr ) )
		  if ( false == myReferencedIdData.containsKey( tempStr ) )
		  {
			 //myIds.add( tempStr );
			 //parentNode = attribute.getParentNode();
			 myReferencedIdData.put( tempStr, parentNode.getNodeName() );
		 }
	   }
   }
 
 
   /**
    * Main method for testing.
    * 
    * @param args 	Command line arguments.
    */
   public static void main( String[] args )
   {
	    try
	    {
			Document svgtDoc = XMLFileParser.parseFile("C:\\SVG\\samples\\Mine.svg", false);
			NonReferredIdRemover.processNonReferencedIDs( svgtDoc, null );
	    }
	    catch ( Exception ex )
	    {
	    	ex.printStackTrace();
	    }
   }

}
