/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.converter;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Stack;
import java.util.StringTokenizer;

import javax.swing.JOptionPane;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.SAXException;

import com.nokia.svg2svgt.SVG2SVGTConstants;
import com.nokia.svg2svgt.ServiceRegistry;
import com.nokia.svg2svgt.configuration.Configuration;
import com.nokia.svg2svgt.configuration.ConfigurationManager;
import com.nokia.svg2svgt.gui.SVGTConverterController;
import com.nokia.svg2svgt.gui.SVGTFileChooser;
import com.nokia.svg2svgt.log.LogListener;
import com.nokia.svg2svgt.log.LogMessages;
import com.nokia.svg2svgt.log.Logger;
import com.nokia.svg2svgt.util.XMLFileParser;

/**
 * <code>SVG2SVGTConveter</code> is responsible for carrying out the SVG to SVGT
 * conversion process. It uses <code>AttributeRemover</code>,
 * <code>ElementRemover</code>, <code>NameSpaceRemover</code>,
 * <code>NonReferredIdRemover</code> and <code>GradientDefinitionMerger</code>
 * classes for carrying out the conversion operations. It also removes
 * unnecessary white spaces and lines changes from the SVGT files and also does
 * a conversion for opacity attribute to a pair of fill-opacity and
 * stroke-opacity attributes.
 * 
 * Once the conversion process is complete, <code>SVGT2SVGTConverter</code>
 * verifies the output SVGT file size and issues a complexity warning if the
 * file size exceeds the value specified by CriticalFileSize configuration
 * parameter.
 * 
 */
public class SVG2SVGTConverter implements ConversionConstants, Logger {

	/**
	 * Log listener registered with the converter.
	 */
	private ArrayList myLogListenersList = new ArrayList();

	/**
	 * Hashtable containing the keys in the Conversions file and the instances
	 * of the corresponding classes handling those conversions.
	 */
	private Hashtable myConversionsInfo = new Hashtable();

	/**
	 * Hashtable specifying the configuration specified by the user to be used
	 * in the conversion process.
	 */
	private Hashtable myConfigParams = new Hashtable();

	/**
	 * Outout document for SVGT.
	 */
	private Document mySVGTDoc = null;

	/**
	 * Black list analyzer for removing black listed tags.
	 */
	private NameSpaceAnalyzer myNameSpaceAnalyzer = null;

	private static Hashtable myNameSpaceData = new Hashtable();

	private Stack myParentNSStack = new Stack();

	// private boolean isDefaultNS = false;
	public static String exceptionMessage[] = new String[8];

	/**
	 * Default constructor. Reads the conversions file and creates the instance
	 * of the corresponding implementation classes using the
	 * <code>ConversionsFactory</code>.
	 * 
	 * @param configurationParams
	 *            Hashtable containing configuration for the converter tool.
	 */
	public SVG2SVGTConverter(Hashtable configurationParams) {
		myConfigParams = configurationParams;

		// create the sub-classes for the relevant purpose
		if (null != configurationParams) {
			String conversionsFileName = (String) configurationParams
					.get(SVG2SVGTConstants.CONVERSIONSFILE_OPTION);

			// read the conversions file
			ConfigurationManager confMgr = new ConfigurationManager(
					conversionsFileName);
			Configuration conf = confMgr.getConfiguration();
			Hashtable confData = conf.toHash();
			// System.out.println("The hash table for command is" + confData);
			// get the elements/attributes for which conversions are defined
			Enumeration enum1 = confData.keys();
			String key = null;
			String keyValue = null;

			// create specific conversions classes using ConversionsFactory
			while (true == enum1.hasMoreElements()) {
				key = (String) enum1.nextElement();
				keyValue = (String) confData.get(key);
				if (null != keyValue) {
					Conversions instance = ConversionsFactory
							.getConversionsInstance(keyValue);
					if (null != instance) {
						myConversionsInfo.put(key, instance);
					}
				} // end-if
			} // end-while

			// create the black list, grey list analysers
			String fileName = (String) configurationParams
					.get(SVG2SVGTConstants.PASSTHROUGHFILE_OPTION);
			try {
				myNameSpaceAnalyzer = new NameSpaceAnalyzer(fileName, this);
				ServiceRegistry.registerService(myNameSpaceAnalyzer);
			} catch (FactoryConfigurationError fcerror) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { fileName, fcerror.getMessage() });
				myNameSpaceAnalyzer = null;
			} catch (ParserConfigurationException pcex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { fileName, pcex.getMessage() });
				myNameSpaceAnalyzer = null;
			} catch (IOException iex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { fileName, iex.getMessage() });
				myNameSpaceAnalyzer = null;
			} catch (SAXException saxex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { fileName, saxex.getMessage() });
				myNameSpaceAnalyzer = null;
			} catch (IllegalArgumentException iaex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { fileName, iaex.getMessage() });
				myNameSpaceAnalyzer = null;
			} catch (Exception ex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { fileName, ex.getMessage() });
				myNameSpaceAnalyzer = null;
			}

		}
	}

	/**
	 * Regsiters the log listener. Checks if this listener is already
	 * registered. If already registered, it deregisters it and registers this
	 * new instance.
	 * 
	 * @param listener
	 *            <code>LogListener</code> instance to be registered.
	 */
	public void registerLogListener(LogListener listener) {
		if (true == myLogListenersList.contains(listener)) {
			myLogListenersList.remove(listener);
		}
		myLogListenersList.add(listener);
	}

	/**
	 * Deregisters the log listener.
	 * 
	 * @param listener
	 *            <code>LogListener</code> instance to be deregistered.
	 */
	public void removeLogListener(LogListener listener) {
		myLogListenersList.remove(listener);
	}

	/**
	 * Converts an input SVG file to SVGT file.
	 */
	public void convert() {

		// parse the document
		// System.out.println("Command line params = " + myConfigParams);
		Document parsedDoc = null;
		ArrayList inputFileList = (ArrayList) myConfigParams
				.get(SVG2SVGTConstants.SVGFILE_OPTION);
		if (inputFileList == null)
			return;
		// parse each file at a time
		for (int i = 0; i < inputFileList.size(); i++) {
			// System.out.println( inputFileList.get( i ));
			try {
				parsedDoc = XMLFileParser.parseFile((String) inputFileList
						.get(i), true);
				// PrintDOMTree.printDOMTree( parsedDoc );
				// parsedDoc = XMLFileParser.parseFile( ( String )
				// inputFileList.get( i ), false );
				mySVGTDoc = XMLFileParser.getEmptyDocument();
			} catch (FactoryConfigurationError fcerror) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { (String) inputFileList.get(i),
								fcerror.getMessage() });
				// System.out.println( fcerror.getMessage() );
				exceptionMessage[0] = fcerror.getMessage();
				fcerror.printStackTrace();
				continue;
			} catch (ParserConfigurationException pcex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { (String) inputFileList.get(i),
								pcex.getMessage() });
				// System.out.println( pcex.getMessage() );
				exceptionMessage[1] = pcex.getMessage();
				// pcex.printStackTrace();
				continue;
			} catch (IOException iex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { (String) inputFileList.get(i),
								iex.getMessage() });
				// System.out.println( iex.getMessage() );
				// iex.printStackTrace();
				exceptionMessage[2] = iex.getMessage();
				continue;
			} catch (SAXException saxex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { (String) inputFileList.get(i),
								saxex.getMessage() });
				// System.out.println( saxex.getMessage() );
				saxex.printStackTrace();
				exceptionMessage[3] = saxex.getMessage();
				continue;
			} catch (IllegalArgumentException iaex) {
				this.logEvent(SVG2SVGTConstants.FILE_PARSING_ERROR,
						new String[] { (String) inputFileList.get(i),
								iaex.getMessage() });
				// System.out.println( iaex.getMessage() );
				// iaex.printStackTrace();
				exceptionMessage[4] = iaex.getMessage();
				continue;
			}

			// successfully parsed the document, now do the conversion
			try {
				this.logEvent(SVG2SVGTConstants.CURRENT_CONVERSION_FILENAME,
						new String[] { (String) inputFileList.get(i) });
				// now the parsed document is available, do the rest
				convertSVGToSVGT(parsedDoc, mySVGTDoc);

				this.logEvent(SVG2SVGTConstants.REMOVE_SVG_NAMESPACE, null);
				// remove SVG name space references
				mySVGTDoc = NameSpaceRemover.removeSVGNameSpaces(mySVGTDoc);

				this.logEvent(SVG2SVGTConstants.REMOVING_NONREFERRED_IDS, null);
				// remove non-referenced identifiers
				NonReferredIdRemover.processNonReferencedIDs(mySVGTDoc, this);

				this.logEvent(SVG2SVGTConstants.MERGING_SIMILAR_GARDIENT_DEFS,
						null);
				// merge similar gradient paint definitions
				GradientDefinitionMerger.mergeGradientDefinitions(mySVGTDoc,
						this);

				this.logEvent(SVG2SVGTConstants.REMOVING_EMPTY_NODES, null);
				// remove any empty nodes if present
				// check for the empty SVGT document
				if (null != mySVGTDoc.getDocumentElement()) {
					removeEmptyNodes(mySVGTDoc.getDocumentElement());
				}

				// PrintDOMTree.printDOMTree( mySVGTDoc );

				// save the file to the output directory
				this.saveSVGTFile(mySVGTDoc, (String) inputFileList.get(i));

			} catch (DOMException domex) {
				this.logEvent(SVG2SVGTConstants.CONVERSION_FAILURE_ERROR,
						new String[] { (String) inputFileList.get(i),
								domex.getMessage() });
				// System.out.println( ex.getMessage() );
				domex.printStackTrace();
				parsedDoc = null;
				mySVGTDoc = null;
				exceptionMessage[4] = domex.getMessage();
				continue;
			} catch (TransformerFactoryConfigurationError tfce) {
				// System.out.println( tfce.getMessage() );
				exceptionMessage[5] = tfce.getMessage();
				continue;
			} catch (TransformerConfigurationException tcex) {
				// tcex.printStackTrace();
				exceptionMessage[6] = tcex.getMessage();
				continue;
			} catch (TransformerException tex) {
				// tex.printStackTrace();
				exceptionMessage[7] = tex.getMessage();
				continue;
			}

			parsedDoc = null;
			mySVGTDoc = null;
		} // end for loop

	}

	/**
	 * @param parsedDoc
	 */
	private void readAllNameSpacesUsed(Document parsedDoc) {
		// get the namespace uri's from SVGNamespace
		// get all the variable names
		Element rootElement = parsedDoc.getDocumentElement();

		ArrayList nsNames = myNameSpaceAnalyzer.getNSInPassThrough();
		Iterator iter = nsNames.iterator();
		SVGNameSpace svgNS = null;
		while (iter.hasNext()) {
			svgNS = (SVGNameSpace) iter.next();
			if (null != svgNS.getMyUri()) {
				myNameSpaceData.put(svgNS.getMyUri(), new ArrayList());
			}
		}
		traverseNode(rootElement);
	}

	private void traverseNode(Node svgNode) throws DOMException {
		NamedNodeMap attributeNodes = null;
		// ArrayList variableList = new ArrayList();
		if (null == svgNode) {
			return;
		}

		Node node = null;
		String nodeName = null;
		String nodeValue = null;
		if (true == svgNode.hasAttributes()) {
			// get the attributes for this element
			attributeNodes = svgNode.getAttributes();

			SVGNameSpace svgNS = null;
			// traverse through the attribute list
			for (int i = 0; i < attributeNodes.getLength(); i++) {
				node = attributeNodes.item(i);
				nodeName = node.getNodeName();
				if ((true == nodeName.startsWith(XML_NAMESPACE_ID))
						&& (-1 != nodeName.indexOf(":"))) {
					String nsVariable = nodeName.substring(XML_NAMESPACE_ID
							.length() + 1, nodeName.length());
					// boolean isWarningReqd = myNameSpaceAnalyzer.
					nodeValue = node.getNodeValue();
					svgNS = getSVGNameSpace(nodeValue);

					if (null == svgNS) {
						continue;
					}
					((ArrayList) myNameSpaceData.get(svgNS.getMyUri()))
							.add(nsVariable);
				}
			} // end for loop
		}
		NodeList childNodes = null;
		if (true == svgNode.hasChildNodes()) {
			childNodes = svgNode.getChildNodes();

			for (int i = 0; i < childNodes.getLength(); i++) {
				node = childNodes.item(i);
				if (Node.ELEMENT_NODE != node.getNodeType()) {
					continue;
				}
				traverseNode(node);
			}
		}
	}

	/**
	 * @param nodeValue
	 * @return
	 */
	private SVGNameSpace getSVGNameSpace(String nodeValue) {
		SVGNameSpace svgNS = null;
		ArrayList nsList = myNameSpaceAnalyzer.getNSInPassThrough();
		String url = null;
		if (null != nsList) {
			for (int i = 0; i < nsList.size(); i++) {
				svgNS = (SVGNameSpace) nsList.get(i);
				url = svgNS.getMyUri();
				if ((null == url) || (0 == url.length())) {
					continue;
				}
				if (true == svgNS.isURI()) {
					if (true == url.equals(nodeValue)) {
						return svgNS;
					}
					continue;
				} else {
					if (-1 != nodeValue.indexOf(url)) {
						return svgNS;
					}
					continue;
				}

			} // end for loop
		}
		return null;
	}

	/**
	 * Removes empty nodes from the SVGT node after the conversion process.
	 * 
	 * @param node
	 *            Node to be analyzed
	 */
	private void removeEmptyNodes(Node node) {

		// remove all nodes except SVG root element
		if (false == node.getNodeName().equalsIgnoreCase(TAG_SVG_ROOT)) {
			if ((false == node.hasAttributes())
					&& (false == isHavingChildNodes(node))) {
				Node parentNode = node.getParentNode();
				parentNode.removeChild(node);
			}
			if (true == node.hasChildNodes()) {
				NodeList childNodes = node.getChildNodes();
				for (int i = 0; i < childNodes.getLength(); i++) {
					if (Node.ELEMENT_NODE != childNodes.item(i).getNodeType()) {
						continue;
					}
					// recursive method call
					removeEmptyNodes(childNodes.item(i));
				}
			}
		} else {
			NodeList childNodes = node.getChildNodes();
			for (int i = 0; i < childNodes.getLength(); i++) {
				// recursive method call
				if (Node.ELEMENT_NODE != childNodes.item(i).getNodeType()) {
					continue;
				}
				// System.out.println( childNodes.item( i ).getNodeName() );
				removeEmptyNodes(childNodes.item(i));
			}
		}
	}

	/**
	 * Checks if a node is empty of has some child elements.
	 * 
	 * @param node
	 *            Node to be checked.
	 * @return True if node is empty, else false.
	 */
	private boolean isHavingChildNodes(Node node) {
		NodeList nodeList = node.getChildNodes();
		ArrayList tempList = new ArrayList();
		if (null == nodeList) {
			return false;
		}

		for (int i = 0; i < nodeList.getLength(); i++) {
			Node tempNode = nodeList.item(i);
			if (Node.ELEMENT_NODE != tempNode.getNodeType()) {
				String value = tempNode.getNodeValue();
				if ((null != value) && (0 < value.trim().length())) {
					tempList.add(tempNode);
				}
			} else {
				tempList.add(tempNode);
			}
		}
		// if list is empty, then there were no child nodes
		if (0 == tempList.size()) {
			return false;
		}

		// if it comes here, then the list has child elements
		return true;
	}

	/**
	 * Parses the input SVG document and performs the conversion to SVGT.
	 * 
	 * @param svgDoc
	 *            Input SVG document reference.
	 * @param svgTDoc
	 *            Output SVGT document reference.
	 * @throws Exception
	 *             If any exception occurs while conversion.
	 */
	private void convertSVGToSVGT(Document svgDoc, Document svgtDoc)
			throws DOMException {
		boolean isDefaultNS = false;

		// conversion process
		// get the root element
		readAllNameSpacesUsed(svgDoc);

		NamedNodeMap attributeNode = null;
		Element rootElement = svgDoc.getDocumentElement();
		// String rootElementName = rootElement.getNodeName();

		isDefaultNS = setParentNameSpace(rootElement);

		// if allowed then check for the attributes
		// check if the root element has any attributes
		String nsName = null; // namespace variable name
		String nsURL = null;
		String nodeName = null; // complete node name

		nodeName = rootElement.getNodeName();
		int index = nodeName.indexOf(":");
		if (-1 != index) {
			nsName = nodeName.substring(0, index);
			nodeName = nodeName.substring(index + 1, nodeName.length());
		}

		if (false == checkNodeInPassThroughList(nsName, nsURL, nodeName,
				Node.ELEMENT_NODE)) {
			// root element is not allowed, means nothing is allowed at all
			return;
		} else {
			if (true == myNameSpaceAnalyzer.isWarningTag(nsName, nsURL,
					nodeName, Node.ELEMENT_NODE)) {
				// raise a warning
				this.logWarning(SVG2SVGTConstants.GREY_TAG_FOUND,
						new String[] { nodeName });
			}
		}

		Element svgtRootElem = svgtDoc.createElement(rootElement.getNodeName());
		svgtDoc.appendChild(svgtRootElem);
		String attrName = null;
		// assuming that root element is always SVG
		if (nodeName.equals(TAG_SVG_ROOT)) {
			// add the baseProfile attribute with value as "tiny"
			if (null != nsName) {
				attrName = nsName + ":" + BASE_PROFILE_ATTRIBUTE;
			} else {
				attrName = BASE_PROFILE_ATTRIBUTE;
			}
			Attr baseProfileAttribute = svgtDoc.createAttribute(attrName);
			baseProfileAttribute.setValue(BASE_PROFILE_TINY);
			svgtRootElem.setAttributeNode(baseProfileAttribute);
		}

		// root element is allowed, check its attributes
		if (true == rootElement.hasAttributes()) {
			// get the attributes for this element
			attributeNode = rootElement.getAttributes();
			Node attrNode = null;
			for (int i = 0; i < attributeNode.getLength(); i++) {
				attrNode = attributeNode.item(i);
				nodeName = attrNode.getNodeName();

				if (true == nodeName.startsWith(XML_NAMESPACE_ID)) {
					// put this node in the output SVGT document
					if (nodeName.endsWith(XML_TAG_XLINK)
							|| nodeName.startsWith(XML_TAG_XLINK)) {
						Node svgtAttr = svgtDoc.importNode(attrNode, false);
						svgtRootElem.setAttributeNode((Attr) svgtAttr);
						continue;
					}
					index = nodeName.indexOf(":");
					String tempNS = null;
					if (-1 != index) {
						tempNS = nodeName.substring(index + 1, nodeName
								.length());
					}
					if ((null != tempNS) && (!tempNS.equals(nsName))) {
						// check if this variable is present in hashtable and
						// passthrough file or not
						String tempNodeName = nodeName.substring(index + 1,
								nodeName.length());
						if (true != checkNodeInPassThroughList(tempNS, attrNode
								.getNodeValue(), tempNodeName,
								Node.ATTRIBUTE_NODE)) {
							continue;
						}
					}
					Node svgtAttr = svgtDoc.importNode(attrNode, false);
					svgtRootElem.setAttributeNode((Attr) svgtAttr);
					continue;
				}

				if ((nodeName.equalsIgnoreCase(TAG_WIDTH))
						|| (nodeName.equalsIgnoreCase(TAG_HEIGHT))) {
					String value = null;
					ResourceBundle bundle = ResourceBundle
							.getBundle("SVG2SVGTProperties");
					try {
						value = bundle.getString("OVERWRITE_DIMENSION");
					} catch (MissingResourceException missingresourceexception) {
					}
					if (myConfigParams.containsKey("-d")) {
						String tempString = (String) (String) myConfigParams
								.get("-d");
						if (tempString.equalsIgnoreCase("normal"))
							value = null;
						else
							value = tempString;
					}
					if (value != null && value.length() > 0) {
						Attr svgtAttr = svgtDoc.createAttribute(nodeName);
						svgtAttr.setValue(value.trim());
						svgtRootElem.setAttributeNode(svgtAttr);
					} else {
						Attr svgtAttr = svgtDoc.createAttribute(nodeName);
						svgtAttr.setValue(((Attr) attrNode).getValue());
						svgtRootElem.setAttributeNode(svgtAttr);
					}
					continue;
				}

				nsName = null;
				nsURL = null;
				// if not namespace declaration
				index = nodeName.indexOf(":");
				if (-1 != index) {
					nsName = nodeName.substring(0, index);
					nodeName = nodeName.substring(index + 1, nodeName.length());
				}

				if (null != nsName) {
					// check if this namespace is allowed or not
					if ((false == checkNSNameInCurrentDoc(nsName))
							&& (false == myNameSpaceAnalyzer
									.isNameSpacePresent(nsName))) {
						// if not allowed go to the next node
						continue;
					}
				} else {
					// nsName = getParentNameSpace();
					Node temp = (Node) getParentNameSpace();
					if (null != temp) {
						nsURL = temp.getNodeValue();
					}
				}

				// perform the conversions first
				if (true == isConversionsRequired(nodeName)) {
					this.logEvent(
							SVG2SVGTConstants.PERFORMING_CONVERSIONS_MESSAGE,
							new String[] { nodeName });
					processConversions(attrNode, svgDoc, svgtRootElem, svgtDoc,
							nsURL);
					continue;
				}

				// check in the pass through if this attribute is allowed or not

				if (true == checkNodeInPassThroughList(nsName, nsURL, nodeName,
						Node.ATTRIBUTE_NODE)) {
					// this.logEvent( SVG2SVGTConstants.BLACK_TAG_REMOVED,
					// new String[]{ attrNode.getNodeName() } );

					// node is allowed, check if a warning needs to be generated
					// or not
					if (true == myNameSpaceAnalyzer.isWarningTag(nsName, nsURL,
							nodeName, Node.ATTRIBUTE_NODE)) {
						// raise a warning
						this.logWarning(SVG2SVGTConstants.GREY_TAG_FOUND,
								new String[] { nodeName });
					}
				} else {
					continue;
				}

				Node svgtAttr = svgtDoc.importNode(attrNode, false);
				svgtRootElem.setAttributeNode((Attr) svgtAttr);
			}
		}

		NodeList childNodes = rootElement.getChildNodes();
		// System.out.println("Number = " + childNodes.getLength());
		Node node = null;
		for (int i = 0; i < childNodes.getLength(); i++) {
			node = childNodes.item(i);
			nodeName = node.getNodeName();
			nsName = null;
			nsURL = null;
			index = nodeName.indexOf(":");
			if (-1 != index) {
				// check if this namespace is allowed or not
				nsName = nodeName.substring(0, index);
				nodeName = nodeName.substring(index + 1, nodeName.length());
				if ((false == checkNSNameInCurrentDoc(nsName))
						&& (false == myNameSpaceAnalyzer
								.isNameSpacePresent(nsName))) {
					// if not allowed go to the next node
					isDefaultNS = false;
					continue;
				}
			}

			// no need to add extra blank spaces and nested SVG nodes
			if ((Node.ELEMENT_NODE != node.getNodeType())
					|| (nodeName.equalsIgnoreCase(TAG_SVG_ROOT))) {
				if (Node.TEXT_NODE == node.getNodeType()) {
					String textValue = node.getNodeValue();
					if ((null != textValue) && (0 < textValue.trim().length())) {
						Text textNode = svgtDoc.createTextNode(textValue);
						svgtRootElem.appendChild(textNode);
					}
				}
				// if ( true == isDefaultNS )
				// {
				// myParentNSStack.pop();
				// }
				// isDefaultNS = false;
				continue;
			}
			isDefaultNS = setParentNameSpace((Element) node);
			if (null == nsName) {
				Node temp = (Node) getParentNameSpace();
				if (null != temp) {
					nsURL = temp.getNodeValue();
				}
			}
			if (true == isConversionsRequired(nodeName)) {
				this.logEvent(SVG2SVGTConstants.PERFORMING_CONVERSIONS_MESSAGE,
						new String[] { nodeName });
				processConversions(node, svgDoc, svgtRootElem, svgtDoc, nsURL);
				if (true == isDefaultNS) {
					myParentNSStack.pop();
				}
				continue;
			}
			if (true == checkNodeInPassThroughList(nsName, nsURL, nodeName,
					Node.ELEMENT_NODE)) {
				// rootElement.removeChild( node );
				// this.logEvent( SVG2SVGTConstants.BLACK_TAG_REMOVED,
				// new String[]{ node.getNodeName() } );
				if (true == myNameSpaceAnalyzer.isWarningTag(nsName, nsURL,
						nodeName, Node.ELEMENT_NODE)) {
					this.logWarning(SVG2SVGTConstants.GREY_TAG_FOUND,
							new String[] { nodeName });
				}
			} else {
				if (true == isDefaultNS) {
					myParentNSStack.pop();
				}
				continue;
			}
			Node parentSVGTNode = null;
			parentSVGTNode = (Node) svgtDoc.createElement(node.getNodeName());
			parentSVGTNode = svgtRootElem.appendChild(parentSVGTNode);
			traverseNode(node, svgDoc, parentSVGTNode, svgtDoc, isDefaultNS);
			if (true == isDefaultNS) {
				myParentNSStack.pop();
			}
		}
		if (true == isDefaultNS) {
			myParentNSStack.pop();
		}
	}

	private boolean checkNSNameInCurrentDoc(String nsName) {
		Enumeration enum1 = myNameSpaceData.keys();
		ArrayList nsList = null;
		while (true == enum1.hasMoreElements()) {
			nsList = (ArrayList) myNameSpaceData.get((String) enum1
					.nextElement());
			if (nsList.contains(nsName)) {
				return true;
			}
		}
		return false;
	}

	public static Hashtable getNSInCurrentDoc() {
		return myNameSpaceData;
	}

	private Object getParentNameSpace() {
		if (myParentNSStack.size() != 0)
			return myParentNSStack.peek();
		return null;
	}

	private boolean setParentNameSpace(Element elemNode) {
		// check for index of ":", if ":" not present check xmlns attribute
		// else <<nsName>>:<<elem>> check xmlns:<<nsName>> attribute
		// push this URL into the stack
		// if for attribute index of ":" is -1, then it belongs to
		// the namespace on the top of the stack
		// if ":" present and not a xmlns declaration, check in
		// namespace hashtable or pass through file
		Attr attribute = null;
		String nsReference = null;
		String elemName = elemNode.getNodeName();

		if (-1 == elemName.indexOf(":")) {
			attribute = elemNode.getAttributeNode(XML_NAMESPACE_ID);
			if (null != attribute) {
				// myParentNSStack.push( attribute.getNodeValue() );
				myParentNSStack.push(attribute);
				return true;
			}
		} else {
			int indexOfColon = elemName.indexOf(":");
			nsReference = elemName.substring(0, indexOfColon);
			attribute = elemNode.getAttributeNode(XML_NAMESPACE_ID + ":"
					+ nsReference);
			if (null != attribute) {
				// myParentNSStack.push( attribute.getNodeValue() );
				myParentNSStack.push(attribute);
				return true;
			}
		}
		return false;
	}

	/**
	 * Creates the DOCTYPE declaration for the SVGT document.
	 * 
	 * @param svgDoc
	 *            SVG document instance.
	 * @param svgtDoc
	 *            SVGT document instance.
	 */
	private Document createDOCTYPEElement(Document svgDoc, Document svgtDoc) {
		DOMImplementation domImpl = svgtDoc.getImplementation();
		DocumentType docType = svgDoc.getDoctype();
		DocumentType svgtDocType = null;
		String docTypeName = null;
		if (null == docType) {
			// add the DOC TYPE header in SVGT doc
			docTypeName = TAG_SVG_ROOT;

		} else {
			docTypeName = docType.getName();
			// public ID and SYSTEM ID shoudl be set to that of SVGT's IDs

		} // end else
		svgtDocType = domImpl.createDocumentType(docTypeName,
				SVGT_DOCUMENT_PUBLIC_ID, SVGT_DOCUMENT_SYSTEM_ID);
		svgtDoc = domImpl.createDocument(SVG_NAMESPACE_ID, TAG_SVG_ROOT,
				svgtDocType);
		return svgtDoc;
		// PrintDOMTree.printDOMTree( svgtDoc );

	}

	/**
	 * Checks if the specified node is contained in the black list or not.
	 * 
	 * @param node
	 *            Node to be searched.
	 * @return True if found, else false.
	 */
	private boolean checkNodeInPassThroughList(String nsName, String nsURL,
			String nodeName, short nodeType) {
		boolean isAllowedTag = false;
		if (null != myNameSpaceAnalyzer) {
			isAllowedTag = myNameSpaceAnalyzer.isNodeAllowed(nsName, nsURL,
					nodeName, nodeType);
		}
		return isAllowedTag;
	}

	/**
	 * Checks the conversions Hashtable for this key.
	 * 
	 * @param attrName
	 *            Tag name to be searched.
	 * @return True if key exists, else false.
	 */
	private boolean isConversionsRequired(String attrName) {
		if (true == myConversionsInfo.containsKey(attrName)) {
			return true;
		}
		return false;
	}

	/**
	 * Traverses a node. Checks if the node has any attributes, then reads them.
	 * If node has any child nodes, it traverses through the child nodes.
	 * 
	 * @param svgNode
	 *            Node to be traversed.
	 * @param svgDoc
	 *            SVG Document reference.
	 * @param svgtNode
	 *            Current Node in the SVGT document.
	 * @param svgtDoc
	 *            SVGT Document reference.
	 */
	private void traverseNode(Node svgNode, Document svgDoc, Node svgtNode,
			Document svgtDoc, boolean isDefaultNS) throws DOMException {
		NamedNodeMap nodeMap = null;
		Node node = null;
		String nsName = null; // namespace variable name
		String nodeName = null; // complete node name
		String nsURL = null;
		int index = -1;
		if (true == svgNode.hasAttributes()) {
			nodeMap = svgNode.getAttributes();
			for (int i = 0; i < nodeMap.getLength(); i++) {
				node = nodeMap.item(i);

				// System.out.println( "Parent Node = " +
				// (tempNode.getParentNode()).getNodeName());
				// System.out.println("Name = " + node.getNodeName() +
				// " Value = " + node.getNodeValue() );
				nodeName = node.getNodeName();
				if (true == nodeName.startsWith(XML_NAMESPACE_ID)) {
					// put this node in the output SVGT document
					index = nodeName.indexOf(":");
					String tempNS = null;
					if (-1 != index) {
						tempNS = nodeName.substring(index + 1, nodeName
								.length());
					}
					if ((null != tempNS) && (!tempNS.equalsIgnoreCase(nsName))) {
						// check if this variable is present in hashtable and
						// passthrough file or not
						String tempNodeName = nodeName.substring(index + 1,
								nodeName.length());
						if (true != checkNodeInPassThroughList(tempNS, node
								.getNodeValue(), tempNodeName,
								Node.ATTRIBUTE_NODE)) {
							continue;
						}
					}
					Node svgtAttr = svgtDoc.importNode(node, false);
					((Element) svgtNode).setAttributeNode((Attr) svgtAttr);
					continue;
				}
				if (true == nodeName.startsWith(XML_TAG_XLINK)) {
					Node svgtAttr = svgtDoc.importNode(node, false);
					((Element) svgtNode).setAttributeNode((Attr) svgtAttr);

					continue;
				}
				nsName = null;
				nsURL = null;

				index = nodeName.indexOf(":");
				if (-1 != index) {

					nsName = nodeName.substring(0, index);
					nodeName = nodeName.substring(index + 1, nodeName.length());
				}

				if (null != nsName) {
					// check if this namespace is allowed or not
					if ((false == checkNSNameInCurrentDoc(nsName))
							&& (false == myNameSpaceAnalyzer
									.isNameSpacePresent(nsName))) {
						// if not allowed go to the next node
						continue;
					}
				} else {
					Node temp = (Node) getParentNameSpace();
					if (null != temp) {
						nsURL = temp.getNodeValue();
					}

				}
				if (true == isConversionsRequired(nodeName)) {
					this.logEvent(
							SVG2SVGTConstants.PERFORMING_CONVERSIONS_MESSAGE,
							new String[] { nodeName });
					processConversions(node, svgDoc, svgtNode, svgtDoc, nsURL);
					continue;
				}

				if (true == checkNodeInPassThroughList(nsName, nsURL, nodeName,
						Node.ATTRIBUTE_NODE)) {
					// Element parent = ( Element ) tempNode.getParentNode( );
					// parent.removeAttributeNode( ( Attr ) tempNode );
					// this.logEvent( SVG2SVGTConstants.BLACK_TAG_REMOVED,
					// new String[]{ node.getNodeName() } );
					if (true == myNameSpaceAnalyzer.isWarningTag(nsName, nsURL,
							nodeName, Node.ATTRIBUTE_NODE)) {
						this.logWarning(SVG2SVGTConstants.GREY_TAG_FOUND,
								new String[] { nodeName });
					}

				} else {
					continue;
				}
				Element parentSVGTNode = (Element) svgtNode;
				Node svgtAttr = svgtDoc.importNode(node, false);
				parentSVGTNode.setAttributeNode((Attr) svgtAttr);
			}
		}

		if (false == svgNode.hasChildNodes()) {
			// if ( true == isDefaultNS )
			// {
			// myParentNSStack.pop();
			// }
			return;
		} else {
			NodeList childNodes = svgNode.getChildNodes();
			for (int i = 0; i < childNodes.getLength(); i++) {
				node = childNodes.item(i);
				nodeName = node.getNodeName();

				index = nodeName.indexOf(":");
				nsName = null;
				nsURL = null;
				if (-1 != index) {
					nsName = nodeName.substring(0, index);
					nodeName = nodeName.substring(index + 1, nodeName.length());
				}

				if (Node.ELEMENT_NODE != node.getNodeType()) {
					if (Node.TEXT_NODE == node.getNodeType()) {
						String textValue = node.getNodeValue();
						if ((null != textValue)
								&& (0 < textValue.trim().length())) {
							Text textNode = svgtDoc.createTextNode(textValue);
							svgtNode.appendChild(textNode);
						}
					}
					// if ( true == isDefaultNS )
					// {
					// myParentNSStack.pop();
					// }
					// isDefaultNS = false;
					continue;
				}
				isDefaultNS = setParentNameSpace((Element) node);

				if (null != nsName) {
					// check if this namespace is allowed or not
					if ((false == checkNSNameInCurrentDoc(nsName))
							&& (false == myNameSpaceAnalyzer
									.isNameSpacePresent(nsName))) {
						// if not allowed go to the next node
						// isDefaultNS = false;
						if (true == isDefaultNS) {
							myParentNSStack.pop();
						}
						continue;
					}
				} else {
					Node temp = (Node) getParentNameSpace();
					if (null != temp) {
						nsURL = temp.getNodeValue();
					}

				}

				// System.out.println( "Element = " + node.getNodeName() );
				if (true == isConversionsRequired(nodeName)) {
					this.logEvent(
							SVG2SVGTConstants.PERFORMING_CONVERSIONS_MESSAGE,
							new String[] { nodeName });
					processConversions(node, svgDoc, svgtNode, svgtDoc, nsURL);
					if (true == isDefaultNS) {
						myParentNSStack.pop();
					}
					continue;
				}

				if (true == checkNodeInPassThroughList(nsName, nsURL, nodeName,
						Node.ELEMENT_NODE)) {
					// (( Element ) node ).removeChild( tempNode );
					// this.logEvent( SVG2SVGTConstants.BLACK_TAG_REMOVED,
					// new String[]{ node.getNodeName() } );
					if (true == myNameSpaceAnalyzer.isWarningTag(nsName, nsURL,
							nodeName, Node.ELEMENT_NODE)) {
						this.logWarning(SVG2SVGTConstants.GREY_TAG_FOUND,
								new String[] { nodeName });

					}
				} else {
					if (true == isDefaultNS) {
						myParentNSStack.pop();
					}
					continue;
				}
				Node parentSVGTNode = null;
				// parentSVGTNode = svgtDoc.importNode( nodeName, false );
				parentSVGTNode = (Node) svgtDoc.createElement(node
						.getNodeName());
				parentSVGTNode = (Element) svgtNode.appendChild(parentSVGTNode);
				// PrintDOMTree.printDOMTree( svgtDoc );
				traverseNode(node, svgDoc, parentSVGTNode, svgtDoc, isDefaultNS);
				if (true == isDefaultNS) {
					myParentNSStack.pop();
				}
			} // end for child nodes
		} // end if-else
	}

	/**
	 * Converts opacity attribute to a pair of fill-opacity and
	 * stroke-opacity attributes.
	 * 
	 * @param svgNode
	 *            SVG Node to be analyzed.
	 * @param svgDoc
	 *            Parsed SVG document reference to which this node belongs.
	 * @param svgtNode
	 *            Parent SVGT node.
	 * @param svgtDoc
	 *            Output svgt document.
	 */
	private void doConversions(Node svgNode, Document svgDoc, Node svgtNode,
			Document svgtDoc, String nsURL) throws DOMException {
		String nodeName = svgNode.getNodeName();
		Conversions conv = (Conversions) myConversionsInfo.get(nodeName);

		// let the actual instance perform the conversion
		conv.doConversion(svgNode, svgDoc, svgtNode, svgtDoc, this, nsURL);
	}

	/**
	 * Checks if the conversion is required for this node or not and performs
	 * the conversion.
	 * 
	 * @param svgNode
	 *            Node to be analyzed.
	 * @param svgDoc
	 *            Parsed document reference to which this node belongs.
	 * @param svgtNode
	 *            Parent SVGT node.
	 * @param svgtDoc
	 *            Output svgt document.
	 * @throws DOMException
	 *             In case of any errors while performing conversions.
	 */
	private void processConversions(Node svgNode, Document svgDoc,
			Node svgtNode, Document svgtDoc, String nsURL) throws DOMException {
		doConversions(svgNode, svgDoc, svgtNode, svgtDoc, nsURL);
	}

	/**
	 * Saves the output SVGT document into a file.
	 * 
	 * @param svgtDoc
	 *            SVGT document to be saved.
	 * @param fileName
	 *            File name with which the file should be saved.
	 */
	private void saveSVGTFile(Document svgtDoc, String fileName)
			throws TransformerFactoryConfigurationError,
			TransformerConfigurationException, TransformerException {
		String outputDirectory = (String) myConfigParams
				.get(SVG2SVGTConstants.OUTDIR_OPTION);

		// check if the file name is already existing in this directory
		File outDir = new File(outputDirectory);
		if (true != outDir.isDirectory()) {
			boolean isDirCreated = outDir.mkdir();
			if (true == isDirCreated) {
				// proceed further
			}
		}

		// remove the absolute path and get the file name only
		fileName = getFileName(fileName);

		File[] filesList = outDir.listFiles();

		ArrayList fileNamesList = null;
		String name = null;
		if (null != filesList) {
			fileNamesList = new ArrayList(filesList.length);
			for (int i = 0; i < filesList.length; i++) {
				fileNamesList.add(filesList[i].getName());
			}
		}

		int count = 1;
		int index = fileName.indexOf(".");
		name = fileName.substring(0, index);
		// System.out.println( "File Name = " + fileName );
		String extension = fileName.substring(index, fileName.length());
		while (false != doesFileNameExist(fileNamesList, fileName)) {
			fileName = name + "_" + count + extension;
			count++;
		}

		this.logEvent(SVG2SVGTConstants.SAVING_SVGT_FILE, new String[] {
				fileName, outputDirectory });
		fileName = outputDirectory + System.getProperty("file.separator")
				+ fileName;
		TransformerFactory factory = TransformerFactory.newInstance();
		Transformer transformer = factory.newTransformer();
		transformer.setOutputProperty(OutputKeys.INDENT, "yes");
		transformer.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC,
				SVGT_DOCUMENT_PUBLIC_ID);
		transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM,
				SVGT_DOCUMENT_SYSTEM_ID);
		// Prepare the DOM document for writing
		Source source = new DOMSource(svgtDoc);

		// Prepare the output file
		File outFile = new File(fileName);
		FileWriter fr = null;
		if (fileName.indexOf(outputDirectory) != -1) {
			String subString = fileName.substring(fileName
					.indexOf(outputDirectory)
					+ outputDirectory.length(), fileName.length());
			subString = subString.substring(0, subString
					.lastIndexOf(File.separator));
			StringTokenizer tok = new StringTokenizer(subString, File.separator);
			String dir = outputDirectory;
			File temp = null;
			while (tok.hasMoreElements()) {
				dir = (new StringBuilder(String.valueOf(dir))).append(
						File.separator).append(tok.nextToken()).toString();
				temp = new File(dir);
				if (!temp.isDirectory())
					temp.mkdir();
			}
		}
		File createDir = outFile.getParentFile();
		if (!createDir.exists()) {
			createDir.mkdir();
		}

		try {
			fr = new FileWriter(fileName);
			outFile.createNewFile();
		} catch (IOException e) {
			e.printStackTrace();
		}

		Result result = new StreamResult(fr);
		// System.out.println( "FileName is " + fileName );
		// transform the document

		transformer.transform(source, result);
		try {
			float size = Float.parseFloat((String) myConfigParams
					.get(SVG2SVGTConstants.CRITICAL_SIZE));
			long fileSize = outFile.length();
			float fileSize1 = fileSize / 1024;
			if (size <= fileSize1) {
				logWarning(SVG2SVGTConstants.CRITICLE_SIZE_EXCEEDING_ERROR,
						null);
			}
		} catch (NumberFormatException nfe) {
			JOptionPane
					.showMessageDialog(null,
							"Only numeric values are allowed. Neglecting the critical size.");
		}
		try {
			// transformer.clearParameters();
			transformer.reset();
			fr.close();
		} catch (IOException ioe) {
		}

	}

	/**
	 * Fetches the file name from the absolute file name.
	 * 
	 * @param absoluteName
	 *            Absolute name of the file.
	 * @return File name.
	 */
	private String getFileName(String absoluteName) {
		String requiredString = " ";

		SVGTFileChooser chooser = (SVGTFileChooser) ServiceRegistry
				.getService("com.nokia.svg2svgt.gui.SVGTFileChooser");
		if (null != chooser) {
			String parentDir = chooser.getMyParentDirectory();
			if (true == chooser.isMaintainFolderStrucSelected()) {
				requiredString = absoluteName.substring(parentDir.length()
						+ File.separator.length(), absoluteName.length());
				return requiredString;
			} else {
				int index = absoluteName.lastIndexOf(System
						.getProperty("file.separator"));
				return absoluteName.substring(index + 1, absoluteName.length());
			}
		} else {
			int index = absoluteName.lastIndexOf(System
					.getProperty("file.separator"));
			return absoluteName.substring(index + 1, absoluteName.length());
		}

	}

	/**
	 * Checks if a file with the specified name is present in the list of files
	 * obtained from the current directory.
	 * 
	 * @param filesList
	 *            List of files in a directory.
	 * @param fileName
	 *            File name to be searched.
	 * @return True if file name exits, else false.
	 */
	private boolean doesFileNameExist(ArrayList filesList, String fileName) {
		boolean isAlreadyExisting = false;
		if (null != filesList) {
			for (int i = 0; i < filesList.size(); i++) {
				if (((String) filesList.get(i)).equalsIgnoreCase(fileName)) {
					isAlreadyExisting = true;
				}
			}
		}
		return isAlreadyExisting;
	}

	/**
	 * Notfies the log listeners about the received log events.
	 * 
	 * @param desc
	 *            Description of the log event.
	 */
	private void notifyLogListeners(String desc) {
		if (null != myLogListenersList) {
			Iterator iter = myLogListenersList.iterator();
			while (true == iter.hasNext()) {
				SVGTConverterController listener = (SVGTConverterController) iter
						.next();
				listener.log(desc);
			}
		}
	}

	/**
	 * Logs an log event.
	 * 
	 * @param errorCode
	 *            Message code for the log message.
	 * @param params
	 *            Parameters for the message.
	 */
	public void logEvent(long errorCode, Object[] params) {
		String desc = null;
		if (null != params) {
			desc = LogMessages.getLogMessage(errorCode, params);
		} else {
			desc = LogMessages.getLogMessage(errorCode);
		}
		notifyLogListeners(desc);
	}

	/**
	 * Logs a warning.
	 * 
	 * @param errorCode
	 *            Message code for the log message.
	 * @param params
	 *            Parameters for the message.
	 */
	public void logWarning(long errorCode, Object[] params) {
		String desc = "Warning: ";
		if (null != params) {
			desc += LogMessages.getLogMessage(errorCode, params);
		} else {
			desc += LogMessages.getLogMessage(errorCode);
		}
		notifyLogListeners(desc);
	}

	/**
	 * Main method for testing purposes.
	 * 
	 * @param args
	 *            Command line arguments.
	 */
	public static void main(String[] args) {
		Hashtable ht = new Hashtable();
		ArrayList al = new ArrayList();
		// al.add( new String("C:\\svgt\\Epoc32\\Wins\\c\\lion.svg"));
		// al.add( new String("C:\\SVG\\samples\\Mine.svg") );
		al.add(new String("C:\\SVG\\qgn_prop_nrtyp_pager.svg"));
		// al.add(new String("C:\\SVG\\samples\\Test1.svg" ));
		// al.add(new String("C:\\SVG\\samples\\Text.svg" ));
		ht.put(SVG2SVGTConstants.SVGFILE_OPTION, al);
		ht.put(SVG2SVGTConstants.CONVERSIONSFILE_OPTION,
				".\\config\\Conversions.properties");
		ht
				.put(SVG2SVGTConstants.PASSTHROUGH_FILE,
						".\\config\\passThrough.xml");

		SVG2SVGTConverter converter = new SVG2SVGTConverter(ht);
		converter.convert();
		// PrintDOMTree.printDOMTree( );
	}

}
