/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.converter;

import java.util.ArrayList;
import java.util.Iterator;

/**
 * <code>SVGElement</code> forms a wrapper class for wrapping an SVG element
 * present in one <code>SVGModule</code>
 * 
 */
public class SVGElement {
	/**
	 * Specified the name of the element
	 */
	private String myName = null;

	/**
	 * Specifies the name of the parent <code>SVGModule</code> to which this
	 * element belongs.
	 */
	private String myParentName = null;

	/**
	 * List of <code>SVGAttributes</code> contained in this element.
	 */
	private ArrayList myAttributes = null;

	/**
	 * Specifies if this element is marked for deletion or for raising a
	 * warning.
	 */
	private boolean isWarned = false;

	/**
	 * Creates an <code>SVGElement</code> with a specified name and the parent
	 * module name to which it belongs.
	 * 
	 * @param elementName
	 *            Name of this <code>SVGElement</code>
	 * @param parentName
	 *            Name of the <code>SVGModule</code> to which this element
	 *            belongs.
	 */
	public SVGElement(String elementName, String parentName) {
		myName = elementName;
		myParentName = parentName;
	}

	/**
	 * Adds an attribute to this <code>SVGElement</code>
	 * 
	 * @param attribute
	 *            <code>SVGAttribute</code> to be added.
	 */
	public void addAttribute(SVGAttribute attribute) {
		if (null == myAttributes) {
			myAttributes = new ArrayList();
		}

		if (null != attribute) {
			myAttributes.add(attribute);
		}
	}

	/**
	 * Adds an attribute to this <code>SVGElement</code>
	 * 
	 * @param attribute
	 *            List of <code>SVGAttribute</code> to be added.
	 */
	public void addAttributes(ArrayList attributeList) {
		if (null == myAttributes) {
			myAttributes = new ArrayList();
		}
		if ((attributeList != null) || (0 < attributeList.size())) {
			myAttributes.addAll(attributeList);
		}

	}

	/**
	 * Removes an attribute from this elements list.
	 * 
	 * @param attributeName
	 *            Attribute name to be removed.
	 */
	public void removeAttribute(String attributeName) {
		int foundElement = containsAttribute(attributeName);
		myAttributes.remove(foundElement);

	}

	/**
	 * Removes an attribute from this elements list, only if it exists with a
	 * specified value.
	 * 
	 * @param attributeName
	 *            Name of the attribute to be removed.
	 * @param value
	 *            Value of the attribute to be removed.
	 */
	public void removeAttribute(String attributeName, String value) {
		int foundElement = containsAttribute(attributeName, value);
		myAttributes.remove(foundElement);
	}

	/**
	 * Checks if the attribute with the specified name is contained in this
	 * elements list.
	 * 
	 * @param attributeName
	 *            Name if the attribute to be searched.
	 * @return Index of the attribute in the list.
	 */
	public int containsAttribute(String attributeName) {
		if (null == myAttributes) {
			return -1;
		}
		Iterator iter = myAttributes.iterator();
		SVGAttribute attribute = null;
		int indexCount = 0;
		while (true == iter.hasNext()) {
			attribute = (SVGAttribute) iter.next();
			if ((attribute.getAttributeName()).equalsIgnoreCase(attributeName)) {
				return indexCount;
			}
			indexCount++;
		}

		return -1;
	}

	/**
	 * Checks if the attribute with the specified name is contained in this
	 * elements list.
	 * 
	 * @param attributeName
	 *            Name of the attribute to be searched.
	 * @param value
	 *            Value of the attribute to be searched.
	 * @return Index of the attribute in the list.
	 */
	public int containsAttribute(String attributeName, Object value) {
		if (null == myAttributes) {
			return -1;
		}
		Iterator iter = myAttributes.iterator();
		SVGAttribute attribute = null;
		int indexCount = 0;
		while (true == iter.hasNext()) {
			attribute = (SVGAttribute) iter.next();
			if ((attribute.getAttributeName()).equalsIgnoreCase(attributeName)
					&& (attribute.getAttributeValue()).equals(value)) {
				return indexCount;
			}
			indexCount++;
		}
		return -1;
	}

	/**
	 * Returns the attribute at the specified index in the attribute list.
	 * 
	 * @param index
	 *            Index of the attribute in the list.
	 * @return <code>SVGAttribute</code> instance or null.
	 */
	public SVGAttribute getAttributeAt(int index) {
		if (((null != myAttributes) || (0 < myAttributes.size()))
				&& (index < myAttributes.size())) {
			return (SVGAttribute) myAttributes.get(index);
		}
		return null;
	}

	/**
	 * Checks if the attribute with the specified name is present in this list
	 * or not.
	 * 
	 * @param attrName
	 *            Name of the attribute to be searched.
	 * @return <code>SVGAttribute</code> instance, or null.
	 */
	public SVGAttribute getAttributeByTagName(String attrName) {
		int attrIndex = containsAttribute(attrName);
		if (-1 != attrIndex) {
			return (SVGAttribute) myAttributes.get(attrIndex);
		}
		return null;
	}

	/**
	 * Returns all the attributes associated with this element.
	 * 
	 * @return List of all attributes for this element.
	 */
	public ArrayList getElementAttributes() {
		return myAttributes;
	}

	/**
	 * Returns the current elements name.
	 * 
	 * @return Name of the element.
	 */
	public String getElementName() {
		return myName;
	}

	/**
	 * Returns name of the parent <code>SVGModule</code> to which this element
	 * belongs.
	 * 
	 * @return Name of the SVGModule.
	 */
	public String getParentModuleName() {
		return myParentName;
	}

	/**
	 * Checks if this element is marked or not ( for deletion or for raising
	 * warnings.)
	 * 
	 * @return True if marked, else false.
	 */
	public boolean isWarned() {
		return isWarned;
	}

	/**
	 * Sets the mark for this element.
	 * 
	 * @param mark
	 *            Mark for this element.
	 */
	public void setWarn(boolean mark) {
		isWarned = mark;
	}

	/**
	 * Overloaded to String method. Returns the XML format for this element.
	 * 
	 * @return XML format for this element.
	 */
	public String toString() {
		return myName;
	}

	/**
	 * Main method for testing.
	 * 
	 * @param args
	 *            Command line arguments.
	 */
	public static void main(String[] args) {
	}
}
