/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.converter;

import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import com.nokia.svg2svgt.util.XMLFileParser;


/**
 * <code>NameSpaceRemover</code> removes all the name space descriptions for 
 * SVG from the input SVG file.
 * 
 */
public class NameSpaceRemover implements ConversionConstants
{

   /**
    * List containing all the SVG namespace prefixes used in the SVG document.
    */
   private static ArrayList mySVGNameSpaces = new ArrayList();


// PUBLIC METHODS

   /**
	* Removes the SVG name spaces from the input SVG document.
	* @param svgDoc		Input SVG document.
	* @return			Document from which the SVG name space prefixes have 
	*                   been removed.
	*/
   public static Document removeSVGNameSpaces( Document svgDoc )
   {
	   // check for empty SVGT document
	   if ( null == svgDoc.getDocumentElement() )
	   {
	   	   return svgDoc;
	   }
	   readSVGNameSpaceDefs( svgDoc );
	   Document svgDocNoNS = null;
   	   
	   try
	   {
		   // create an empty document into which the new nodes are written
		   svgDocNoNS = XMLFileParser.getEmptyDocument();
	   }
	   catch( FactoryConfigurationError fcerror )
	   {
		   //this.logEvent( SVG2SVGTConstants.FILE_PARSING_ERROR, 
			//		        new String[]{ ( String ) inputFileList.get( i ), 
			// 				              fcerror.getMessage() });
		   fcerror.printStackTrace();
	   }
	   catch( ParserConfigurationException pcex )
	   {
		   //this.logEvent( SVG2SVGTConstants.FILE_PARSING_ERROR, 
			//		        new String[]{ ( String ) inputFileList.get( i ), 
			//						      pcex.getMessage() });
		   pcex.printStackTrace();
	   }

	   // get the root element
	   Element rootElem = svgDoc.getDocumentElement();
	   String nodeName = getNSFreeName( rootElem.getNodeName() );
	   Element newElem = svgDocNoNS.createElement( nodeName );
	   svgDocNoNS.appendChild( newElem );
       
	   removeNSInDOMTree( rootElem, svgDoc, newElem, svgDocNoNS );
	   //PrintDOMTree.printDOMTree( svgDocNoNS );
	   return svgDocNoNS;
   }

// PRIVATE METHODS


   /**
    * Reads all the name space prefixes for SVG from the SVGT file.
    * @param svgDoc 	SVG Document reference.
    */
   private static void readSVGNameSpaceDefs( Document svgDoc ) 
   {
   	   Element rootElem = svgDoc.getDocumentElement();
   	   readSVGNameSpacesUsed( rootElem );

   }


   /**
    * Reads all the name space prefixes for SVG from the SVGT file.
    * @param node	Node in the SVG DOM tree.
    */
   private static void readSVGNameSpacesUsed( Node node )
   {
	   NamedNodeMap attrList = null;
	   String attrName = null;
	   String attrValue = null; 
	   if ( true == node.hasAttributes() )
	   {
		   // get the attribute nodes for this node
		   attrList = node.getAttributes();
		   for ( int i = 0; i < attrList.getLength(); i++ )
		   {
			   Attr attr = ( Attr ) attrList.item( i );
			   attrName = attr.getNodeName();
			   attrValue = attr.getNodeValue();
			   
			   // is any SVG namespace attribute defined for this element
			   if ( ( -1 != attrName.indexOf( XML_NAMESPACE_ID ) ) &&
				    ( true == attrValue.equalsIgnoreCase( SVG_NAMESPACE_ID ) ) )
			   {
				    int index = attrName.indexOf( ":" );
				    String svgNameSpace = null;
				    if ( -1 != index )
				    {
					    svgNameSpace = attrName.substring( index+1, attrName.length() );
					    mySVGNameSpaces.add( svgNameSpace );
				    }
			   }
		   }// end for
	   }
	   
	   if ( true == node.hasChildNodes() )
	   {
	   	   // get the child elements in this node
	   	   NodeList childNodeList = node.getChildNodes();
	   	   Node childNode = null;
	   	   for ( int i = 0; i < childNodeList.getLength(); i++ )
	   	   {
	   	   	   childNode = childNodeList.item( i );
	   	   	   if ( Node.ELEMENT_NODE != childNode.getNodeType() )
	   	   	   {
	   	   	   	   continue;
	   	   	   }
	   	   	   readSVGNameSpacesUsed( childNode );
	   	   }
	   	   
	   }// end if
   }
   
   
   
   /**
    * Returns the list of SVG name space prefixes used in the input SVG document.
    * 
    * @return	List of the SVG namespace prefixes.
    */
   private static ArrayList getSVGNameSpaceDefs()
   {
   	   return mySVGNameSpaces;
   }



   /**
    * Removes the SVG name space prefixes from the elements in the input node and save the
    * new nodes in the new document reference.
    * @param node		SVG document node with name space prefix.
    * @param svgDoc		SVG document instance.
    * @param noNSNode	SVG document node without name space prefix.
    * @param svgDocNoNS	SVG document instance containing no namespace prefixes.
    */
   private static void removeNSInDOMTree( Node node, Document svgDoc, Node noNSNode, Document svgDocNoNS )
   {
	   NamedNodeMap attrList = null;
	   // get the attributes for this node
	   if ( true == node.hasAttributes() )
	   {
	   	   attrList = node.getAttributes();
	   	   String attrName = null;
	   	   String attrValue = null;
	   	   Node attrNode = null;
	   	   Attr newAttr = null;
	   	   for ( int i = 0; i < attrList.getLength(); i++ )
	   	   {
	   	   	   attrNode = attrList.item( i );
	   	   	   attrName = getNSFreeName( attrNode.getNodeName() );
	   	   	   attrValue = attrNode.getNodeValue();
	   	   	   newAttr = svgDocNoNS.createAttribute( attrName );
	   	   	   newAttr.setNodeValue( attrValue );
	   	   	   ( ( Element )noNSNode).setAttributeNode( newAttr );
	   	   }
	   }
	   
	   // get any child elements if present
	   NodeList childNodeList = null;
	   Node childNode = null;
	   if ( true == node.hasChildNodes() )
	   {
	   	   childNodeList = node.getChildNodes();
	   	   for( int i = 0; i < childNodeList.getLength(); i++ )
	   	   {
	   	   	   childNode = childNodeList.item( i );
	   	   	   // retain any text data as it is
	   	   	   if ( Node.ELEMENT_NODE != childNode.getNodeType() )
	   	   	   {
					 if ( Node.TEXT_NODE == childNode.getNodeType() )
					 {
						 String textValue = childNode.getNodeValue();
						 if ( ( null != textValue ) && ( 0 < textValue.trim().length() ) )
						 {
							 Text textNode = svgDocNoNS.createTextNode( textValue );
							 noNSNode.appendChild( textNode );
						 }
					 }
					 continue;
				}
			    String nodeName = getNSFreeName( childNode.getNodeName() );
			    Element newElem = svgDocNoNS.createElement( nodeName );
			    noNSNode.appendChild( newElem );
	   	   	    // recursive method call
	   	   	    removeNSInDOMTree( childNode, svgDoc, newElem, svgDocNoNS );
	   	    }
	   	}
   }
 
 

   /**
    * Returns the name free from the name space prefixes in the input SVG document.
    * @param string	Input name containing SVG name space prefixes.
    * @return		Output name without namespace prefixes.
    */
   private static String getNSFreeName( String nodeName )
   {
       Iterator iter = mySVGNameSpaces.iterator();
       while( true == iter.hasNext() )
       {
       	   String nsName = ( String ) iter.next();
       	   //System.out.println( nsName );
       	   if ( true == nodeName.startsWith( nsName + ":") )
       	   {
       	   	   return nodeName.substring( nsName.length() + 1 , nodeName.length() );
       	   }
       }
       return nodeName;
   }


   /**
    * Main method for testing purposes.
    * 
    * @param args	Command line arguments.
    */
   public static void main( String[] args )
   {
	  try
	  {
		  Document svgtDoc = XMLFileParser.parseFile( "C:\\SVG\\samples\\Mine.svg", false);
		  //System.out.println("Merge similar definitions");
		  readSVGNameSpaceDefs( svgtDoc );
		  removeSVGNameSpaces( svgtDoc );
	  }
	  catch ( Exception ex )
	  {
		  ex.printStackTrace();
	  }
   }

}
