/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Symbian Foundation License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.svg2svgt.converter;

import java.util.ArrayList;
import java.util.Iterator;

/**
 * <code>SVGModule</code> acts as a wrapper class for holding the
 * <code>SVGElements</code> that form a part of this module.
 * 
 */
public class SVGModule {
	/**
	 * List of elements contained in this module.
	 */
	private ArrayList myElements = null;

	/**
	 * Name of this module.
	 */
	private String myName = null;

	/**
	 * Creates an empty module with the specified module name.
	 * 
	 * @param moduleName
	 *            Name of the module.
	 */
	public SVGModule(String moduleName) {
		myName = moduleName;
	}

	/**
	 * Adds an <code>SVGElement</code> to this module.
	 * 
	 * @param element
	 *            Element to be added
	 */
	public void addElement(SVGElement element) {
		if (null == myElements) {
			myElements = new ArrayList();
		}

		if (null != element) {
			myElements.add(element);
		}
	}

	/**
	 * Adds an <code>SVGElement</code> to this module.
	 * 
	 * @param elementList
	 *            List of elements to be added
	 */
	public void addElements(ArrayList elementList) {
		if (null == myElements) {
			myElements = new ArrayList();
		}
		if ((null != elementList) || (0 < elementList.size())) {
			myElements.addAll(elementList);
		}
	}

	/**
	 * Removes an <code>SVGElement</code> from this module.
	 * 
	 * @param elementName
	 *            Element name to be removed.
	 */
	public void removeElement(String elementName) {
		int elementIndex = this.containsElement(elementName);
		if (-1 != elementIndex) {
			myElements.remove(elementIndex);
		}
	}

	/**
	 * Returns the name for this module.
	 * 
	 * @return Module name.
	 */
	public String getModuleName() {
		return myName;
	}

	/**
	 * Returns all the <code>SVGElements</code> contained in this module.
	 * 
	 * @return List of all the <code>SVGElements</code>.
	 */
	public ArrayList getModuleElements() {
		return myElements;
	}

	/**
	 * Checks if the element with the specified name is contained in this module
	 * or not.
	 * 
	 * @param elementName
	 *            Name of the element to be searched.
	 * @return Index of the element if found, else -1.
	 */
	public int containsElement(String elementName) {
		if (null == myElements) {
			return -1;
		}
		Iterator iter = myElements.iterator();
		SVGElement element = null;
		int indexCount = 0;
		while (true == iter.hasNext()) {
			element = (SVGElement) iter.next();
			if ((element.getElementName()).equalsIgnoreCase(elementName)) {
				return indexCount;
			}
			indexCount++;
		}
		return -1;
	}

	/**
	 * Checks if the attribute with the specified name is contained in this
	 * module or not.
	 * 
	 * @param attrName
	 *            Attribute name to be searched.
	 * @return <code>SVGAttribute</code> instance or null.
	 */
	public SVGAttribute getAttributeByTagName(String attrName) {
		if (null == myElements) {
			return null;
		}

		Iterator iter = myElements.iterator();
		SVGElement element = null;
		int indexCount = 0;
		while (true == iter.hasNext()) {
			element = (SVGElement) iter.next();
			int attrIndex = element.containsAttribute(attrName);
			if (-1 != attrIndex) {
				return element.getAttributeAt(attrIndex);
			}
		}
		return null;
	}

	/**
	 * Searches the module for the element specified by input name.
	 * 
	 * @param elemName
	 *            Name of the element to be searched.
	 * @return <code>SVGElement</code> instance or null.
	 */
	public SVGElement getElementByTagName(String elemName) {
		int elementIndex = containsElement(elemName);
		if (-1 != elementIndex) {
			return (SVGElement) myElements.get(elementIndex);
		}
		return null;
	}

	/**
	 * Obtains the <code>SVGElement</code> instance containg the attribute by
	 * the specified name. First matching <code>SVGElement</code> instance is
	 * returned.
	 * 
	 * @param attrName
	 *            Attribute name.
	 * @return </code>SVGElement</code> instance.
	 */
	public SVGElement getElementContainingAttribute(String attrName) {
		if (null == myElements) {
			return null;
		}
		Iterator iter = myElements.iterator();
		SVGElement element = null;
		while (true == iter.hasNext()) {
			element = (SVGElement) iter.next();
			int attrIndex = element.containsAttribute(attrName);
			if (-1 != attrIndex) {
				return element;
			}
		}
		return null;
	}

	/**
	 * Overloaded to String method. Returns the XML format for this module
	 * 
	 * @return XML format for this module.
	 */
	public String toString() {
		return myName;
	}

	/**
	 * Main method for testing
	 * 
	 * @param args
	 *            Command line arguments
	 */
	public static void main(String[] args) {
	}
}
