/*
* Copyright (c) 2006-2010 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/
package com.nokia.tools.media.symbian.mbm;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * This is ported from bmconv and can be used to extract bitmaps from .mbm/.mbr
 * file.<br/> Note: mbm creation is not ported.
 */
public class BitmapConverter {
	private static final int EMonochromeBitmap = 0;
	private static final int EColorBitmap = 1;

	private static final int ENoBitmapCompression = 0;
	private static final int EByteRLECompression = 1;
	private static final int ETwelveBitRLECompression = 2;
	private static final int ESixteenBitRLECompression = 3;
	private static final int ETwentyFourBitRLECompression = 4;

	private static final int KWriteOnceFileStoreUid = 268435511;
	private static final int KMultiBitmapRomImageUid = 268435521;
	private static final int KMultiBitmapFileImageUid = 268435522;
	private static final int KMultiBitmapFileImageChecksum = 1194943545;

	private InputStream in;
	private int numberOfBitmaps = -1;
	private Header header;
	private byte[] pbm;

	/**
	 * Constructs a converter with the specific input stream from where the mbm
	 * data will be read.
	 * 
	 * @param input the input stream to read mbm data.
	 */
	public BitmapConverter(InputStream input) {
		in = new BufferedInputStream(input);
		in.mark(Integer.MAX_VALUE);
	}

	/**
	 * Returns the number of bitmaps in the mbm stream.
	 * 
	 * @return the total number of bitmaps.
	 * @throws IOException if I/O error occurred.
	 */
	public int getNumberOfBitmaps() throws IOException {
		if (numberOfBitmaps < 0) {
			loadMbm(0, false);
		}
		return numberOfBitmaps;
	}

	/**
	 * Extracts the bitmap at the given index to the output stream.
	 * 
	 * @param index the bitmap index.
	 * @param out the output stream for writing the data.
	 * @throws IOException if I/O error occurred.
	 */
	public void extract(int index, OutputStream out) throws IOException {
		loadMbm(index, true);

		BitmapInfoHeader infoHeader = new BitmapInfoHeader();
		infoHeader.biSize = BitmapInfoHeader.size();
		infoHeader.biWidth = header.widthInPixels;
		infoHeader.biHeight = header.heightInPixels;
		infoHeader.biPlanes = 1;
		infoHeader.biBitCount = 24;
		infoHeader.biCompression = 0;
		infoHeader.biSizeImage = 0;
		infoHeader.biXPelsPerMeter = 0;
		infoHeader.biYPelsPerMeter = 0;
		infoHeader.biClrUsed = 0;
		infoHeader.biClrImportant = 0;

		int byteWidth = ((infoHeader.biWidth * 3) + 3) & ~3;
		int destLength = infoHeader.biHeight * byteWidth;

		BitmapFileHeader fileHeader = new BitmapFileHeader();
		fileHeader.bfType = 'B' + ('M' << 8);
		fileHeader.bfSize = BitmapFileHeader.size() + BitmapInfoHeader.size()
				+ destLength;
		fileHeader.bfReserved1 = 0;
		fileHeader.bfReserved2 = 0;
		fileHeader.bfOffBits = BitmapFileHeader.size()
				+ BitmapInfoHeader.size();

		byte[] buf = new byte[destLength];
		for (int y = 0; y < infoHeader.biHeight; y++) {
			int p = (infoHeader.biHeight - y - 1) * byteWidth;
			for (int x = 0; x < infoHeader.biWidth; x++) {
				RGB pixel = getPixel(x, y);
				buf[p++] = (byte) (pixel.b & 0xff);
				buf[p++] = (byte) (pixel.g & 0xff);
				buf[p++] = (byte) (pixel.r & 0xff);
			}
		}
		try {
			fileHeader.write(out);
			infoHeader.write(out);
			out.write(buf);
			out.flush();
		} finally {
			out.close();
		}
	}

	/**
	 * Closes the input stream.
	 */
	public void close() {
		try {
			if (in != null) {
				in.close();
			}
		} catch (Exception e) {
		}
	}

	private void loadMbm(int index, boolean readContent) throws IOException {
		in.reset();
		int wordBuffer = readInt(in);
		if (wordBuffer == KMultiBitmapRomImageUid) {
			loadRom(index, readContent);
		} else {
			loadFile(index, readContent);
		}
	}

	private void loadRom(int index, boolean readContent) throws IOException {
		in.reset();
		in.mark(Integer.MAX_VALUE);
		int wordBuffer = readInt(in);
		if (wordBuffer != KMultiBitmapRomImageUid) {
			throw new IOException("Wrong format: KMultiBitmapRomImageUid");
		}
		numberOfBitmaps = readInt(in);

		if (!readContent) {
			return;
		}

		if (index >= numberOfBitmaps) {
			throw new IOException("Out of range: count: " + wordBuffer
					+ " index: " + index);
		}

		skip(in, index * 4);
		wordBuffer = readInt(in);
		in.reset();

		skip(in, wordBuffer);

		Bitmap bitmap = new Bitmap();
		bitmap.uid = readInt(in);
		bitmap.dispMode = readInt(in);
		readInt(in); // heap
		readInt(in); // pile
		bitmap.byteWidth = readInt(in);
		bitmap.header = new Header(in);

		this.header = bitmap.header;

		bitmap.dataOffset = readInt(in);
		int size = bitmap.header.bitmapSize - 10 * 4;
		pbm = new byte[size];
		int read = 0;
		while (read < size) {
			read += in.read(pbm, read, size - read);
		}
	}

	private void loadFile(int index, boolean readContent) throws IOException {
		in.reset();
		int wordBuffer = readInt(in);
		if (wordBuffer != KWriteOnceFileStoreUid) {
			throw new IOException("Wrong format: not KWriteOnceFileStoreUid");
		}
		wordBuffer = readInt(in);
		if (wordBuffer != KMultiBitmapFileImageUid) {
			throw new IOException("Wrong format: not KMultiBitmapFileImageUid");
		}

		wordBuffer = readInt(in);
		if (wordBuffer != 0) {
			throw new IOException("Wrong format: not 0");
		}
		wordBuffer = readInt(in);
		if (wordBuffer != KMultiBitmapFileImageChecksum) {
			throw new IOException("Wrong format: KMultiBitmapFileImageChecksum");
		}
		wordBuffer = readInt(in);
		in.reset();

		skip(in, wordBuffer);

		numberOfBitmaps = readInt(in);

		if (!readContent) {
			return;
		}
		if (index >= numberOfBitmaps) {
			throw new IOException("Out of range: bitmaps: " + numberOfBitmaps
					+ ", index: " + index);
		}

		skip(in, index * 4);

		wordBuffer = readInt(in);
		in.reset();

		skip(in, wordBuffer);

		header = new Header(in);
		int size = header.bitmapSize - header.structSize;

		pbm = new byte[size];
		int read = 0;
		while (read < size) {
			read += in.read(pbm, read, size - read);
		}

		if (header.compression != ENoBitmapCompression) {
			int byteWidth = byteWidth(header.widthInPixels, header.bitsPerPixel);
			int expandedSize = byteWidth * header.heightInPixels;
			byte[] buf = new byte[expandedSize];
			switch (header.compression) {
			case EByteRLECompression:
				expandByteRLEData(pbm, buf);
				break;
			case ETwelveBitRLECompression:
				expandTwelveBitRLEData(pbm, buf);
				break;
			case ESixteenBitRLECompression:
				expandSixteenBitRLEData(pbm, buf);
				break;
			case ETwentyFourBitRLECompression:
				expandTwentyFourBitRLEData(pbm, buf);
				break;
			default:
				throw new IOException("Unknown compression: "
						+ header.compression);
			}
			pbm = buf;
			header.compression = ENoBitmapCompression;
			header.bitmapSize += expandedSize - size;
		}
	}

	private static void skip(InputStream in, long toSkip) throws IOException {
		long skipped = 0;
		while (skipped < toSkip) {
			skipped += in.skip(toSkip - skipped);
		}
	}

	private static byte[] decode(short s) {
		byte[] buf = new byte[2];
		buf[0] = (byte) (s & 0xff);
		buf[1] = (byte) ((s >> 8) & 0xff);
		return buf;
	}

	private static byte[] decode(int i) {
		byte[] buf = new byte[4];
		buf[0] = (byte) (i & 0xff);
		buf[1] = (byte) ((i >> 8) & 0xff);
		buf[2] = (byte) ((i >> 16) & 0xff);
		buf[3] = (byte) ((i >> 24) & 0xff);
		return buf;
	}

	private static short readShort(byte[] buf, int offset) {
		int b1 = ((buf[offset + 1] & 0xff) << 8);
		int b2 = buf[offset] & 0xff;
		return (short) (b1 | b2);
	}

	private static int readInt(InputStream in) throws IOException {
		byte[] buf = new byte[4];
		int read = 0;
		while (read < buf.length) {
			read += in.read(buf, read, buf.length - read);
		}
		return ((buf[3] & 0xff) << 24) | ((buf[2] & 0xff) << 16)
				| ((buf[1] & 0xff) << 8) | (buf[0] & 0xff);
	}

	private static void writeData(int i, OutputStream out) throws IOException {
		out.write(i & 0xff);
		out.write((i >> 8) & 0xff);
		out.write((i >> 16) & 0xff);
		out.write((i >> 24) & 0xff);
	}

	private static void writeData(short s, OutputStream out) throws IOException {
		out.write(s & 0xff);
		out.write((s >> 8) & 0xff);
	}

	class Bitmap {
		int uid;
		int dispMode;
		// Heap heap;
		// Pile pile;
		int byteWidth;
		Header header;
		int chunk;
		int dataOffset;

		int size() {
			return 7 * 4 + header.size();
		}
	}

	class Header {
		int bitmapSize;
		int structSize;
		int widthInPixels;
		int heightInPixels;
		int widthInTwips;
		int heightInTwips;
		int bitsPerPixel;
		int color;
		int paletteEntries;
		int compression;

		Header(InputStream in) throws IOException {
			bitmapSize = readInt(in);
			structSize = readInt(in);
			widthInPixels = readInt(in);
			heightInPixels = readInt(in);
			widthInTwips = readInt(in);
			heightInTwips = readInt(in);
			bitsPerPixel = readInt(in);
			color = readInt(in);
			paletteEntries = readInt(in);
			compression = readInt(in);
		}

		int size() {
			return 10 * 4;
		}
	}

	public static int byteWidth(int pixelWidth, int bitsPerPixel) {
		int wordWidth = 0;

		switch (bitsPerPixel) {
		case 1:
			wordWidth = (pixelWidth + 31) / 32;
			break;
		case 2:
			wordWidth = (pixelWidth + 15) / 16;
			break;
		case 4:
			wordWidth = (pixelWidth + 7) / 8;
			break;
		case 8:
			wordWidth = (pixelWidth + 3) / 4;
			break;
		case 12:
		case 16:
			wordWidth = (pixelWidth + 1) / 2;
			break;
		case 24:
			wordWidth = (((pixelWidth * 3) + 11) / 12) * 3;
			break;
		case 32:
			wordWidth = pixelWidth;
			break;
		default:
			break;
		}

		return wordWidth * 4;
	}

	void expandByteRLEData(byte[] src, byte[] dst) throws IOException {
		int sp = 0, dp = 0;
		while (sp < src.length && dp < dst.length) {
			byte count = src[sp++];
			if (count < 0) {
				int runLength = -count;
				System.arraycopy(src, sp, dst, dp, runLength);
				sp += runLength;
				dp += runLength;
			} else {
				byte value = src[sp++];
				while (count >= 0) {
					dst[dp++] = value;
					count--;
				}
			}
		}
		if (sp != src.length || dp != dst.length) {
			throw new IOException("Decompression error");
		}
	}

	void expandTwelveBitRLEData(byte[] src, byte[] dst) throws IOException {
		int sp = 0, dp = 0;
		while (sp < src.length && dp < dst.length) {
			short value = readShort(src, sp);
			sp += 2;
			int runLength = ((value & 0xffff) >> 12) + 1;
			value &= 0x0fff;

			byte[] buf = decode(value);
			for (; runLength > 0; runLength--) {
				dst[dp++] = buf[0];
				dst[dp++] = buf[1];
			}
		}
		if (sp != src.length || dp != dst.length) {
			throw new IOException("Decompression error");
		}
	}

	void expandSixteenBitRLEData(byte[] src, byte[] dst) throws IOException {
		int sp = 0, dp = 0;
		while (sp < src.length && dp < dst.length) {
			int runLength = src[sp++];

			if (runLength >= 0) {
				short value = readShort(src, sp);
				sp += 2;
				byte[] buf = decode(value);
				for (runLength++; runLength > 0; runLength--) {
					dst[dp++] = buf[0];
					dst[dp++] = buf[1];
				}
			} else {
				runLength = -runLength;
				int byteLength = runLength * 2;
				System.arraycopy(src, sp, dst, dp, byteLength);
				sp += byteLength;
				dp += byteLength;
			}
		}
		if (sp != src.length || dp != dst.length) {
			throw new IOException("Decompression error");
		}
	}

	void expandTwentyFourBitRLEData(byte[] src, byte[] dst) throws IOException {
		int sp = 0, dp = 0;

		while (sp < src.length && dp < dst.length) {
			int runLength = src[sp++];

			if (runLength >= 0) {
				byte component1 = src[sp++];
				byte component2 = src[sp++];
				byte component3 = src[sp++];
				for (runLength++; runLength > 0; runLength--) {
					dst[dp++] = component1;
					dst[dp++] = component2;
					dst[dp++] = component3;
				}
			} else {
				runLength = -runLength;
				int byteLength = runLength * 3;
				System.arraycopy(src, sp, dst, dp, byteLength);
				sp += byteLength;
				dp += byteLength;
			}
		}

		if (sp != src.length || dp != dst.length) {
			throw new IOException("Decompression error");
		}
	}

	RGB getPixel(int x, int y) {
		byte col;
		x %= header.widthInPixels;
		y %= header.heightInPixels;
		int byteWidth = byteWidth(header.widthInPixels, header.bitsPerPixel);
		int yOffset = y * byteWidth;

		switch (header.bitsPerPixel) {
		case 1:
			col = pbm[yOffset + (x / 8)];
			col = (byte) ((col >> (x & 7)) & 0xff);
			return RGB.Gray2(col & 1);
		case 2:
			col = pbm[yOffset + (x / 4)];
			col = (byte) ((col >> (2 * (x % 4))) & 0xff);
			return RGB.Gray4(col & 3);
		case 4:
			col = pbm[yOffset + (x / 2)];
			if ((x & 1) == 1) {
				col = (byte) ((col >> 4) & 0xff);
			}
			col &= 0xf;
			if (header.color == EColorBitmap)
				return RGB.Color16(col);
			else
				return RGB.Gray16(col);
		case 8:
			col = pbm[yOffset + x];
			if (header.color == EColorBitmap)
				return RGB.Color256(col);
			else
				return RGB.Gray256(col);
		case 12:
		case 16: {
			short b = readShort(pbm, yOffset + (x * 2));
			if (header.bitsPerPixel == 12)
				return RGB.Color4K((int) b);

			return RGB.Color64K((int) b);
		}
		case 24: {
			int pp = yOffset + (x * 3);
			RGB pixelColor = new RGB();
			pixelColor.b = pbm[pp++];
			pixelColor.g = pbm[pp++];
			pixelColor.r = pbm[pp];
			return pixelColor;
		}
		case 32: {
			int p = yOffset + (x * 4);
			RGB pixelColor = new RGB();
			pixelColor.b = pbm[p++];
			pixelColor.g = pbm[p++];
			pixelColor.r = pbm[p++];
			pixelColor.s = pbm[p];
			return pixelColor;
		}
		default:
			return new RGB(0);
		}
	}

	static class RGB {
		private static final int[] color16array = { 0x00000000, 0x00555555,
				0x00000080, 0x00008080, 0x00008000, 0x000000ff, 0x0000ffff,
				0x0000ff00, 0x00ff00ff, 0x00ff0000, 0x00ffff00, 0x00800080,
				0x00800000, 0x00808000, 0x00aaaaaa, 0x00ffffff };

		private static final byte[] color16inverse = { 0x00, 0x00, 0x02, 0x02,
				0x02, 0x02, 0x05, 0x05, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02,
				0x05, 0x05, 0x04, 0x04, 0x01, 0x03, 0x03, 0x03, 0x05, 0x05,
				0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x03, 0x05, 0x04, 0x04,
				0x03, 0x03, 0x03, 0x03, 0x03, 0x06, 0x04, 0x04, 0x03, 0x03,
				0x03, 0x03, 0x06, 0x06, 0x07, 0x07, 0x07, 0x03, 0x03, 0x06,
				0x06, 0x06, 0x07, 0x07, 0x07, 0x07, 0x06, 0x06, 0x06, 0x06,
				0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x05, 0x05, 0x00, 0x00,
				0x01, 0x02, 0x02, 0x02, 0x05, 0x05, 0x04, 0x01, 0x01, 0x01,
				0x03, 0x03, 0x05, 0x05, 0x04, 0x04, 0x01, 0x03, 0x03, 0x03,
				0x03, 0x05, 0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x03, 0x06,
				0x04, 0x04, 0x03, 0x03, 0x03, 0x03, 0x06, 0x06, 0x07, 0x07,
				0x07, 0x03, 0x03, 0x06, 0x06, 0x06, 0x07, 0x07, 0x07, 0x07,
				0x06, 0x06, 0x06, 0x06, 0x0c, 0x0c, 0x01, 0x0b, 0x0b, 0x0b,
				0x05, 0x05, 0x0c, 0x01, 0x01, 0x01, 0x0b, 0x0b, 0x05, 0x05,
				0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x05, 0x0d, 0x01,
				0x01, 0x01, 0x01, 0x01, 0x01, 0x05, 0x0d, 0x0d, 0x01, 0x01,
				0x03, 0x0e, 0x0e, 0x06, 0x0d, 0x0d, 0x01, 0x01, 0x0e, 0x0e,
				0x0e, 0x06, 0x07, 0x07, 0x01, 0x01, 0x0e, 0x0e, 0x06, 0x06,
				0x07, 0x07, 0x07, 0x07, 0x06, 0x06, 0x06, 0x06, 0x0c, 0x0c,
				0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x05, 0x0c, 0x0c, 0x01, 0x0b,
				0x0b, 0x0b, 0x0b, 0x05, 0x0d, 0x01, 0x01, 0x01, 0x01, 0x01,
				0x01, 0x05, 0x0d, 0x0d, 0x01, 0x01, 0x01, 0x0e, 0x0e, 0x0e,
				0x0d, 0x0d, 0x01, 0x01, 0x0e, 0x0e, 0x0e, 0x0e, 0x0d, 0x0d,
				0x01, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0d, 0x0d, 0x01, 0x0e,
				0x0e, 0x0e, 0x0e, 0x06, 0x07, 0x07, 0x07, 0x0e, 0x0e, 0x0e,
				0x06, 0x06, 0x0c, 0x0c, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x08,
				0x0c, 0x0c, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x08, 0x0d, 0x0d,
				0x01, 0x01, 0x0b, 0x0e, 0x0e, 0x08, 0x0d, 0x0d, 0x01, 0x01,
				0x0e, 0x0e, 0x0e, 0x0e, 0x0d, 0x0d, 0x0d, 0x0e, 0x0e, 0x0e,
				0x0e, 0x0e, 0x0d, 0x0d, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e,
				0x0d, 0x0d, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0f, 0x0a, 0x0a,
				0x0a, 0x0e, 0x0e, 0x0e, 0x0f, 0x0f, 0x0c, 0x0c, 0x0b, 0x0b,
				0x0b, 0x0b, 0x08, 0x08, 0x0c, 0x0c, 0x0b, 0x0b, 0x0b, 0x0b,
				0x08, 0x08, 0x0d, 0x0d, 0x01, 0x01, 0x0e, 0x0e, 0x0e, 0x08,
				0x0d, 0x0d, 0x01, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0d, 0x0d,
				0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0d, 0x0d, 0x0e, 0x0e,
				0x0e, 0x0e, 0x0e, 0x0e, 0x0a, 0x0a, 0x0e, 0x0e, 0x0e, 0x0e,
				0x0e, 0x0f, 0x0a, 0x0a, 0x0a, 0x0e, 0x0e, 0x0e, 0x0f, 0x0f,
				0x09, 0x09, 0x09, 0x0b, 0x0b, 0x08, 0x08, 0x08, 0x09, 0x09,
				0x09, 0x0b, 0x0b, 0x08, 0x08, 0x08, 0x09, 0x09, 0x01, 0x01,
				0x0e, 0x0e, 0x08, 0x08, 0x0d, 0x0d, 0x01, 0x0e, 0x0e, 0x0e,
				0x0e, 0x08, 0x0d, 0x0d, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0f,
				0x0a, 0x0a, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0f, 0x0a, 0x0a,
				0x0a, 0x0e, 0x0e, 0x0e, 0x0f, 0x0f, 0x0a, 0x0a, 0x0a, 0x0a,
				0x0f, 0x0f, 0x0f, 0x0f, 0x09, 0x09, 0x09, 0x09, 0x08, 0x08,
				0x08, 0x08, 0x09, 0x09, 0x09, 0x09, 0x08, 0x08, 0x08, 0x08,
				0x09, 0x09, 0x09, 0x09, 0x08, 0x08, 0x08, 0x08, 0x09, 0x09,
				0x09, 0x0e, 0x0e, 0x0e, 0x08, 0x08, 0x0a, 0x0a, 0x0a, 0x0e,
				0x0e, 0x0e, 0x0f, 0x0f, 0x0a, 0x0a, 0x0a, 0x0e, 0x0e, 0x0e,
				0x0f, 0x0f, 0x0a, 0x0a, 0x0a, 0x0a, 0x0f, 0x0f, 0x0f, 0x0f,
				0x0a, 0x0a, 0x0a, 0x0a, 0x0f, 0x0f, 0x0f, 0x0f };

		private static final int[] color256array = { 0x00000000, 0x00000033,
				0x00000066, 0x00000099, 0x000000cc, 0x000000ff, 0x00003300,
				0x00003333, 0x00003366, 0x00003399, 0x000033cc, 0x000033ff,
				0x00006600, 0x00006633, 0x00006666, 0x00006699, 0x000066cc,
				0x000066ff, 0x00009900, 0x00009933, 0x00009966, 0x00009999,
				0x000099cc, 0x000099ff, 0x0000cc00, 0x0000cc33, 0x0000cc66,
				0x0000cc99, 0x0000cccc, 0x0000ccff, 0x0000ff00, 0x0000ff33,
				0x0000ff66, 0x0000ff99, 0x0000ffcc, 0x0000ffff,

				0x00330000, 0x00330033, 0x00330066, 0x00330099, 0x003300cc,
				0x003300ff, 0x00333300, 0x00333333, 0x00333366, 0x00333399,
				0x003333cc, 0x003333ff, 0x00336600, 0x00336633, 0x00336666,
				0x00336699, 0x003366cc, 0x003366ff, 0x00339900, 0x00339933,
				0x00339966, 0x00339999, 0x003399cc, 0x003399ff, 0x0033cc00,
				0x0033cc33, 0x0033cc66, 0x0033cc99, 0x0033cccc, 0x0033ccff,
				0x0033ff00, 0x0033ff33, 0x0033ff66, 0x0033ff99, 0x0033ffcc,
				0x0033ffff,

				0x00660000, 0x00660033, 0x00660066, 0x00660099, 0x006600cc,
				0x006600ff, 0x00663300, 0x00663333, 0x00663366, 0x00663399,
				0x006633cc, 0x006633ff, 0x00666600, 0x00666633, 0x00666666,
				0x00666699, 0x006666cc, 0x006666ff, 0x00669900, 0x00669933,
				0x00669966, 0x00669999, 0x006699cc, 0x006699ff, 0x0066cc00,
				0x0066cc33, 0x0066cc66, 0x0066cc99, 0x0066cccc, 0x0066ccff,
				0x0066ff00, 0x0066ff33, 0x0066ff66, 0x0066ff99, 0x0066ffcc,
				0x0066ffff,

				0x00111111, 0x00222222, 0x00444444, 0x00555555, 0x00777777,
				0x00000011, 0x00000022, 0x00000044, 0x00000055, 0x00000077,
				0x00001100, 0x00002200, 0x00004400, 0x00005500, 0x00007700,
				0x00110000, 0x00220000, 0x00440000, 0x00550000, 0x00770000,

				0x00880000, 0x00aa0000, 0x00bb0000, 0x00dd0000, 0x00ee0000,
				0x00008800, 0x0000aa00, 0x0000bb00, 0x0000dd00, 0x0000ee00,
				0x00000088, 0x000000aa, 0x000000bb, 0x000000dd, 0x000000ee,
				0x00888888, 0x00aaaaaa, 0x00bbbbbb, 0x00dddddd, 0x00eeeeee,

				0x00990000, 0x00990033, 0x00990066, 0x00990099, 0x009900cc,
				0x009900ff, 0x00993300, 0x00993333, 0x00993366, 0x00993399,
				0x009933cc, 0x009933ff, 0x00996600, 0x00996633, 0x00996666,
				0x00996699, 0x009966cc, 0x009966ff, 0x00999900, 0x00999933,
				0x00999966, 0x00999999, 0x009999cc, 0x009999ff, 0x0099cc00,
				0x0099cc33, 0x0099cc66, 0x0099cc99, 0x0099cccc, 0x0099ccff,
				0x0099ff00, 0x0099ff33, 0x0099ff66, 0x0099ff99, 0x0099ffcc,
				0x0099ffff,

				0x00cc0000, 0x00cc0033, 0x00cc0066, 0x00cc0099, 0x00cc00cc,
				0x00cc00ff, 0x00cc3300, 0x00cc3333, 0x00cc3366, 0x00cc3399,
				0x00cc33cc, 0x00cc33ff, 0x00cc6600, 0x00cc6633, 0x00cc6666,
				0x00cc6699, 0x00cc66cc, 0x00cc66ff, 0x00cc9900, 0x00cc9933,
				0x00cc9966, 0x00cc9999, 0x00cc99cc, 0x00cc99ff, 0x00cccc00,
				0x00cccc33, 0x00cccc66, 0x00cccc99, 0x00cccccc, 0x00ccccff,
				0x00ccff00, 0x00ccff33, 0x00ccff66, 0x00ccff99, 0x00ccffcc,
				0x00ccffff,

				0x00ff0000, 0x00ff0033, 0x00ff0066, 0x00ff0099, 0x00ff00cc,
				0x00ff00ff, 0x00ff3300, 0x00ff3333, 0x00ff3366, 0x00ff3399,
				0x00ff33cc, 0x00ff33ff, 0x00ff6600, 0x00ff6633, 0x00ff6666,
				0x00ff6699, 0x00ff66cc, 0x00ff66ff, 0x00ff9900, 0x00ff9933,
				0x00ff9966, 0x00ff9999, 0x00ff99cc, 0x00ff99ff, 0x00ffcc00,
				0x00ffcc33, 0x00ffcc66, 0x00ffcc99, 0x00ffcccc, 0x00ffccff,
				0x00ffff00, 0x00ffff33, 0x00ffff66, 0x00ffff99, 0x00ffffcc,
				0x00ffffff };

		private static final short[] color256inverse = { 0x00, 0x71, 0x72,
				0x01, 0x73, 0x74, 0x02, 0x75, 0x8a, 0x03, 0x8b, 0x8c, 0x04,
				0x8d, 0x8e, 0x05, 0x76, 0x6c, 0x72, 0x01, 0x73, 0x74, 0x02,
				0x75, 0x8a, 0x03, 0x8b, 0x8c, 0x04, 0x8d, 0x8e, 0x05, 0x77,
				0x77, 0x07, 0x07, 0x07, 0x08, 0x08, 0x08, 0x09, 0x09, 0x09,
				0x0a, 0x0a, 0x0a, 0x0b, 0x0b, 0x06, 0x06, 0x07, 0x07, 0x07,
				0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x0a, 0x0a, 0x0a, 0x0b,
				0x0b, 0x78, 0x78, 0x07, 0x07, 0x07, 0x08, 0x08, 0x08, 0x09,
				0x09, 0x09, 0x0a, 0x0a, 0x0a, 0x0b, 0x0b, 0x79, 0x79, 0x0d,
				0x0d, 0x0d, 0x0e, 0x0e, 0x0e, 0x0f, 0x0f, 0x0f, 0x10, 0x10,
				0x10, 0x11, 0x11, 0x0c, 0x0c, 0x0d, 0x0d, 0x0d, 0x0e, 0x0e,
				0x0e, 0x0f, 0x0f, 0x0f, 0x10, 0x10, 0x10, 0x11, 0x11, 0x7a,
				0x7a, 0x0d, 0x0d, 0x0d, 0x0e, 0x0e, 0x0e, 0x0f, 0x0f, 0x0f,
				0x10, 0x10, 0x10, 0x11, 0x11, 0x85, 0x85, 0x13, 0x13, 0x13,
				0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x16, 0x16, 0x16, 0x17,
				0x17, 0x12, 0x12, 0x13, 0x13, 0x13, 0x14, 0x14, 0x14, 0x15,
				0x15, 0x15, 0x16, 0x16, 0x16, 0x17, 0x17, 0x86, 0x86, 0x13,
				0x13, 0x13, 0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x16, 0x16,
				0x16, 0x17, 0x17, 0x87, 0x87, 0x19, 0x19, 0x19, 0x1a, 0x1a,
				0x1a, 0x1b, 0x1b, 0x1b, 0x1c, 0x1c, 0x1c, 0x1d, 0x1d, 0x18,
				0x18, 0x19, 0x19, 0x19, 0x1a, 0x1a, 0x1a, 0x1b, 0x1b, 0x1b,
				0x1c, 0x1c, 0x1c, 0x1d, 0x1d, 0x88, 0x88, 0x19, 0x19, 0x19,
				0x1a, 0x1a, 0x1a, 0x1b, 0x1b, 0x1b, 0x1c, 0x1c, 0x1c, 0x1d,
				0x1d, 0x89, 0x89, 0x1f, 0x1f, 0x1f, 0x20, 0x20, 0x20, 0x21,
				0x21, 0x21, 0x22, 0x22, 0x22, 0x23, 0x23, 0x1e, 0x1e, 0x1f,
				0x1f, 0x1f, 0x20, 0x20, 0x20, 0x21, 0x21, 0x21, 0x22, 0x22,
				0x22, 0x23, 0x23, 0x7b, 0x6c, 0x72, 0x01, 0x73, 0x74, 0x02,
				0x75, 0x8a, 0x03, 0x8b, 0x8c, 0x04, 0x8d, 0x8e, 0x05, 0x6c,
				0x6c, 0x6c, 0x01, 0x73, 0x74, 0x02, 0x75, 0x8a, 0x03, 0x8b,
				0x8c, 0x04, 0x8d, 0x8e, 0x05, 0x77, 0x6c, 0x6d, 0x07, 0x07,
				0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x0a, 0x0a, 0x0a, 0x0b,
				0x0b, 0x06, 0x06, 0x07, 0x07, 0x07, 0x08, 0x08, 0x08, 0x09,
				0x09, 0x09, 0x0a, 0x0a, 0x0a, 0x0b, 0x0b, 0x78, 0x78, 0x07,
				0x07, 0x07, 0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x0a, 0x0a,
				0x0a, 0x0b, 0x0b, 0x79, 0x79, 0x0d, 0x0d, 0x0d, 0x0e, 0x0e,
				0x0e, 0x0f, 0x0f, 0x0f, 0x10, 0x10, 0x10, 0x11, 0x11, 0x0c,
				0x0c, 0x0d, 0x0d, 0x0d, 0x0e, 0x0e, 0x0e, 0x0f, 0x0f, 0x0f,
				0x10, 0x10, 0x10, 0x11, 0x11, 0x7a, 0x7a, 0x0d, 0x0d, 0x0d,
				0x0e, 0x0e, 0x0e, 0x0f, 0x0f, 0x0f, 0x10, 0x10, 0x10, 0x11,
				0x11, 0x85, 0x85, 0x13, 0x13, 0x13, 0x14, 0x14, 0x14, 0x15,
				0x15, 0x15, 0x16, 0x16, 0x16, 0x17, 0x17, 0x12, 0x12, 0x13,
				0x13, 0x13, 0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x16, 0x16,
				0x16, 0x17, 0x17, 0x86, 0x86, 0x13, 0x13, 0x13, 0x14, 0x14,
				0x14, 0x15, 0x15, 0x15, 0x16, 0x16, 0x16, 0x17, 0x17, 0x87,
				0x87, 0x19, 0x19, 0x19, 0x1a, 0x1a, 0x1a, 0x1b, 0x1b, 0x1b,
				0x1c, 0x1c, 0x1c, 0x1d, 0x1d, 0x18, 0x18, 0x19, 0x19, 0x19,
				0x1a, 0x1a, 0x1a, 0x1b, 0x1b, 0x1b, 0x1c, 0x1c, 0x1c, 0x1d,
				0x1d, 0x88, 0x88, 0x19, 0x19, 0x19, 0x1a, 0x1a, 0x1a, 0x1b,
				0x1b, 0x1b, 0x1c, 0x1c, 0x1c, 0x1d, 0x1d, 0x89, 0x89, 0x1f,
				0x1f, 0x1f, 0x20, 0x20, 0x20, 0x21, 0x21, 0x21, 0x22, 0x22,
				0x22, 0x23, 0x23, 0x1e, 0x1e, 0x1f, 0x1f, 0x1f, 0x20, 0x20,
				0x20, 0x21, 0x21, 0x21, 0x22, 0x22, 0x22, 0x23, 0x23, 0x7c,
				0x7c, 0x25, 0x25, 0x25, 0x26, 0x26, 0x26, 0x27, 0x27, 0x27,
				0x28, 0x28, 0x28, 0x29, 0x29, 0x7c, 0x6c, 0x6d, 0x25, 0x25,
				0x26, 0x26, 0x26, 0x27, 0x27, 0x27, 0x28, 0x28, 0x28, 0x29,
				0x29, 0x2a, 0x6d, 0x6d, 0x6d, 0x6d, 0x2c, 0x2c, 0x2c, 0x2d,
				0x2d, 0x2d, 0x2e, 0x2e, 0x2e, 0x2f, 0x2f, 0x2a, 0x2a, 0x6d,
				0x2b, 0x2b, 0x2c, 0x2c, 0x2c, 0x2d, 0x2d, 0x2d, 0x2e, 0x2e,
				0x2e, 0x2f, 0x2f, 0x2a, 0x2a, 0x6d, 0x2b, 0x6e, 0x2c, 0x2c,
				0x2c, 0x2d, 0x2d, 0x2d, 0x2e, 0x2e, 0x2e, 0x2f, 0x2f, 0x30,
				0x30, 0x31, 0x31, 0x31, 0x32, 0x32, 0x32, 0x33, 0x33, 0x33,
				0x34, 0x34, 0x34, 0x35, 0x35, 0x30, 0x30, 0x31, 0x31, 0x31,
				0x32, 0x32, 0x32, 0x33, 0x33, 0x33, 0x34, 0x34, 0x34, 0x35,
				0x35, 0x30, 0x30, 0x31, 0x31, 0x31, 0x32, 0x32, 0x32, 0x33,
				0x33, 0x33, 0x34, 0x34, 0x34, 0x35, 0x35, 0x36, 0x36, 0x37,
				0x37, 0x37, 0x38, 0x38, 0x38, 0x39, 0x39, 0x39, 0x3a, 0x3a,
				0x3a, 0x3b, 0x3b, 0x36, 0x36, 0x37, 0x37, 0x37, 0x38, 0x38,
				0x38, 0x39, 0x39, 0x39, 0x3a, 0x3a, 0x3a, 0x3b, 0x3b, 0x36,
				0x36, 0x37, 0x37, 0x37, 0x38, 0x38, 0x38, 0x39, 0x39, 0x39,
				0x3a, 0x3a, 0x3a, 0x3b, 0x3b, 0x3c, 0x3c, 0x3d, 0x3d, 0x3d,
				0x3e, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f, 0x40, 0x40, 0x40, 0x41,
				0x41, 0x3c, 0x3c, 0x3d, 0x3d, 0x3d, 0x3e, 0x3e, 0x3e, 0x3f,
				0x3f, 0x3f, 0x40, 0x40, 0x40, 0x41, 0x41, 0x3c, 0x3c, 0x3d,
				0x3d, 0x3d, 0x3e, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f, 0x40, 0x40,
				0x40, 0x41, 0x41, 0x42, 0x42, 0x43, 0x43, 0x43, 0x44, 0x44,
				0x44, 0x45, 0x45, 0x45, 0x46, 0x46, 0x46, 0x47, 0x47, 0x42,
				0x42, 0x43, 0x43, 0x43, 0x44, 0x44, 0x44, 0x45, 0x45, 0x45,
				0x46, 0x46, 0x46, 0x47, 0x47, 0x24, 0x24, 0x25, 0x25, 0x25,
				0x26, 0x26, 0x26, 0x27, 0x27, 0x27, 0x28, 0x28, 0x28, 0x29,
				0x29, 0x24, 0x24, 0x25, 0x25, 0x25, 0x26, 0x26, 0x26, 0x27,
				0x27, 0x27, 0x28, 0x28, 0x28, 0x29, 0x29, 0x2a, 0x2a, 0x6d,
				0x2b, 0x2b, 0x2c, 0x2c, 0x2c, 0x2d, 0x2d, 0x2d, 0x2e, 0x2e,
				0x2e, 0x2f, 0x2f, 0x2a, 0x2a, 0x2b, 0x2b, 0x2b, 0x2c, 0x2c,
				0x2c, 0x2d, 0x2d, 0x2d, 0x2e, 0x2e, 0x2e, 0x2f, 0x2f, 0x2a,
				0x2a, 0x2b, 0x2b, 0x6e, 0x2c, 0x2c, 0x2c, 0x2d, 0x2d, 0x2d,
				0x2e, 0x2e, 0x2e, 0x2f, 0x2f, 0x30, 0x30, 0x31, 0x31, 0x31,
				0x32, 0x32, 0x32, 0x33, 0x33, 0x33, 0x34, 0x34, 0x34, 0x35,
				0x35, 0x30, 0x30, 0x31, 0x31, 0x31, 0x32, 0x32, 0x32, 0x33,
				0x33, 0x33, 0x34, 0x34, 0x34, 0x35, 0x35, 0x30, 0x30, 0x31,
				0x31, 0x31, 0x32, 0x32, 0x32, 0x33, 0x33, 0x33, 0x34, 0x34,
				0x34, 0x35, 0x35, 0x36, 0x36, 0x37, 0x37, 0x37, 0x38, 0x38,
				0x38, 0x39, 0x39, 0x39, 0x3a, 0x3a, 0x3a, 0x3b, 0x3b, 0x36,
				0x36, 0x37, 0x37, 0x37, 0x38, 0x38, 0x38, 0x39, 0x39, 0x39,
				0x3a, 0x3a, 0x3a, 0x3b, 0x3b, 0x36, 0x36, 0x37, 0x37, 0x37,
				0x38, 0x38, 0x38, 0x39, 0x39, 0x39, 0x3a, 0x3a, 0x3a, 0x3b,
				0x3b, 0x3c, 0x3c, 0x3d, 0x3d, 0x3d, 0x3e, 0x3e, 0x3e, 0x3f,
				0x3f, 0x3f, 0x40, 0x40, 0x40, 0x41, 0x41, 0x3c, 0x3c, 0x3d,
				0x3d, 0x3d, 0x3e, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f, 0x40, 0x40,
				0x40, 0x41, 0x41, 0x3c, 0x3c, 0x3d, 0x3d, 0x3d, 0x3e, 0x3e,
				0x3e, 0x3f, 0x3f, 0x3f, 0x40, 0x40, 0x40, 0x41, 0x41, 0x42,
				0x42, 0x43, 0x43, 0x43, 0x44, 0x44, 0x44, 0x45, 0x45, 0x45,
				0x46, 0x46, 0x46, 0x47, 0x47, 0x42, 0x42, 0x43, 0x43, 0x43,
				0x44, 0x44, 0x44, 0x45, 0x45, 0x45, 0x46, 0x46, 0x46, 0x47,
				0x47, 0x7d, 0x7d, 0x25, 0x25, 0x25, 0x26, 0x26, 0x26, 0x27,
				0x27, 0x27, 0x28, 0x28, 0x28, 0x29, 0x29, 0x7d, 0x7d, 0x25,
				0x25, 0x25, 0x26, 0x26, 0x26, 0x27, 0x27, 0x27, 0x28, 0x28,
				0x28, 0x29, 0x29, 0x2a, 0x2a, 0x6d, 0x2b, 0x6e, 0x2c, 0x2c,
				0x2c, 0x2d, 0x2d, 0x2d, 0x2e, 0x2e, 0x2e, 0x2f, 0x2f, 0x2a,
				0x2a, 0x2b, 0x2b, 0x6e, 0x2c, 0x2c, 0x2c, 0x2d, 0x2d, 0x2d,
				0x2e, 0x2e, 0x2e, 0x2f, 0x2f, 0x2a, 0x2a, 0x6e, 0x6e, 0x6e,
				0x6e, 0x2c, 0x2c, 0x2d, 0x2d, 0x2d, 0x2e, 0x2e, 0x2e, 0x2f,
				0x2f, 0x30, 0x30, 0x31, 0x31, 0x6e, 0x6f, 0x32, 0x32, 0x33,
				0x33, 0x33, 0x34, 0x34, 0x34, 0x35, 0x35, 0x30, 0x30, 0x31,
				0x31, 0x31, 0x32, 0x32, 0x32, 0x33, 0x33, 0x33, 0x34, 0x34,
				0x34, 0x35, 0x35, 0x30, 0x30, 0x31, 0x31, 0x31, 0x32, 0x32,
				0x32, 0x33, 0x33, 0x33, 0x34, 0x34, 0x34, 0x35, 0x35, 0x36,
				0x36, 0x37, 0x37, 0x37, 0x38, 0x38, 0x38, 0x39, 0x39, 0x39,
				0x3a, 0x3a, 0x3a, 0x3b, 0x3b, 0x36, 0x36, 0x37, 0x37, 0x37,
				0x38, 0x38, 0x38, 0x39, 0x39, 0x39, 0x3a, 0x3a, 0x3a, 0x3b,
				0x3b, 0x36, 0x36, 0x37, 0x37, 0x37, 0x38, 0x38, 0x38, 0x39,
				0x39, 0x39, 0x3a, 0x3a, 0x3a, 0x3b, 0x3b, 0x3c, 0x3c, 0x3d,
				0x3d, 0x3d, 0x3e, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f, 0x40, 0x40,
				0x40, 0x41, 0x41, 0x3c, 0x3c, 0x3d, 0x3d, 0x3d, 0x3e, 0x3e,
				0x3e, 0x3f, 0x3f, 0x3f, 0x40, 0x40, 0x40, 0x41, 0x41, 0x3c,
				0x3c, 0x3d, 0x3d, 0x3d, 0x3e, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f,
				0x40, 0x40, 0x40, 0x41, 0x41, 0x42, 0x42, 0x43, 0x43, 0x43,
				0x44, 0x44, 0x44, 0x45, 0x45, 0x45, 0x46, 0x46, 0x46, 0x47,
				0x47, 0x42, 0x42, 0x43, 0x43, 0x43, 0x44, 0x44, 0x44, 0x45,
				0x45, 0x45, 0x46, 0x46, 0x46, 0x47, 0x47, 0x7e, 0x7e, 0x49,
				0x49, 0x49, 0x4a, 0x4a, 0x4a, 0x4b, 0x4b, 0x4b, 0x4c, 0x4c,
				0x4c, 0x4d, 0x4d, 0x7e, 0x7e, 0x49, 0x49, 0x49, 0x4a, 0x4a,
				0x4a, 0x4b, 0x4b, 0x4b, 0x4c, 0x4c, 0x4c, 0x4d, 0x4d, 0x4e,
				0x4e, 0x4f, 0x4f, 0x4f, 0x50, 0x50, 0x50, 0x51, 0x51, 0x51,
				0x52, 0x52, 0x52, 0x53, 0x53, 0x4e, 0x4e, 0x4f, 0x4f, 0x4f,
				0x50, 0x50, 0x50, 0x51, 0x51, 0x51, 0x52, 0x52, 0x52, 0x53,
				0x53, 0x4e, 0x4e, 0x4f, 0x4f, 0x6e, 0x6f, 0x50, 0x50, 0x51,
				0x51, 0x51, 0x52, 0x52, 0x52, 0x53, 0x53, 0x54, 0x54, 0x55,
				0x55, 0x6f, 0x6f, 0x6f, 0x6f, 0x57, 0x57, 0x57, 0x58, 0x58,
				0x58, 0x59, 0x59, 0x54, 0x54, 0x55, 0x55, 0x55, 0x6f, 0x56,
				0x56, 0x57, 0x57, 0x57, 0x58, 0x58, 0x58, 0x59, 0x59, 0x54,
				0x54, 0x55, 0x55, 0x55, 0x6f, 0x56, 0x70, 0x57, 0x57, 0x57,
				0x58, 0x58, 0x58, 0x59, 0x59, 0x5a, 0x5a, 0x5b, 0x5b, 0x5b,
				0x5c, 0x5c, 0x5c, 0x5d, 0x5d, 0x5d, 0x5e, 0x5e, 0x5e, 0x5f,
				0x5f, 0x5a, 0x5a, 0x5b, 0x5b, 0x5b, 0x5c, 0x5c, 0x5c, 0x5d,
				0x5d, 0x5d, 0x5e, 0x5e, 0x5e, 0x5f, 0x5f, 0x5a, 0x5a, 0x5b,
				0x5b, 0x5b, 0x5c, 0x5c, 0x5c, 0x5d, 0x5d, 0x5d, 0x5e, 0x5e,
				0x5e, 0x5f, 0x5f, 0x60, 0x60, 0x61, 0x61, 0x61, 0x62, 0x62,
				0x62, 0x63, 0x63, 0x63, 0x64, 0x64, 0x64, 0x65, 0x65, 0x60,
				0x60, 0x61, 0x61, 0x61, 0x62, 0x62, 0x62, 0x63, 0x63, 0x63,
				0x64, 0x64, 0x64, 0x65, 0x65, 0x60, 0x60, 0x61, 0x61, 0x61,
				0x62, 0x62, 0x62, 0x63, 0x63, 0x63, 0x64, 0x64, 0x64, 0x65,
				0x65, 0x66, 0x66, 0x67, 0x67, 0x67, 0x68, 0x68, 0x68, 0x69,
				0x69, 0x69, 0x6a, 0x6a, 0x6a, 0x6b, 0x6b, 0x66, 0x66, 0x67,
				0x67, 0x67, 0x68, 0x68, 0x68, 0x69, 0x69, 0x69, 0x6a, 0x6a,
				0x6a, 0x6b, 0x6b, 0x48, 0x48, 0x49, 0x49, 0x49, 0x4a, 0x4a,
				0x4a, 0x4b, 0x4b, 0x4b, 0x4c, 0x4c, 0x4c, 0x4d, 0x4d, 0x48,
				0x48, 0x49, 0x49, 0x49, 0x4a, 0x4a, 0x4a, 0x4b, 0x4b, 0x4b,
				0x4c, 0x4c, 0x4c, 0x4d, 0x4d, 0x4e, 0x4e, 0x4f, 0x4f, 0x4f,
				0x50, 0x50, 0x50, 0x51, 0x51, 0x51, 0x52, 0x52, 0x52, 0x53,
				0x53, 0x4e, 0x4e, 0x4f, 0x4f, 0x4f, 0x50, 0x50, 0x50, 0x51,
				0x51, 0x51, 0x52, 0x52, 0x52, 0x53, 0x53, 0x4e, 0x4e, 0x4f,
				0x4f, 0x4f, 0x50, 0x50, 0x50, 0x51, 0x51, 0x51, 0x52, 0x52,
				0x52, 0x53, 0x53, 0x54, 0x54, 0x55, 0x55, 0x55, 0x6f, 0x56,
				0x56, 0x57, 0x57, 0x57, 0x58, 0x58, 0x58, 0x59, 0x59, 0x54,
				0x54, 0x55, 0x55, 0x55, 0x56, 0x56, 0x56, 0x57, 0x57, 0x57,
				0x58, 0x58, 0x58, 0x59, 0x59, 0x54, 0x54, 0x55, 0x55, 0x55,
				0x56, 0x56, 0x70, 0x57, 0x57, 0x57, 0x58, 0x58, 0x58, 0x59,
				0x59, 0x5a, 0x5a, 0x5b, 0x5b, 0x5b, 0x5c, 0x5c, 0x5c, 0x5d,
				0x5d, 0x5d, 0x5e, 0x5e, 0x5e, 0x5f, 0x5f, 0x5a, 0x5a, 0x5b,
				0x5b, 0x5b, 0x5c, 0x5c, 0x5c, 0x5d, 0x5d, 0x5d, 0x5e, 0x5e,
				0x5e, 0x5f, 0x5f, 0x5a, 0x5a, 0x5b, 0x5b, 0x5b, 0x5c, 0x5c,
				0x5c, 0x5d, 0x5d, 0x5d, 0x5e, 0x5e, 0x5e, 0x5f, 0x5f, 0x60,
				0x60, 0x61, 0x61, 0x61, 0x62, 0x62, 0x62, 0x63, 0x63, 0x63,
				0x64, 0x64, 0x64, 0x65, 0x65, 0x60, 0x60, 0x61, 0x61, 0x61,
				0x62, 0x62, 0x62, 0x63, 0x63, 0x63, 0x64, 0x64, 0x64, 0x65,
				0x65, 0x60, 0x60, 0x61, 0x61, 0x61, 0x62, 0x62, 0x62, 0x63,
				0x63, 0x63, 0x64, 0x64, 0x64, 0x65, 0x65, 0x66, 0x66, 0x67,
				0x67, 0x67, 0x68, 0x68, 0x68, 0x69, 0x69, 0x69, 0x6a, 0x6a,
				0x6a, 0x6b, 0x6b, 0x66, 0x66, 0x67, 0x67, 0x67, 0x68, 0x68,
				0x68, 0x69, 0x69, 0x69, 0x6a, 0x6a, 0x6a, 0x6b, 0x6b, 0x7f,
				0x7f, 0x49, 0x49, 0x49, 0x4a, 0x4a, 0x4a, 0x4b, 0x4b, 0x4b,
				0x4c, 0x4c, 0x4c, 0x4d, 0x4d, 0x7f, 0x7f, 0x49, 0x49, 0x49,
				0x4a, 0x4a, 0x4a, 0x4b, 0x4b, 0x4b, 0x4c, 0x4c, 0x4c, 0x4d,
				0x4d, 0x4e, 0x4e, 0x4f, 0x4f, 0x4f, 0x50, 0x50, 0x50, 0x51,
				0x51, 0x51, 0x52, 0x52, 0x52, 0x53, 0x53, 0x4e, 0x4e, 0x4f,
				0x4f, 0x4f, 0x50, 0x50, 0x50, 0x51, 0x51, 0x51, 0x52, 0x52,
				0x52, 0x53, 0x53, 0x4e, 0x4e, 0x4f, 0x4f, 0x4f, 0x50, 0x50,
				0x50, 0x51, 0x51, 0x51, 0x52, 0x52, 0x52, 0x53, 0x53, 0x54,
				0x54, 0x55, 0x55, 0x55, 0x6f, 0x56, 0x70, 0x57, 0x57, 0x57,
				0x58, 0x58, 0x58, 0x59, 0x59, 0x54, 0x54, 0x55, 0x55, 0x55,
				0x56, 0x56, 0x70, 0x57, 0x57, 0x57, 0x58, 0x58, 0x58, 0x59,
				0x59, 0x54, 0x54, 0x55, 0x55, 0x55, 0x70, 0x70, 0x70, 0x70,
				0x57, 0x57, 0x58, 0x58, 0x58, 0x59, 0x59, 0x5a, 0x5a, 0x5b,
				0x5b, 0x5b, 0x5c, 0x5c, 0x70, 0x8f, 0x5d, 0x5d, 0x5e, 0x5e,
				0x5e, 0x5f, 0x5f, 0x5a, 0x5a, 0x5b, 0x5b, 0x5b, 0x5c, 0x5c,
				0x5c, 0x5d, 0x5d, 0x5d, 0x5e, 0x5e, 0x5e, 0x5f, 0x5f, 0x5a,
				0x5a, 0x5b, 0x5b, 0x5b, 0x5c, 0x5c, 0x5c, 0x5d, 0x5d, 0x5d,
				0x5e, 0x5e, 0x5e, 0x5f, 0x5f, 0x60, 0x60, 0x61, 0x61, 0x61,
				0x62, 0x62, 0x62, 0x63, 0x63, 0x63, 0x64, 0x64, 0x64, 0x65,
				0x65, 0x60, 0x60, 0x61, 0x61, 0x61, 0x62, 0x62, 0x62, 0x63,
				0x63, 0x63, 0x64, 0x64, 0x64, 0x65, 0x65, 0x60, 0x60, 0x61,
				0x61, 0x61, 0x62, 0x62, 0x62, 0x63, 0x63, 0x63, 0x64, 0x64,
				0x64, 0x65, 0x65, 0x66, 0x66, 0x67, 0x67, 0x67, 0x68, 0x68,
				0x68, 0x69, 0x69, 0x69, 0x6a, 0x6a, 0x6a, 0x6b, 0x6b, 0x66,
				0x66, 0x67, 0x67, 0x67, 0x68, 0x68, 0x68, 0x69, 0x69, 0x69,
				0x6a, 0x6a, 0x6a, 0x6b, 0x6b, 0x80, 0x80, 0x80, 0x95, 0x95,
				0x96, 0x96, 0x96, 0x97, 0x97, 0x97, 0x98, 0x98, 0x98, 0x99,
				0x99, 0x80, 0x80, 0x80, 0x95, 0x95, 0x96, 0x96, 0x96, 0x97,
				0x97, 0x97, 0x98, 0x98, 0x98, 0x99, 0x99, 0x80, 0x80, 0x9b,
				0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e,
				0x9e, 0x9f, 0x9f, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c,
				0x9c, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0x9f, 0x9f, 0x9a,
				0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d,
				0x9e, 0x9e, 0x9e, 0x9f, 0x9f, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1,
				0x6f, 0xa2, 0x70, 0x8f, 0xa3, 0xa3, 0xa4, 0xa4, 0xa4, 0xa5,
				0xa5, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0x70, 0x8f,
				0xa3, 0xa3, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa0, 0xa0, 0xa1,
				0xa1, 0xa1, 0x70, 0x70, 0x70, 0x8f, 0x8f, 0x8f, 0xa4, 0xa4,
				0xa4, 0xa5, 0xa5, 0xa6, 0xa6, 0xa7, 0xa7, 0xa7, 0x8f, 0x8f,
				0x8f, 0x8f, 0x8f, 0x8f, 0x8f, 0xaa, 0xaa, 0xab, 0xab, 0xa6,
				0xa6, 0xa7, 0xa7, 0xa7, 0xa8, 0xa8, 0x8f, 0x8f, 0xa9, 0xa9,
				0xaa, 0xaa, 0xaa, 0xab, 0xab, 0xa6, 0xa6, 0xa7, 0xa7, 0xa7,
				0xa8, 0xa8, 0x8f, 0x8f, 0xa9, 0x90, 0x90, 0xaa, 0xaa, 0xab,
				0xab, 0xac, 0xac, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0x8f,
				0xaf, 0x90, 0x91, 0xb0, 0xb0, 0xb1, 0xb1, 0xac, 0xac, 0xad,
				0xad, 0xad, 0xae, 0xae, 0xae, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0,
				0xb0, 0xb1, 0xb1, 0xac, 0xac, 0xad, 0xad, 0xad, 0xae, 0xae,
				0xae, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1, 0xb2,
				0xb2, 0xb3, 0xb3, 0xb3, 0xb4, 0xb4, 0xb4, 0xb5, 0xb5, 0xb5,
				0xb6, 0xb6, 0xb6, 0xb7, 0xb7, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3,
				0xb4, 0xb4, 0xb4, 0xb5, 0xb5, 0xb5, 0xb6, 0xb6, 0xb6, 0xb7,
				0xb7, 0x94, 0x94, 0x95, 0x95, 0x95, 0x96, 0x96, 0x96, 0x97,
				0x97, 0x97, 0x98, 0x98, 0x98, 0x99, 0x99, 0x94, 0x94, 0x95,
				0x95, 0x95, 0x96, 0x96, 0x96, 0x97, 0x97, 0x97, 0x98, 0x98,
				0x98, 0x99, 0x99, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c,
				0x9c, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0x9f, 0x9f, 0x9a,
				0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d,
				0x9e, 0x9e, 0x9e, 0x9f, 0x9f, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b,
				0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0x9f,
				0x9f, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0xa2, 0xa3,
				0xa3, 0xa3, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa0, 0xa0, 0xa1,
				0xa1, 0xa1, 0xa2, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
				0xa4, 0xa5, 0xa5, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2,
				0x70, 0x8f, 0xa3, 0xa3, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa6,
				0xa6, 0xa7, 0xa7, 0xa7, 0xa8, 0xa8, 0x8f, 0x8f, 0xa9, 0xa9,
				0xaa, 0xaa, 0xaa, 0xab, 0xab, 0xa6, 0xa6, 0xa7, 0xa7, 0xa7,
				0xa8, 0xa8, 0xa8, 0xa9, 0xa9, 0xa9, 0xaa, 0xaa, 0xaa, 0xab,
				0xab, 0xa6, 0xa6, 0xa7, 0xa7, 0xa7, 0xa8, 0xa8, 0xa8, 0xa9,
				0xa9, 0x90, 0x90, 0xaa, 0xaa, 0xab, 0xab, 0xac, 0xac, 0xad,
				0xad, 0xad, 0xae, 0xae, 0xae, 0xaf, 0xaf, 0x90, 0x91, 0xb0,
				0xb0, 0xb1, 0xb1, 0xac, 0xac, 0xad, 0xad, 0xad, 0xae, 0xae,
				0xae, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1, 0xac,
				0xac, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xaf, 0xaf, 0xaf,
				0xb0, 0xb0, 0xb0, 0xb1, 0xb1, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3,
				0xb4, 0xb4, 0xb4, 0xb5, 0xb5, 0xb5, 0xb6, 0xb6, 0xb6, 0xb7,
				0xb7, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb4, 0xb4, 0xb4, 0xb5,
				0xb5, 0xb5, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7, 0x81, 0x81, 0x81,
				0x95, 0x95, 0x96, 0x96, 0x96, 0x97, 0x97, 0x97, 0x98, 0x98,
				0x98, 0x99, 0x99, 0x81, 0x81, 0x81, 0x95, 0x95, 0x96, 0x96,
				0x96, 0x97, 0x97, 0x97, 0x98, 0x98, 0x98, 0x99, 0x99, 0x81,
				0x81, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d,
				0x9e, 0x9e, 0x9e, 0x9f, 0x9f, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b,
				0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0x9f,
				0x9f, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9d,
				0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0x9f, 0x9f, 0xa0, 0xa0, 0xa1,
				0xa1, 0xa1, 0xa2, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
				0xa4, 0xa5, 0xa5, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2,
				0xa2, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa0,
				0xa0, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0x70, 0x8f, 0xa3, 0x90,
				0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa6, 0xa6, 0xa7, 0xa7, 0xa7,
				0xa8, 0xa8, 0x8f, 0x8f, 0xa9, 0x90, 0x90, 0xaa, 0xaa, 0xab,
				0xab, 0xa6, 0xa6, 0xa7, 0xa7, 0xa7, 0xa8, 0xa8, 0xa8, 0xa9,
				0xa9, 0x90, 0x90, 0xaa, 0xaa, 0xab, 0xab, 0xa6, 0xa6, 0xa7,
				0xa7, 0xa7, 0xa8, 0xa8, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
				0x90, 0xab, 0xab, 0xac, 0xac, 0xad, 0xad, 0xad, 0xae, 0xae,
				0xae, 0x90, 0x90, 0x90, 0x91, 0x91, 0x91, 0xb1, 0xb1, 0xac,
				0xac, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xaf, 0xaf, 0x90,
				0x91, 0xb0, 0xb0, 0xb1, 0xb1, 0xac, 0xac, 0xad, 0xad, 0xad,
				0xae, 0xae, 0xae, 0xaf, 0xaf, 0x90, 0x91, 0xb0, 0x92, 0xb1,
				0xb1, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb4, 0xb4, 0xb4, 0xb5,
				0xb5, 0xb5, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7, 0xb2, 0xb2, 0xb3,
				0xb3, 0xb3, 0xb4, 0xb4, 0xb4, 0xb5, 0xb5, 0xb5, 0xb6, 0xb6,
				0xb6, 0xb7, 0xb7, 0x82, 0x82, 0x82, 0xb9, 0xb9, 0xba, 0xba,
				0xba, 0xbb, 0xbb, 0xbb, 0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0x82,
				0x82, 0x82, 0xb9, 0xb9, 0xba, 0xba, 0xba, 0xbb, 0xbb, 0xbb,
				0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0x82, 0x82, 0xbf, 0xbf, 0xbf,
				0xc0, 0xc0, 0xc0, 0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xc3,
				0xc3, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc1,
				0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xbe, 0xbe, 0xbf,
				0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc1, 0xc1, 0xc1, 0xc2, 0xc2,
				0xc2, 0xc3, 0xc3, 0xc4, 0xc4, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6,
				0xc6, 0xc7, 0xc7, 0xc7, 0xc8, 0xc8, 0xc8, 0xc9, 0xc9, 0xc4,
				0xc4, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7,
				0xc8, 0xc8, 0xc8, 0xc9, 0xc9, 0xc4, 0xc4, 0xc5, 0xc5, 0xc5,
				0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc8, 0xc8, 0xc8, 0xc9,
				0xc9, 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0x8f,
				0xcd, 0x90, 0x91, 0xce, 0xce, 0xcf, 0xcf, 0xca, 0xca, 0xcb,
				0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0x90, 0x91, 0xce,
				0xce, 0xcf, 0xcf, 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc,
				0xcc, 0x90, 0x90, 0x90, 0x91, 0x91, 0x91, 0xcf, 0xcf, 0xd0,
				0xd0, 0xd1, 0xd1, 0xd1, 0xd2, 0xd2, 0xd2, 0x91, 0x91, 0x91,
				0x91, 0x91, 0x91, 0x91, 0xd5, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1,
				0xd2, 0xd2, 0xd2, 0xd3, 0xd3, 0x91, 0x91, 0xd4, 0xd4, 0xd5,
				0xd5, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd2, 0xd2, 0xd2, 0xd3,
				0xd3, 0x91, 0x91, 0xd4, 0x92, 0x92, 0xd5, 0xd6, 0xd6, 0xd7,
				0xd7, 0xd7, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9, 0x91, 0xda,
				0x92, 0x93, 0xdb, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8,
				0xd8, 0xd9, 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xdb, 0xdb, 0xb8,
				0xb8, 0xb9, 0xb9, 0xb9, 0xba, 0xba, 0xba, 0xbb, 0xbb, 0xbb,
				0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9,
				0xba, 0xba, 0xba, 0xbb, 0xbb, 0xbb, 0xbc, 0xbc, 0xbc, 0xbd,
				0xbd, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc1,
				0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xbe, 0xbe, 0xbf,
				0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc1, 0xc1, 0xc1, 0xc2, 0xc2,
				0xc2, 0xc3, 0xc3, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0,
				0xc0, 0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc4,
				0xc4, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7,
				0xc8, 0xc8, 0xc8, 0xc9, 0xc9, 0xc4, 0xc4, 0xc5, 0xc5, 0xc5,
				0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc8, 0xc8, 0xc8, 0xc9,
				0xc9, 0xc4, 0xc4, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc7,
				0xc7, 0xc7, 0xc8, 0xc8, 0xc8, 0xc9, 0xc9, 0xca, 0xca, 0xcb,
				0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0xcd, 0xce, 0xce,
				0xce, 0xcf, 0xcf, 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc,
				0xcc, 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xca,
				0xca, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0x90,
				0x91, 0xce, 0xce, 0xcf, 0xcf, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1,
				0xd2, 0xd2, 0xd2, 0xd3, 0xd3, 0x91, 0x91, 0xd4, 0xd4, 0xd5,
				0xd5, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd2, 0xd2, 0xd2, 0xd3,
				0xd3, 0xd3, 0xd4, 0xd4, 0xd4, 0xd5, 0xd5, 0xd0, 0xd0, 0xd1,
				0xd1, 0xd1, 0xd2, 0xd2, 0xd2, 0xd3, 0xd3, 0xd3, 0xd4, 0xd4,
				0x92, 0x92, 0xd5, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8,
				0xd8, 0xd9, 0xd9, 0xd9, 0xda, 0xda, 0x92, 0x93, 0xdb, 0xd6,
				0xd6, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9,
				0xda, 0xda, 0xda, 0xdb, 0xdb, 0x83, 0x83, 0x83, 0xb9, 0xb9,
				0xba, 0xba, 0xba, 0xbb, 0xbb, 0xbb, 0xbc, 0xbc, 0xbc, 0xbd,
				0xbd, 0x83, 0x83, 0x83, 0xb9, 0xb9, 0xba, 0xba, 0xba, 0xbb,
				0xbb, 0xbb, 0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0x83, 0x83, 0xbf,
				0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc1, 0xc1, 0xc1, 0xc2, 0xc2,
				0xc2, 0xc3, 0xc3, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0,
				0xc0, 0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xbe,
				0xbe, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc1, 0xc1, 0xc1,
				0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc4, 0xc4, 0xc5, 0xc5, 0xc5,
				0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc8, 0xc8, 0xc8, 0xc9,
				0xc9, 0xc4, 0xc4, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc7,
				0xc7, 0xc7, 0xc8, 0xc8, 0xc8, 0xc9, 0xc9, 0xc4, 0xc4, 0xc5,
				0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc8, 0xc8,
				0xc8, 0xc9, 0xc9, 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc,
				0xcc, 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xca,
				0xca, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0xcd,
				0xce, 0xce, 0xce, 0xcf, 0xcf, 0xca, 0xca, 0xcb, 0xcb, 0xcb,
				0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0x90, 0x91, 0xce, 0x92, 0xcf,
				0xcf, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd2, 0xd2, 0xd2, 0xd3,
				0xd3, 0x91, 0x91, 0xd4, 0x92, 0x92, 0xd5, 0xd0, 0xd0, 0xd1,
				0xd1, 0xd1, 0xd2, 0xd2, 0xd2, 0xd3, 0xd3, 0xd3, 0xd4, 0xd4,
				0x92, 0x92, 0xd5, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd2, 0xd2,
				0xd2, 0xd3, 0xd3, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0xd6,
				0xd6, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9,
				0x92, 0x92, 0x92, 0x93, 0x93, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7,
				0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9, 0xda, 0xda, 0x92, 0x93,
				0xdb, 0x84, 0x84, 0x84, 0xdd, 0xdd, 0xde, 0xde, 0xde, 0xdf,
				0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1, 0x84, 0x84, 0x84,
				0xdd, 0xdd, 0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0,
				0xe0, 0xe1, 0xe1, 0x84, 0x84, 0xe3, 0xe3, 0xe3, 0xe4, 0xe4,
				0xe4, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe2,
				0xe2, 0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5,
				0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3,
				0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe7,
				0xe7, 0xe8, 0xe8, 0xe9, 0xe9, 0xe9, 0xea, 0xea, 0xea, 0xeb,
				0xeb, 0xeb, 0xec, 0xec, 0xec, 0xed, 0xed, 0xe8, 0xe8, 0xe9,
				0xe9, 0xe9, 0xea, 0xea, 0xea, 0xeb, 0xeb, 0xeb, 0xec, 0xec,
				0xec, 0xed, 0xed, 0xe8, 0xe8, 0xe9, 0xe9, 0xe9, 0xea, 0xea,
				0xea, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xed, 0xed, 0xee,
				0xee, 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1,
				0xf2, 0xf2, 0xf2, 0xf3, 0xf3, 0xee, 0xee, 0xef, 0xef, 0xef,
				0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf2, 0xf2, 0xf2, 0xf3,
				0xf3, 0xee, 0xee, 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0, 0xf1,
				0xf1, 0xf1, 0xf2, 0xf2, 0xf2, 0xf3, 0xf3, 0xf4, 0xf4, 0xf5,
				0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0x91, 0xf8,
				0x92, 0x93, 0xf9, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6,
				0xf6, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0x92, 0x93, 0xf9, 0xf4,
				0xf4, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf7,
				0x92, 0x92, 0x92, 0x93, 0x93, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb,
				0xfc, 0xfc, 0xfc, 0xfd, 0xfd, 0xfd, 0x93, 0x93, 0x93, 0x93,
				0x93, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfc, 0xfc, 0xfc, 0xfd,
				0xfd, 0xfd, 0xfe, 0xfe, 0x93, 0x93, 0xff, 0xdc, 0xdc, 0xdd,
				0xdd, 0xdd, 0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0,
				0xe0, 0xe1, 0xe1, 0xdc, 0xdc, 0xdd, 0xdd, 0xdd, 0xde, 0xde,
				0xde, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1, 0xe2,
				0xe2, 0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5,
				0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3,
				0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe7,
				0xe7, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe5,
				0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe8, 0xe8, 0xe9,
				0xe9, 0xe9, 0xea, 0xea, 0xea, 0xeb, 0xeb, 0xeb, 0xec, 0xec,
				0xec, 0xed, 0xed, 0xe8, 0xe8, 0xe9, 0xe9, 0xe9, 0xea, 0xea,
				0xea, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xed, 0xed, 0xe8,
				0xe8, 0xe9, 0xe9, 0xe9, 0xea, 0xea, 0xea, 0xeb, 0xeb, 0xeb,
				0xec, 0xec, 0xec, 0xed, 0xed, 0xee, 0xee, 0xef, 0xef, 0xef,
				0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf2, 0xf2, 0xf2, 0xf3,
				0xf3, 0xee, 0xee, 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0, 0xf1,
				0xf1, 0xf1, 0xf2, 0xf2, 0xf2, 0xf3, 0xf3, 0xee, 0xee, 0xef,
				0xef, 0xef, 0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf2, 0xf2,
				0xf2, 0xf3, 0xf3, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6,
				0xf6, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf4,
				0xf4, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf7,
				0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5,
				0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0x92, 0x93,
				0xf9, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfc, 0xfc, 0xfc, 0xfd,
				0xfd, 0xfd, 0xfe, 0xfe, 0x93, 0x93, 0xff, 0xfa, 0xfa, 0xfb,
				0xfb, 0xfb, 0xfc, 0xfc, 0xfc, 0xfd, 0xfd, 0xfd, 0xfe, 0xfe,
				0xfe, 0xff, 0xff };

		static byte[] color256InversePalette;
		static RGB[] color256Palette;

		int r = 255, g = 255, b = 255, s = 0;

		public RGB() {
		}

		public RGB(int v) {
			r = v & 0xff;
			g = (v >> 8) & 0xff;
			b = (v >> 16) & 0xff;
		}

		public RGB(int r, int g, int b) {
			this.r = r;
			this.g = g;
			this.b = b;
		}

		public int Difference(RGB col) {
			return Math.abs(r - col.r) + Math.abs(g - col.g)
					+ Math.abs(b - col.b);

		}

		public int Gray2() {
			return Gray256() / 128;
		}

		public int Gray4() {
			return Gray256() / 64;
		}

		public int Gray16() {
			return Gray256() / 16;
		}

		public int Gray256() {
			return (2 * r + 5 * g + b) / 8;
		}

		public int Color16() {
			int index = (r >> 5) & 0x007;
			index |= (g >> 2) & 0x038;
			index |= (b << 1) & 0x1c0;
			return color16inverse[index];
		}

		public int Color256() {
			int index = (r >> 4) & 0x00f;
			index |= g & 0x0f0;
			index |= (b << 4) & 0xf00;
			if (color256InversePalette != null) {
				return color256InversePalette[index];
			}
			return color256inverse[index];
		}

		public int Color4k() {
			return ((r & 0xf0) << 4) | (g & 0xf0) | ((b & 0xf0) >> 4);
		}

		public int Color64k() {
			return ((r & 0xf8) << 8) | ((g & 0xfc) << 3) | ((b & 0xf8) >> 3);
		}

		public int Color16M() {
			return (r << 16) | (g << 8) | b;
		}

		public static RGB Gray2(int gray) {
			gray *= 255;
			return new RGB(gray, gray, gray);
		}

		public static RGB Gray4(int gray) {
			gray *= 85;
			return new RGB(gray, gray, gray);
		}

		public static RGB Gray16(int gray) {
			gray *= 17;
			return new RGB(gray, gray, gray);
		}

		public static RGB Gray256(int gray) {
			return new RGB(gray, gray, gray);
		}

		public static RGB Color16(int color) {
			return new RGB(color16array[color & 0xf]);
		}

		public static RGB Color256(int color) {
			if (color256Palette != null) {
				return color256Palette[color & 0xff];
			}
			return new RGB(color256array[color & 0xff]);
		}

		public static RGB Color4K(int color) {
			return new RGB(((color >> 8) & 0xf) * 17,
					((color >> 4) & 0xf) * 17, (color & 0xf) * 17);
		}

		public static RGB Color64K(int color) {
			return new RGB(((color >> 11) & 0x1f) * 255 / 31,
					((color >> 5) & 0x3f) * 255 / 63, (color & 0x1f) * 255 / 31);
		}

		public static RGB Color16M(int color) {
			return new RGB((color >> 16) & 0xff, (color >> 8) & 0xff,
					color & 0xff);
		}
	}

	/**
	 * Verifies whether the given input stream contains valid bitmap data that
	 * is recognized by the symbian platform.
	 * 
	 * @param in the input stream from where the bitmap data is read.
	 * @return true if the bitmap is valid symbian bitmap, false otherwise.
	 * @throws IOException if I/O error occurred while processing the stream.
	 */
	public static boolean verifyBitmap(InputStream in) throws IOException {
		try {
			int fileSize = BitmapFileHeader.size();
			BitmapFileHeader header = BitmapFileHeader.read(in);

			int infoSize = BitmapInfoHeader.size();
			BitmapInfoHeader info = BitmapInfoHeader.read(in);

			if (info.biCompression != 0) {
				return false;
			}
			int size = header.bfSize - infoSize - fileSize;
			int bitcount = info.biBitCount;
			int colors = info.biClrUsed;
			int iNumBmpColors;
			if (colors == 0) {
				if (bitcount == 24) {
					iNumBmpColors = 0;
				} else {
					iNumBmpColors = 1 << bitcount;
				}
			} else {
				iNumBmpColors = colors;
			}
			if (iNumBmpColors > 256) {
				return false;
			}
			size -= iNumBmpColors * RgbQuad.size();
			skip(in, iNumBmpColors * RgbQuad.size());
			skip(in, size);
			return true;
		} finally {
			try {
				if (in != null) {
					in.close();
				}
			} catch (Exception e) {
			}
		}
	}

	public static boolean isBMPType(File image) {
		try {
			return verifyBitmap(new FileInputStream(image));
		} catch (Exception e) {
			return false;
		}
	}

	private static short readShort(InputStream in) throws IOException {
		byte[] buf = new byte[2];
		int read = 0;
		while (read < buf.length) {
			read += in.read(buf, read, buf.length - read);
		}
		int b1 = ((buf[1] & 0xff) << 8);
		int b2 = buf[0] & 0xff;
		return (short) (b1 | b2);
	}

	static class BitmapFileHeader {
		short bfType;
		int bfSize;
		short bfReserved1;
		short bfReserved2;
		int bfOffBits;

		public static int size() {
			return (3 * Short.SIZE + 2 * Integer.SIZE) / 8;
		}

		public static BitmapFileHeader read(InputStream in) throws IOException {
			BitmapFileHeader header = new BitmapFileHeader();
			header.bfType = readShort(in);
			header.bfSize = readInt(in);
			header.bfReserved1 = readShort(in);
			header.bfReserved2 = readShort(in);
			header.bfOffBits = readInt(in);
			return header;
		}

		void write(OutputStream out) throws IOException {
			writeData(bfType, out);
			writeData(bfSize, out);
			writeData(bfReserved1, out);
			writeData(bfReserved2, out);
			writeData(bfOffBits, out);
		}
	}

	static class BitmapInfoHeader {
		int biSize;
		int biWidth;
		int biHeight;
		short biPlanes;
		short biBitCount;
		int biCompression;
		int biSizeImage;
		int biXPelsPerMeter;
		int biYPelsPerMeter;
		int biClrUsed;
		int biClrImportant;

		public static int size() {
			return (2 * Short.SIZE + 10 * Integer.SIZE) / 8;
		}

		public static BitmapInfoHeader read(InputStream in) throws IOException {
			BitmapInfoHeader header = new BitmapInfoHeader();
			header.biSize = readInt(in);
			header.biWidth = readInt(in);
			header.biHeight = readInt(in);
			header.biPlanes = readShort(in);
			header.biBitCount = readShort(in);
			header.biCompression = readInt(in);
			header.biSizeImage = readInt(in);
			header.biXPelsPerMeter = readInt(in);
			header.biYPelsPerMeter = readInt(in);
			header.biClrUsed = readInt(in);
			header.biClrImportant = readInt(in);
			return header;
		}

		void write(OutputStream out) throws IOException {
			writeData(biSize, out);
			writeData(biWidth, out);
			writeData(biHeight, out);
			writeData(biPlanes, out);
			writeData(biBitCount, out);
			writeData(biCompression, out);
			writeData(biSizeImage, out);
			writeData(biXPelsPerMeter, out);
			writeData(biYPelsPerMeter, out);
			writeData(biClrUsed, out);
			writeData(biClrImportant, out);
		}
	}

	static class RgbQuad {
		byte iBlue;
		byte iGreen;
		byte iRed;
		byte iReserved;

		public static int size() {
			return 4;
		}
	}
}
