/*
* Copyright (c) 2007-2008 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:   items order manager class
*
*/


#ifndef HNITEMSORDER_H_
#define HNITEMSORDER_H_

#include <e32base.h>
#include <e32cmn.h>

/**
 * Items order manager.
 * 
 * Instances of this class control order of the items 
 * displayed within a suite. The keep and manage the
 * complex relation between CHnMdItem, and consequenltly 
 * created CHnItemModels.
 * 
 * @lib hnpresentationmodel
 * @since S60 5.0
 * @ingroup group_hnpresentationmodel
 */
NONSHARABLE_CLASS( CHnItemsOrder ) : public CBase
    {
    
public:
    
    /**
     * Two-phased constructor.
     * 
     * @since S60 5.0
     * @return Fully constructed object.
     */
    IMPORT_C static CHnItemsOrder* NewL();
    
    /**
     * Two-phased constructor.
     * 
     * @since S60 5.0
     * @return Fully constructed object.
     */
    IMPORT_C static CHnItemsOrder* NewLC();

    /**
     * Standard destructor.
     * 
     * @since S60 5.0
     */
    ~CHnItemsOrder();
        
    /**
     * @since S60 5.0
     * @param aId Suite id, should be generated by IdGenrator
     */
    IMPORT_C void SetSuiteId( TInt aId );
    
    /**
     * @since S60 5.0
     * @return id of the associated suite 
     */
    IMPORT_C TInt GetSuiteId() const;
    
    /**
     * Removes item with given id.
     * 
     * @since S60 5.0
     * @param aId id of the item to be removed
     */
    IMPORT_C void RemoveItemId( TInt aId );
    
    /**
     * Insert an item referred to by an aId, to a suite.
     * It is indicated that this item has been produced from
     * CHnMdItem at index aDataModelItem. Evaluation indicates also
     * that this item is produced on the index aIndex in that particular
     * CHnMdItem.
     * 
     * @since S60 5.0
     * @param aDataModelItem Index of CHnMdItem instance in the CHnMdSuite.
     * @param aIndex Position in that CHnMdItem instance
     * @param aId Id of the item model.
     */
    IMPORT_C void InsertItemIdToL( TInt aDataModelItem, TInt aIndex, TInt aId );
    
    /**
     * Gets the item id at position aIndex, relative to the beginning
     * of the items generated from CHnMdItem at index aDataModelItem
     * 
     * @since S60 5.0
     * @param aDataModelItem Index of CHnMdItem instance in the CHnMdSuite.
     * @param aIndex Position in that CHnMdItem instance.
     */
    IMPORT_C TInt GetItemIdAt( TInt aDataModelItem, TInt aIndex );
    
    /**
     * Gets item id at specified position.
     * 
     * @since S60 5.0
     * @param aPosition Position in the suite model. 
     * @return Item id
     */
    IMPORT_C TInt GetItemIdAt( TInt aPosition );
    
    /**
     * Sets item id at given position.
     * 
     * @since S60 5.0
     * @param aPosition Position in the suite model
     * @param aId Item id
     */
    IMPORT_C void SetItemIdAtL( TInt aPosition, TInt aId );
    
    /**
     * Gets item position.
     * 
     * @since S60 5.0
     * @param aId Item id
     * @return Item position.
     */
    IMPORT_C TInt GetItemPosition( TInt aId );
    
    /**
     * Count property getter.
     * 
     * @since S60 5.0
     * @return Count
     */
    IMPORT_C TInt Count();
    
    /**
     * Informs the class of the count of CHnMdItem instances
     * in the suite.
     * 
     * @since S60 5.0
     * @param aItemCounts Number of CHnMdItems in the suite.
     */
    IMPORT_C void DefineModelItemCountL( TInt aItemCounts );
    
    /**
     * Gets the count of CHnMdItem instances
     * in the suite.
     * 
     * @since S60 5.0
     */
    IMPORT_C TInt GetModelItemCount();
    
    /**
     * Gets the ids for a particular CHnMdItem instance.
     * 
     * @since S60 5.0
     * @param aDataModelItem Index of the CHnMdItem instance
     * @param aIds Array of ids.
     */
    IMPORT_C void GetIdsForL( TInt aDataModelItem, 
            RArray< TInt >& aIds );
    
    /**
     * Operator to access id at a particular position in the suite.
     * 
     * @since S60 5.0
     * @param aPosition Position in the suite.
     */
    IMPORT_C TInt operator[]( TInt aPosition );
    
    /**
     * Informs the instance of the class that the associated suite
     * has been evaluated.
     * 
     * @since S60 5.0
     */
    IMPORT_C void SuiteEvaluated();
    
    /**
     * Informs the instance of the class that the associated item
     * has been evaluated.
     * 
     * @since S60 5.0
     * @param aIndex Item of the model that has been evaluated.
     */
    IMPORT_C void ItemEvaluated( TInt aIndex );
    
    /**
     * Determined if the suite is ready to be shown.
     * 
     * @since S60 5.0
     * @return Value determinig if associated suite is ready to show.
     */
    IMPORT_C TBool IsSuiteReadyToShow();
    
    /**
     * Marks suite as uninitialized.
     * 
     */
    IMPORT_C void MarkSuiteUninitialized();
    
private:
    
    /**
     * Default constructor.
     * 
     * @since S60 5.0
     */
    CHnItemsOrder();
    
    /**
     * Stnadard 2nd stage symbian constructor.
     * 
     * @since S60 5.0
     */
    void ConstructL();
    
    /**
     * @since S60 5.0
     * @param aDataModelItem data model item
     * @return start index
     */
    TInt CountStartIndexFor( TInt aDataModelItem );
    
    /**
     * @since S60 5.0
     * @param aIndex index
     * @return model number
     */
    TInt CountModelNumberFor( TInt aIndex );
    
private: // data
    
    /**
     * Unique id for the associated suite. 
     * Should be generated via IdGenerator mechanism.
     */ 
    TInt iSuiteId;

    /**
     * Indicates if the associated suite has been initialized
     * with the first evaluation.
     */
    TBool iSuiteEvaluated;

    /**
     * Array of values indication if each of associated CHnMdItems
     * has been initialized with first evaluation.
     */
    RArray<TBool> iItemsEvaluated;
    
    /**
     * Items order counts
     * Holds the counts of CHnItemModels, for each of CHnMdItems
     */
    RArray<TInt> iItemsOrderCounts;

    /**
     * Complex items order.
     * Hold the order of CHnItemModels in a suite associated with the
     * instance of this class.
     */
    RArray<TInt> iComplexItemsOrder;
    
    };

#endif // HNITEMSORDER_H_
