/*
* Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
*
*/


/**
 @file ROMMANAGER.H
 @internalComponent
 @released
*/
#ifndef	ROMMANAGER_H
#define	ROMMANAGER_H

// System includes
#include <istream>
#include <list>
#include <vector>
#include <map>

// User includes
#include "symbiantypes.h"
#include "parameterlist.h" 

// Classes referenced
class Options2;
struct SBinarySecurityInfo;

/**
 * @publishedPartner
 * @released
 */
class RomManagerException 
    {
public: // Enumerations
    enum TType
        {
        ETypeUnableToOpenCorrespondingOBY = 0,
        ETypeUnableToFindCorrespondingOBY,
        ETypeUnableToOpenLogFile		
        };

public:
    inline RomManagerException( TType aType, const std::string& aFileName )
    :   iType( aType ), iFileName( aFileName )
        {
        }

public:
    void Display() const;

private:
    TType iType;
    const std::string iFileName;
    };


/**
 * Abstract class to access ROM file details (file presence and security info).
 * 
 * @internalComponent
 * @released
 */
class RomManager
	{
public:
	// factory method to create derived class objects based on the user options  
    static RomManager* New(const CParameterList& aParamList);
    
protected:
    RomManager();

public:
	// checks the specified file presense in the ROM
    virtual bool RomFileExists( const std::wstring& aFileName ) = 0;
	
	// returns security info of the file, if it is executable (zero returns)
    virtual TInt ReadSecurityInfo( SBinarySecurityInfo& aInfo, const std::wstring aFileName ) = 0;

	// searches for all variants of an adorned file name
	virtual void FindAllAdornedVariants(const std::wstring& aSearchNameWild, std::list<std::wstring>& aAdornedFileNamesFound) = 0;
    
	// checks to see if the SID passed is owned by a binary in ROM
	virtual bool SidExistsInRom(std::wstring& aFile, TUint32 aSid) = 0;
	};

/**
 * @internalComponent
 * @released
 */
class RomEntry
    {
public: // Enumerations
    enum TType
        {
        ETypeUndefined = 0,
        ETypeBinary,
        ETypeData
        };

public: // Constructors & destructor
    RomEntry( const std::string& aSourceLogFile );
    ~RomEntry();

public: // Access
    inline TType Type() const { return iType; }
    inline void SetType( TType aType ) { iType = aType; }

    inline std::string EnvFileName() const { return iEnvFileName; }
    inline void SetEnvFileName( std::string aFileName ) { iEnvFileName = aFileName; }

    inline std::string RomFileName() const { return iRomFileName; }
    inline void SetRomFileName( std::string aFileName ) { iRomFileName = aFileName; }

    inline TUint32 Uid1() const { return iUid1; }
    inline void SetUid1( TUint32 aUid ) { iUid1 = aUid; }

    inline TUint32 Uid2() const { return iUid2; }
    inline void SetUid2( TUint32 aUid ) { iUid2 = aUid; }

    inline TUint32 Uid3() const { return iUid3; }
    inline void SetUid3( TUint32 aUid ) { iUid3 = aUid; }

    inline SBinarySecurityInfo& SecurityInfo() { return *iSecInfo; }
    inline const SBinarySecurityInfo& SecurityInfo() const { return *iSecInfo; }

private: // Data members
    const std::string& iSourceLogFile;
    TType iType;
    std::string iEnvFileName;
    std::string iRomFileName;
	TUint32 iUid1;
	TUint32 iUid2;
	TUint32 iUid3;
	SBinarySecurityInfo* iSecInfo;
    };


// This class reads ROM drive details from the file system path provided.
class RomManagerFileSystem : public RomManager
    {
public:
    RomManagerFileSystem( const std::wstring& aPath );

public: // From RomManager
    bool RomFileExists( const std::wstring& aFileName );
    TInt ReadSecurityInfo( SBinarySecurityInfo& aInfo, const std::wstring aFileName );
	void FindAllAdornedVariants(const std::wstring& aSearchNameWild, std::list<std::wstring>& aAdornedFileNamesFound);
	bool SidExistsInRom(std::wstring& aFile, TUint32 aSid);

private:
    std::wstring iRomPath;
    };

// This class reads ROM drive details from the .log & .oby files generated by rombuild tools.
class RomManagerLogFiles : public RomManager
    {
public:
    RomManagerLogFiles( const std::list<std::wstring>& aLogFileNames );
    ~RomManagerLogFiles();

public: // From RomManager
    bool RomFileExists( const std::wstring& aFileName );
    TInt ReadSecurityInfo( SBinarySecurityInfo& aInfo, const std::wstring aFileName );
	void FindAllAdornedVariants(const std::wstring& aSearchNameWild, std::list<std::wstring>& aAdornedFileNamesFound);
	bool SidExistsInRom(std::wstring& aFile, TUint32 aSid);

private: // Internal methods
    void ReadObyFile( const std::string& aFileName );
    void ProcessObyFileLine( const std::string& aFileName, std::string& aLine );
    //
    bool ReadLogFile( const std::string& aFileName );
    void ProcessLogFileLine( std::string& aLine, RomEntry*& aCurrentEntry, const std::string& aFileName, bool& aObyParsingRequired );

private: // Type definitions
    typedef std::map< std::string /* file name */, RomEntry* /* associated data */ > RomEntryMap;

private: // Data members
    RomEntryMap iEntriesIndexedByEnvFileName;
    RomEntryMap iEntriesIndexedByRomFileName;
    std::list<std::wstring> iLogFileNames;
    };

// This class is empty holder, used when user selects Z drive eclipsing checks disable option
class RomManagerEmpty : public RomManager
    {

public: // From RomManager
    bool RomFileExists( const std::wstring& aFileName );
    TInt ReadSecurityInfo( SBinarySecurityInfo& aInfo, const std::wstring aFileName );
	void FindAllAdornedVariants(const std::wstring& aSearchNameWild, std::list<std::wstring>& aAdornedFileNamesFound);
	bool SidExistsInRom(std::wstring& aFile,TUint32 aSid);

    };

#endif
