/*
* Copyright (c) 2004 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Implementation.
*
*/

#include "TTcCommandBase.h"
#include "TTcIdentifier.h"
#include "TcLog.h"

_LIT8( KTcOctetStringData, "data" );


EXPORT_C TTcCommandBase::TTcCommandBase( CTcContextBase& aContext )
	: iContextBase( aContext )
	{
	}

EXPORT_C TTcCommandBase::~TTcCommandBase()
	{
	}

//
// - Command name matching functions ------------------------------------------
//

EXPORT_C TBool TTcCommandBase::Match( const TTcIdentifier& aId,
									  const TDesC8& aFunction )
	{
	if( aId.iFunction.MatchF( aFunction ) != KErrNotFound )
		{
		return ETrue;
		}
	return EFalse;
	}

EXPORT_C TBool TTcCommandBase::Match( const TTcIdentifier& aId,
									  const TDesC8& aContainer,
									  const TDesC8& aFunction )
	{
	if( ( aId.iFunction.MatchF( aFunction ) != KErrNotFound ) &&
		( aId.iContainer.MatchF( aContainer ) != KErrNotFound ) )
		{
		return ETrue;
		}
	return EFalse;
	}

//
// - Response creation functions ----------------------------------------------
//

EXPORT_C void TTcCommandBase::AddIntegerResponseL( const TDesC8& aName,
												   TInt aValue )
	{
	CTcNameValue* item = CTcNameValue::NewLC();
	item->SetL( aName, aValue );
	iContextBase.ReturnList().AddParameterL( item );
	CleanupStack::Pop( item );
	}

EXPORT_C void TTcCommandBase::AddBooleanResponseL( const TDesC8& aName,
												   TBool aValue )
	{
	const TDesC8* value = aValue ? &KTcTrue : &KTcFalse;
	AddTextResponseL( aName, *value );
	}

EXPORT_C void TTcCommandBase::AddTextResponseL( const TDesC8& aName,
												const TDesC8& aValue )
	{
	if( aValue != KNullDesC8 )
		{
		CTcNameValue* item = CTcNameValue::NewLC();
		item->SetL( aName, aValue );
		iContextBase.ReturnList().AddParameterL( item );
		CleanupStack::Pop( item );
		}
	}

EXPORT_C void TTcCommandBase::AddArrayResponseL( const TDesC8& aName,
												 const MDesC8Array& aValues )
	{
	CTcArray* array = CTcArray::NewLC();
	array->SetName( aName );
	array->AddItemsL( aValues );
	iContextBase.ReturnList().AddParameterL( array );
	CleanupStack::Pop( array );
	}

EXPORT_C CTcStructure* TTcCommandBase::AddStructureResponseL(
														const TDesC8& aName )
	{
	CTcStructure* structure = CTcStructure::NewLC();
	structure->SetName( aName );
	iContextBase.ReturnList().AddParameterL( structure );
	CleanupStack::Pop( structure );
	return structure;
	}

//
// -- SIP object / registry access functions ----------------------------------
//

EXPORT_C TAny* TTcCommandBase::GetObjectForIdL( const TDesC8& aName,
												TBool aMandatory ) const
	{
	TBuf8< KTcMaxTypeName > objectName( aName );
	TBuf8< KTcOptionalLength > optional; if( aMandatory ) optional = KTcMandatory; else optional = KTcOptional;
	TcLog::WriteFormat( _L8("-- Get %s object \"%s\""), optional.PtrZ(), objectName.PtrZ() );

	TPtrC8 idName = ExtractIdL( aName, aMandatory );
	if( idName != KNullDesC8 )
		{
		TcLog::Write( KLogOk );
		return iContextBase.Registry().ObjectPtrL( idName );
		}

	TcLog::Write( KLogNotFound );
	return NULL;
	}

//
// -- Middle-level parameter list wrapper functions ---------------------------
//

EXPORT_C TInt TTcCommandBase::ExtractIntegerL( const TDesC8& aName,
											   TInt aDefaultValue,
											   TBool aIsMandatory ) const
	{
	CTcNameValue* item = FindNameValueL( aName, aIsMandatory );
	if( item )
		{
		return item->ValueAsInt();
		}
	return aDefaultValue;
	}

EXPORT_C TBool TTcCommandBase::ExtractBooleanL( const TDesC8& aName,
												TBool aIsMandatory ) const
	{
	TPtrC8 value = ExtractTextL( aName, aIsMandatory );
	if( value == KTcTrue )
		{
		return ETrue;
		}
	return EFalse;
	}

EXPORT_C TPtrC8 TTcCommandBase::ExtractTextL( const TDesC8& aName,
											  TBool aIsMandatory ) const
	{
	// Check if we want to extract the actual data
	if ( aName.FindF( KTcOctetStringData ) == 0 )
		{
		CTcNameValue* item = FindNameValueL( KTcOctetStringData, aIsMandatory );
		if( item )
			{
			return item->Value();
			}
		}
	else
		{
		CTcNameValue* item = FindNameValueL( aName, aIsMandatory );
		if( item )
			{
			return item->Value();
			}
		}
	return TPtrC8();
	}

EXPORT_C TPtrC8 TTcCommandBase::ExtractIdL( const TDesC8& aName,
											TBool aIsMandatory ) const
	{
	// Do some nasty TRAPs in order to leave with a more informative
	// error code than what is generated by the called functions.
	CTcStructure* tcIds = NULL;
	CTcNameValue* tcId = NULL;
	TInt err = KErrNone;

	TRAP( err, tcIds = FindStructureL( KTcTestClientIds,
										  aIsMandatory ) );
	if( err )
		{
		User::Leave( KTcErrMandatoryIdNotFound );
		}

	TRAP( err, tcId = FindNameValueL( tcIds, aName, aIsMandatory ) );
	if( err )
		{
		User::Leave( KTcErrMandatoryIdNotFound );
		}

	if( tcId )
		{
		return tcId->Value();
		}
	return TPtrC8( KNullDesC8 );
	}

//
// -- Low-level parameter list wrapper functions ------------------------------
//

EXPORT_C CTcStructure* TTcCommandBase::FindStructureL( const TDesC8& aName,
													   TBool aMandatory ) const
	{
	TInt pos = iContextBase.List().FindParameter( aName, MTcTyped::EStructure );

	if( pos != KErrNotFound )
		{
		return &iContextBase.List().AsStructure( pos );
		}

	if( aMandatory )
		{
		TBuf8< KTcMaxTypeName > objectName( aName );
		TcLog::WriteFormat( _L8("-- Get mandatory structure \"%s\""), objectName.PtrZ() );
		TcLog::Write( KLogNotFound );
		User::Leave( KTcErrMandatoryParameterNotFound );
		}

	return NULL;
	}

EXPORT_C CTcArray* TTcCommandBase::FindArrayL( const TDesC8& aName,
											   TBool aMandatory ) const
	{
	TInt pos = iContextBase.List().FindParameter( aName, MTcTyped::EArray );

	if( pos != KErrNotFound )
		{
		return &iContextBase.List().AsArray( pos );
		}

	if( aMandatory )
		{
		TBuf8< KTcMaxTypeName > objectName( aName );
		TcLog::WriteFormat( _L8("-- Get mandatory array \"%s\""), objectName.PtrZ() );
		TcLog::Write( KLogNotFound );
		User::Leave( KTcErrMandatoryParameterNotFound );
		}

	return NULL;
	}

EXPORT_C CTcNameValue* TTcCommandBase::FindNameValueL(
													const TDesC8& aName,
													TBool aIsMandatory ) const
	{
	TBuf8< KTcMaxTypeName > objectName( aName );
	TBuf8< KTcOptionalLength > optional; if( aIsMandatory ) optional = KTcMandatory; else optional = KTcOptional;
	TcLog::WriteFormat( _L8("-- Get %s name-value \"%s\""), optional.PtrZ(), objectName.PtrZ() );

	TInt pos = iContextBase.List().FindParameter( aName, MTcTyped::ENameValue );
	if( pos != KErrNotFound )
		{
		TcLog::Write( KLogOk );
		return &iContextBase.List().AsNameValue( pos );
		}

	TcLog::Write( KLogNotFound );
	if( aIsMandatory )
		{
		User::Leave( KTcErrMandatoryParameterNotFound );
		}

	return NULL;
	}

EXPORT_C CTcNameValue* TTcCommandBase::FindNameValueL(
													CTcStructure* aStructure,
													const TDesC8& aName,
													TBool aIsMandatory ) const
	{
	if( !aStructure )
		{
		if( aIsMandatory )
			{
			TBuf8< KTcMaxTypeName > objectName( aName );
			TcLog::WriteFormat( _L8("-- Get mandatory name-value \"%s\" from structure [%d]"), objectName.PtrZ(), aStructure );
			User::Leave( KTcErrMandatoryParameterNotFound );
			}
		else
			{
			return NULL;
			}
		}

	return aStructure->ItemL( aName, aIsMandatory );
	}

EXPORT_C TBool TTcCommandBase::HasBooleanL( const TDesC8& aName ) const
    {
   	TPtrC8 value = ExtractTextL( aName, EFalse );
	if( value == KTcTrue || value == KTcFalse )
		{
		return ETrue;
		}
	return EFalse;
    }

// End of File
