/*
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of "Eclipse Public License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description:TMSGainEffect class
 *
 */

#ifndef TMS_GAIN_EFFECT_H
#define TMS_GAIN_EFFECT_H

#include <tms.h>
#include <tmseffect.h>

namespace TMS {

// FORWARD DECLARATIONS
class TMSGainEffectBody;
class TMSEffectObserver;

/**
 * TMSGainEffect class
 *
 * This class provides microphone gain level control in the stream scope.
 *
 * @lib tmsapi.lib
 *
 */
class TMSGainEffect : public TMSEffect
    {
public:
    /**
     * Destructor
     */
    IMPORT_C virtual ~TMSGainEffect();

    /**
     * Add client as an observer to events generated by this object.
     *
     * This function can be called at any time. It is recommended adding
     * observer before calling any functions on the Effect. Otherwise, the
     * observer may miss a callback.
     *
     * A reference to the obsrvr is stored in the effect. EffectsEvent callback
     * will be called when an effect event occurs. When the effect instance is
     * deleted, all TMSEffectObserver references are automatically released.
     * The ownership of the obsrvr is still maintained by the client. Should
     * the client need to delete the obsrvr before deleting the effect, it
     * should remove the obsrvr reference from the effect by calling
     * RemoveObserver() function.
     *
     * @param  obsrvr
     *      Listener to be added to the effect.
     *
     * @param  user_data
     *      Any user data passed to the function.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the obsrvr is successfully added to the list.
     *      TMS_RESULT_ALREADY_EXIST if the obsrvr is already in the list.
     *
     */
    IMPORT_C gint AddObserver(TMSEffectObserver& obsrvr, gpointer user_data);

    /**
     * Remove a client observer from this Effect.
     *
     * This function can be called at any time.
     *
     * @param  obsrvr
     *      The listener to remove.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the obsrvr is removed successfully from list.
     *      TMS_RESULT_DOES_NOT_EXIST if obsrvr is not already in the list.
     *
     */
    IMPORT_C gint RemoveObserver(TMSEffectObserver& obsrvr);

    /**
     * Set mic gain level. The gain level should not be higher than
     * the value returned by GetMaxLevel().
     *
     * @param  level
     *      Gain level to be set.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the operation was successful.
     *
     */
    IMPORT_C gint SetLevel(const guint level);

    /**
     * Return mic gain level.
     *
     * @param  level
     *      Current mic gain level.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the operation was successful.
     *
     */
    IMPORT_C gint GetLevel(guint& level);

    /**
     * Return max microphone gain level supported by the device.
     *
     * @param  level
     *      Max gain level.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the operation was successful.
     *
     */
    IMPORT_C gint GetMaxLevel(guint& level);

    /**
     * Returns type of the effect.
     *
     * This function can be called at any time.
     *
     * @param  effecttype
     *      Variable that contains the effect type on return.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the operation was successful.
     *
     */
    IMPORT_C virtual gint GetType(TMSEffectType& effecttype);

protected:
    /**
     * Constructor
     */
    IMPORT_C TMSGainEffect();

protected:
    TMSGainEffectBody *iBody;
    };

} //namespace TMS

#endif //TMS_GAIN_EFFECT_H
