/*
* Copyright (c) 2005-2007 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Report generator
*
*/

#ifndef C_REPORT_GENERATOR_H
#define C_REPORT_GENERATOR_H

#include "hidreportroot.h"
#include "hidtranslate.h"

/**
 * HID report generator
 *
 * Class used for generating output or feature reports to send to the device.
 *
 * @lib  generichid.lib
 * @since S60 v5.0
 */
class CReportGenerator : public CBase
    {
public:
    /**
     * Constructs a CReportGenerator.
     *
     * @since S60 v5.0
     * @param aReportRoot Tree generated by CParser from the report
     *                    descriptor.
     * @param aReportId Report ID of the report to generate.
     * @param aType Type of report to generate (output or feature).
     */
    IMPORT_C static CReportGenerator* NewLC(const CReportRoot* aReportRoot,
        TInt aReportId, CField::TType aType);

    /**
     * Constructs a CReportGenerator.
     *
     * @since S60 v5.0
     * @param aReportRoot Tree generated by CParser from the report
     *                    descriptor.
     * @param aReportId Report ID of the report to generate.
     * @param aType Type of report to generate (output or feature).
     */
    IMPORT_C static CReportGenerator* NewL(const CReportRoot* aReportRoot,
        TInt aReportId, CField::TType aType);

    virtual ~CReportGenerator();

    /**
     * Sets the value of a single field in the report.  Stores the requested
     * control value in a field in the report buffer.  For array fields, adds
     * the requested usage ID to the array.
     *
     * @since S60 v5.0
     * @param aField Pointer to the field in which to set a value.
     * @param aUsageId Usage ID of the control to set.
     * @param aValue Value to write to the field, used for variable
     *               fields only.
     * @param aControlOffset Which control to set when more than one
     *                       have the same usage ID.
     * @return Error code indicating success or reason for failure.
     */
    IMPORT_C TInt SetField(const CField* aField, TInt aUsageId, TInt aValue,
        TInt aControlOffset = 0);

    /**
     * Get the buffer holding the generated report
     *
     * @since S60 v5.0
     * @return pointer to buffer
     */
    IMPORT_C TPtr8 Report();

private:
    /**
     * Constructor for CReportGenerator.
     *
     * @since S60 v5.0
     * @param aReportId Report ID of the report to generate.
     * @param aType Type of report to generate (output or feature).
     */
    CReportGenerator(TInt aReportId, CField::TType aType);

    /**
     * 2nd phase constructor for CReportGenerator.  Allocates the
     * report buffer iBuf.
     *
     * @since S60 v5.0
     * @param aReportRoot Tree generated by CParser from the report
     *                    descriptor.
     */
    void ConstructL(const CReportRoot* aReportRoot);

    /**
     * Report ID of the report to generate
     */
    TInt iReportId;

    /**
     * Type of report to generate
     */
    CField::TType iType;

    /**
     * Tree of fields from the report descriptor
     * Not own.
     */
    const CReportRoot* iReportRoot;

    /**
     * Buffer to hold the generated report
     * Own.
     */
    HBufC8* iBuf;
    };

#endif  // __REPORT_GENERATOR_H__
