/**
 * Copyright (c) 2010 Sasken Communication Technologies Ltd.
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of the "Eclipse Public License v1.0"
 * which accompanies  this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html"
 *
 * Initial Contributors:
 * Chandradeep Gandhi, Sasken Communication Technologies Ltd - Initial contribution
 *
 * Contributors:
 * Manasij Roy, Nalina Hariharan
 *
 * Description:
 * The SmfEvent class represents an event
 *
 */

#ifndef SMFGALLERY_H
#define SMFGALLERY_H

#include "smfglobal.h"
#include "smfprovider.h"
#include "../common/SmfClientGlobal.h"
class SmfProvider;
class SmfComment; //user id, string, and url
class SmfPicture;


#include <QObject>
#include <QDateTime>
#include <QStringList>
//List of SmfPicture
typedef QList<SmfPicture> SmfPictureList;
/**
 * @ingroup smf_client_group 
 * Interface to a remote gallery service. This class
 * provides some basic gallery functionality to allow applications
 * to interact with a picture gallery in a social network.
 *
 * Note that branding information for the particular service implementation
 * is available from getProvider() API. See also:
 * SmfProvider::serviceName(), SmfProvider::serviceIcon()
 *
 * All of the functionality described here should be implemented by a service
 * specific plug-in object.
 * Interface name for SmfGallery org.symbian.smf.client.gallery
 */
class SMFCLIENT_EXPORT SmfGallery : public QObject
{
  Q_OBJECT

public:
  /**
   * Constructs SmfGallery.
   * @param baseProvider The base provider info
   * Seeing as this is a plug-in implementation, these will realistically
   be generated by SMF factory of some kind
   */

  SmfGallery(SmfProvider* baseprovider = 0);
  ~SmfGallery();

public:
  /**
   * Get the picture listing asynchronously.
   * The picturesAvailable() signal is emitted with SmfPictureList once the pictures have arrived.
   * When the list is big user can specify the page number and per page item data.
   * If not supplied by the user default values are used.
   * @param pageNum Page number to download, SMF_FIRST_PAGE denotes fresh query.
   * @param perPage Item per page, default is SMF_ITEMS_PER_PAGE
   */
  void pictures(int pageNum=SMF_FIRST_PAGE,int perPage=SMF_ITEMS_PER_PAGE);

  /**
   * Returns a user title/caption for the picture
   */
  QString description(SmfPicture& picture); // A user title or caption, maybe?

  //APIs to get/set base provider info (SmfProvider)

  /**
   * Gets the base provider info
   */
   SmfProvider* getProvider() ;

public slots:
	/**
	 * Upload an image.Implemented as slot to connect to UI controls more easily
	 * uploadFinished() signal is emitted with the success value of the upload
	 * @param image the image to be uploaded
	 */
   void upload(SmfPicture* image) ;

	/**
	 * Upload an list image.Implemented as slot to connect to UI controls more easily
	 * uploadFinished() signal is emitted with the success value of the upload
	 * @param images the list image to be uploaded
	 */
   void upload(SmfPictureList* images) ;

  /**
   * Posts a comment for an image. uploadFinished() signal is emitted
   * with success of the post once comment is posted.
   * @param image Image to comment on
   * @param comment Comment to post
   */
   void postComment(SmfPicture image, SmfComment comment) ;

signals:
	/*
	 * Notification on arrival of list of SmfPicture as a result of request.
	 * Note if number of friends is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 * through pictures().
	 * @param pics Picture list
	 * @param error Error string
	 * @param resultPage Page number info
	 */
	void picturesAvailable(SmfPictureList* pics, QString error, SmfResultPage resultPage);

  /**
   * Notification of the success of the uploading of image/comment
   * @param error The upload success result of each individual pictures
   */
  void uploadFinished(QList<bool> error);
private:
  SmfProvider* m_baseProvider;
};

SMF_SERVICE_NAME(SmfGallery, "org.symbian.smf.client.gallery\0.2")

#endif // SMFGALLERY_H

