/*
* Copyright (c) 2007 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:   Events generated by scrollers.
*
*/



#ifndef T_ALFSCROLLEREVENT_H
#define T_ALFSCROLLEREVENT_H

#include <e32std.h>
/**
 * ScrollerEvent class.
 * Usage:
 * @code
 * 
 * TBool CMyControl::OfferEventL(const TAlfEvent& aEvent)
 *   {
 *   TBool consumed(EFalse);
 *   if ( aEvent.IsPointerEvent() )
 *       {
 *       CAlfVisual* visual = aEvent.Visual();
 *       //Orientation for scroller
 *       TAlfScrollerEvent::TOrientation orientation = TAlfScrollerEvent::EVertical;
 *       
 *       if ( aEvent.PointerDown() )
 *           {
 *           //if ( some condition to start scroller event handling )
 *               {
 *               //Create event instance
 *               TAlfScrollerEvent scrollEvent(TAlfScrollerEvent::EMoveByItem, orientation, -1);
 *               TAlfEvent event( *(visual->Display()), scrollEvent );
 *               //Pass it to handler
 *               scrollerInputHandler->OfferEventL( event ); 
 *               consumed = ETrue;
 *               }
 *           }
 *        }
 *    return consumed;
 *    }
 * @endcode
 * @lib alfclient.lib
 * @since S60 xxx
 */
NONSHARABLE_CLASS( TAlfScrollerEvent )
    {
    
public:
    /**
     * Indicates what type of scroller the event originated in
     */
    enum TOrientation
        {
        EVertical,
        EHorizontal
        };

    /** 
     *Scroller event code
     */
    enum TCode
        {
        EMoveByItem, // Moves the view by a single logical unit. i.e. a text line or a list item
        EMoveByView, // Moves the view by an amount similar to a view height/width
        EMoveToPosition // Moves the view top to the requested position
        };

public: 
    /* Constructors. */
   
    /**
     * Constructor.
     */
    IMPORT_C TAlfScrollerEvent( TCode aCode, TOrientation aOrientation, TInt aMagnitude );

    /**
     * Constructor.
     *
     * Supplied only for default construction within TAlfEvent
     */
    TAlfScrollerEvent();
    
public: // Access methods

    /**
     * @return the event code for the scroller event
     */
    IMPORT_C TCode Code() const;
    
    /**
     * Orientation of the scroller
     *
     * @return orientation of the scroller generating this event
     */
    IMPORT_C TOrientation Orientation() const;
    
    /**
     * Access to the size of the scroller event.
     *
     * This magnitude represents the number of items, views, or other event code-dependent quantity
     *
     * For EMoveByItem, this number is the number of items to move forward. If negative, then it represents 
     * moving the view backward by this number of items.
     *
     * For EMoveByView, this number is the number of views forward to move.
     *
     * For EMoveToPosition, this number is in the arbitrary units of the associated scroller model. (i.e,.in order to
     * generate this type of event you must be observing the corresponding @c MAlfScrollerModelReader )
     *
     *  @return the magnitude of the event
     */
    IMPORT_C TInt Magnitude() const;
    
private: 
    
    /** 
     * The scroller event code. 
     */
    TCode iCode;

    /**
     * Orientation of this scroller
     */
    TOrientation iOrientation;    

    /**
     * A magnitude representing the number of items, views, or other code-dependent quantity
     * @see Magnitude()
     */
    TInt iMagnitude;
    
    TInt iSpare;    
    };


#endif // T_ALFSCROLLEREVENT_H
