// Copyright (c) 1995-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of the License "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// e32\include\e32event.h
// 
//

#ifndef __E32EVENT_H__
#define __E32EVENT_H__
#include <e32cmn.h>
#ifdef BTRACE_TRAWEVENT	
#include <e32btrace.h>
#endif

/**
@publishedAll
@released
*/
const TInt KUndefinedDeviceNumber = -1;

/**
@publishedAll
@released

Represents a raw hardware event as generated by hardware drivers of
a pointer device, a keyboard etc.

Raw hardware events are added to the kernel's event queue.
*/
class TRawEvent
	{
public:

    /**
    Defines the event type.
    */
	enum TType
		{
		
		/**
		Represents an uninitialised event object.
		*/
		ENone,
		
		/**
		A pointer device (e.g. a pen) has moved. Only changes in Cartesian coordinates are tracked.
		*/
		EPointerMove,
		
		/**
		A switch on event caused by a screen tap using a pointer device. 
		*/
        EPointerSwitchOn,
        
        /**
        Represents a keyboard key down event.
        */
		EKeyDown,
		
		/**
		Represents a keyboard key up event.
		*/
		EKeyUp,
		
		/**
		Represents a redraw event.
		
		Typically generated in an emulator environment in response
		to a host OS redraw event.
		*/
		ERedraw,
		
		/**
		Represents a device switch on event.
		*/
		ESwitchOn,
		
		/**
		*/
        EActive,
        
  		/**
		*/
		EInactive,
		
		/**
		Represents a modifier key being pressed.
		*/
		EUpdateModifiers,
		
		/**
		Represents a button down event.
		
		This is typically used to represent a pointing device
		coming into contact with a touch sensitive screen.
		*/
		EButton1Down,
		
		/**
		Represents a button up event.
		
		This is typically used to represent a pointing device
		being lifted away from a touch sensitive screen.
		*/
		EButton1Up,
		
		/**
		Represents a button down event.
		
		This is typically used to represent a pointing device
		coming into contact with a touch sensitive screen.
		*/
		EButton2Down,
		
		/**
		Represents a button up event.
		
		This is typically used to represent a pointing device
		being lifted away from a touch sensitive screen.
		*/
		EButton2Up,
		
		/**
		Represents a button down event.
		
		This is typically used to represent a pointing device
		coming into contact with a touch sensitive screen.
		*/
		EButton3Down,
		
		/**
		Represents a button up event.
		
		This is typically used to represent a pointing device
		being lifted away from a touch sensitive screen.
		*/
		EButton3Up,
		
		/**
		Represents a device switch off event.
		*/
		ESwitchOff,
		
		/**
		Represents a key being continually pressed event.
		*/
		EKeyRepeat,
		
		/**
		Represents a case open event.
		
		The meaning of a case is hardware specific.
		*/
		ECaseOpen,
		
		/**
        Represents a case close event.
		
		The meaning of a case is hardware specific.
		*/
		ECaseClose,

		/**
		@prototype
        Represents a 3D pointer entering the detection volume event.
		
		This is typically used to represent a pointing device
		coming into detection range of a sensitive screen.
		*/
		EPointer3DInRange,

		/**
		@prototype
        Represents a 3D pointer leaving the detection volume event.
		
		This is typically used to represent a pointing device
		leaving the detection range of a sensitive screen.
		*/
		EPointer3DOutOfRange,

		/**
		@prototype
        Represents a 3D pointer tilt changes only event.
		
		This is typically sent by a 3D pointer driver when it detects
		a change in the pointer's polar coordinates (Theta or Phi).
		*/
		EPointer3DTilt,

		/**
		@prototype
        Represents a 3D pointer rotation only event.
		
		This is typically sent by a 3D pointer driver when it detects
		a rotation of the pointing device (along ist main axis).
		*/
		EPointer3DRotation,

		/**
		@prototype
        Represents a general 3D pointer changes event.
		
		This is typically sent by a 3D pointer driver when it detects
		a change in the pointer's linear and/or polar coordinates and/or rotation.
		*/
		EPointer3DTiltAndMove,

		/**
		@prototype
		Reserved for a 3D pointing device button down event.
		*/
		EButton4Down,
		
		/**
		@prototype
		Reserved for a 3D pointing device button up event.
		*/
		EButton4Up,

		/**
		@prototype
		Reserved for a 3D pointing device button down event.
		*/
		EButton5Down,
		
		/**
		@prototype
		Reserved for a 3D pointing device button up event.
		*/
		EButton5Up,

		/**
		@prototype
		Reserved for a 3D pointing device button down event.
		*/
		EButton6Down,
		
		/**
		@prototype
		Reserved for a 3D pointing device button up event.
		*/
		EButton6Up,

		/**
		Represents a device restart event.
		*/
		ERestartSystem
		};
public:

    /**
    Default constructor
    */
	inline TRawEvent()
		{ *(TInt*)&iType=0; }
		
	/**
	Gets the event type
	
	@return The event type.
	*/	
	inline TType Type() const
		{return TType(iType);}

	/**
	Gets the device number (eg. screen number)

	@return The device number.
	*/
	inline TInt DeviceNumber() const
		{return TInt(iDeviceNumber-1);}

	/**
	Sets the device number (eg. screen number)

	@param aDeviceNumber The device number
	*/
	inline void SetDeviceNumber(TInt aDeviceNumber)
		{iDeviceNumber = TUint8(aDeviceNumber+1);}

	IMPORT_C TPoint Pos() const;
	IMPORT_C TInt ScanCode() const;
    IMPORT_C TInt Modifiers() const;
	IMPORT_C TInt Repeats() const;
    
    /**
    Gets the Cartesian coordinates of the 3D pointer end that is closer to the screen.
    
    @return The Cartesian coordinates of the point defined by the end of the 3D pointing device that is closer to the screen.
    */
	IMPORT_C TPoint3D Pos3D() const;
    

	/**
    Gets the angular spherical polar coordinates of the 3D pointer end that is closer to the screen.
    
    @return The angular spherical polar coordinates of the point defined by the end of the 3D pointing device that is closer to the screen.
	@see TAngle3D
	*/
	IMPORT_C TAngle3D Tilt() const;

	/**
    Gets the rotation angle of 3D pointing device.

    Some 3D pointing devices support information regarding the angle formed between the physical and magnetical poles,
	as obtaining when rotating the pen along its main axis.

    @return The rotation angle of the 3D pointing device.
    */
	IMPORT_C TInt Rotation() const;

    /**
    Gets the tick count value associated with the event.
    
    Note that the interval between tick counts is
    hardware dependent.
    
    @return The tick count value
    */
	inline TUint Ticks() const
		{return iTicks;}
	/**
	Gets the information on which end of the pointing device is closer to the screen
	
	@return A Boolean indicating which end of the pointing device is closer to the screen.
	*/	
	inline TBool IsTip() const
		{return TBool(iTip);}
	
	/**
	Sets the information on which end of the pointing device is closer to the screen
	
	@param aTip A Boolean indicating which end of the pointing device is closer to the screen.
	*/	
	inline void SetTip(TBool aTip)
		{iTip = TUint8(aTip);
#ifdef BTRACE_TRAWEVENT	
		BTraceContext4(BTrace::ERawEvent, BTrace::ESetTipEvent,(TUint32)aTip);
#endif
		}

	IMPORT_C void Set(TType aType,TInt aScanCode);
	IMPORT_C void Set(TType aType,TInt aX,TInt aY);
	IMPORT_C void Set(TType aType);

	IMPORT_C void Set(TType aType,TInt aX,TInt aY,TInt aZ);
	IMPORT_C void Set(TType aType,TInt aX,TInt aY,TInt aZ,TInt aPhi,TInt aTheta,TInt aAlpha);

	IMPORT_C void SetTilt(TType aType,TInt aPhi,TInt aTheta);
	IMPORT_C void SetRotation(TType aType,TInt aAlpha);
	IMPORT_C void SetRepeat(TType aType,TInt aScanCode,TInt aRepeats);

	/**
	@return the pointer number of the the event
	*/
	inline TUint8 PointerNumber() const
		{
		return iPointerNumber;
		}


	/**
	The pointer number for the event is set
	The pointer  number should be more or equal 0 and should be less than HALData::EPointerMaxPointers,
	otherwise WSERV would ignore the event,ie 	0 <= aPointerNumber < HALData::EPointerMaxPointers 
	@param	aPointerNumber
	Error conditions: None.
	*/
	inline void SetPointerNumber(const TUint8 aPointerNumber)
		{
		iPointerNumber=aPointerNumber;	
#ifdef BTRACE_TRAWEVENT	
		BTraceContext4(BTrace::ERawEvent, BTrace::ESetPointerNumberEvent,(TUint32)aPointerNumber);
#endif
		}

    /**
	Initialises the event with the supplied information	
	@param	aType	The event type.
	@param	aX		The X position.
	@param	aY		The Y position.
	@param	aZ		The Z position.
	@param  aPointerNumber   The pointer number for the event
	*/
	IMPORT_C void Set (TType aType, TInt aX, TInt aY, TInt aZ, TUint8 aPointerNumber);
	
protected:
	TUint8 iType;
	TUint8 iTip;	/**< Indicates whether the tip or head of pointing device is closer to screen.*/
	TUint8 iPointerNumber; //
	TUint8 iDeviceNumber;
	TUint iTicks;
	union
		{
		struct {TInt x;TInt y;} pos;
		struct {TInt x;TInt y;TInt z;TInt phi;TInt theta;TInt alpha;} pos3D;
		struct {TInt scanCode;TInt repeats;} key;
		TInt modifiers;
		} iU;
	};




/**
@publishedAll
@released

Encapsulates a hardware event object as a descriptor (package buffer) for
the purpose of data transfer.
*/
class TRawEventBuf : public TPckgBuf<TRawEvent>
	{
public:

    /**
    Gets the hardware event object from the package buffer.
    
    @return The hardware event object.
    */
	inline TRawEvent &Event() const {return(*((TRawEvent *)&iBuf[0]));}
	};

#ifndef SYMBIAN_ENABLE_SPLIT_HEADERS
#include <e32event_private.h>
#endif

#endif

