// Copyright (c) 2004-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Implementation for the PPP ini configurator
// 
//

/**
 @file
*/
 
#include "inifileconfigurator.h"
#include "utf.h"

using te_ppploopback::CIniFileConfigurator;


namespace
	{
	
	/** Used internally to write entries in .ini files. 
	*/	
	_LIT8(KTestingLabel, "; This file is generated by te_incoming_ppp");
	_LIT8(KEqSignLit, "= "); 
	_LIT8(KNewlineLit, "\r\n");
	_LIT16(KOneLit, "1");

	
	/** PPP Specific initialization values 
	*/	
	// [lcp]
	_LIT(KPppEnableMaxFailure,   "PPPEnableMaxFailure");
	_LIT(KPppMaxFailureCount,    "PPPMaxFailureCount");
	_LIT(KPppEnableMaxRestart,   "PPPEnableMaxRestart");
	_LIT(KPppMaxRestartCount,    "PPPMaxRestartCount");
	_LIT(KPppEnableRestartTimer, "PPPEnableRestartTimer");
	_LIT(KPppRestartPeriod,      "PPPRestartPeriod");
	
	// [lrd]
	_LIT(KPppEnableLrd, "PPPEnableLRD");
	_LIT(KPppLrdPeriod, "PPPLRDPeriod");
	}

/**
 Static Construction
 
 @param aIniFileName fully qualified name of the ini file to configure.
 
 @post configurator constructed on file. If file existed, it was overwriten.
 */
CIniFileConfigurator* CIniFileConfigurator::NewL(TDesC& aIniFileName)
	{
	CIniFileConfigurator* self = NewLC(aIniFileName);
	CleanupStack::Pop(self);	
	return self;	
	}

/**
 * Static Construction
 
 @param aIniFileName fully qualified name of the ini file to configure.
 
 @post configurator constructed on file. If file existed, it was overwriten.
 */
CIniFileConfigurator* CIniFileConfigurator::NewLC(TDesC& aIniFileName)
	{
	CIniFileConfigurator* self = new (ELeave)CIniFileConfigurator();
	CleanupStack::PushL(self);
	self->ConstructL(aIniFileName);
	
	return self;	
	}

/**
 * Writes a UTF7 string to the ini file.
 *
 
 @param aCharacters the characters to write
 
 @post String was written to the file.
 */
void CIniFileConfigurator::WriteUTF7L(const TDesC& aCharacters)
	{
	HBufC8* charBuf = HBufC8::NewLC(aCharacters.Length());
	TPtr8 charBufDes = charBuf->Des();
	User::LeaveIfError(
		CnvUtfConverter::ConvertFromUnicodeToUtf7(charBufDes, aCharacters, EFalse));
	User::LeaveIfError(iIniFile.Write(*charBuf));
	CleanupStack::PopAndDestroy(charBuf);	
	}
	
/**
 * Writes a new section into the ini file.
 *

 @param aSectionName the section (eg: [section1])
 
 @post Section was written.
 */
void CIniFileConfigurator::CreateFileSectionL(const TDesC& aSectionName)
	{	
	User::LeaveIfError(iIniFile.Write(KNewlineLit));
	WriteUTF7L(aSectionName);
	User::LeaveIfError(iIniFile.Write(KNewlineLit));	
	}
	

/**
 * Writes a entry
 *

 @param aEntry the entry 
 @param aValue entry's value
 
 @post String of form "entry= value " was written.
 */
void CIniFileConfigurator::CreateEntryL(const TDesC& aEntry, const TDesC& aValue)
	{	
	WriteUTF7L(aEntry);
    User::LeaveIfError(iIniFile.Write(KEqSignLit));
	WriteUTF7L(aValue);
	User::LeaveIfError(iIniFile.Write(KNewlineLit));	
	}
	
/**
 * PPP specific: enables PPPMaxFailure.
 *

 
 @post PPPMaxFailure= 1 written to file.
 */
void CIniFileConfigurator::EnableMaxFailureL()
	{
	CreateEntryL(KPppEnableMaxFailure, KOneLit);
	}

/**
 * PPP specific: sets PPPMaxFailureCount
 *

 
 @post PPPMaxFailureCount= aValue written to file.
 */
void CIniFileConfigurator::SetMaxFailureCountL(TDesC& aValue)
	{
	CreateEntryL(KPppMaxFailureCount, aValue);
	}
	
/**
 * PPP specific: enables PPPMaxRestart
 *

 
 @post PPPEnableMaxRestart= 1 written to file.
 */
void CIniFileConfigurator::EnableMaxRestartL()
	{
	CreateEntryL(KPppEnableMaxRestart, KOneLit);
	}

/**
 * PPP sets PPPMaxRestartCount 
 *

 
 @post PPPMaxRestartCount= aValue written to file.
 */	
void CIniFileConfigurator::SetMaxRestarteCountL(TDesC& aValue)
	{
	CreateEntryL(KPppMaxRestartCount, aValue);
	}
/**
 * PPP specific: enables PPPMaxRestartTimer 
 *

 
 @post PPPMaxRestartTimer= 1 written to file.
 */	
void CIniFileConfigurator::EnableRestartTimerL()
	{
	CreateEntryL(KPppEnableRestartTimer, KOneLit);
	}
		
/**
 * PPP specific: sets PPPMaxRestartPeriod 
 *

 
 @post PPPMaxRestartPeriod= aValue written to file.
 */	
void CIniFileConfigurator::SetMaxRestartPeriodL(TDesC& aValue)
	{
	CreateEntryL(KPppRestartPeriod, aValue);
	}
/**
 * PPP specific: enables PPPLrd 
 *

 
 @post PPPLrd= 1 is written to file.
 */	
void CIniFileConfigurator::EnableLrdL()
	{
	CreateEntryL(KPppEnableLrd, KOneLit);
	}

/**
 * PPP specific: sets PPPLrdPeriod 
 *

 
 @post PPPLrdPeriod= aValue is written to file
 */	
void CIniFileConfigurator::SetLrdPeriodL(TDesC& aValue)
	{
	CreateEntryL(KPppLrdPeriod, aValue);
	}


/**
 * Consstruction
 *

 
 @post None
 */
CIniFileConfigurator::CIniFileConfigurator()
	{
	}

/**
 * Destruction
 *

 
 @post underlying file resources are released.
 */	
CIniFileConfigurator::~CIniFileConfigurator()
	{
	iIniFile.Flush();  // Nothing we can do if error.
	iIniFile.Close();
	iFileServerSess.Close();	
	}
/**
 2nd phase of construction 
 Replaces the existing .ini file with an empty one (or creates a new one)
 

 @param aIniFileName the fully qualified name of the file to replace / create. 
 
 @post File replaced.
 */	
void CIniFileConfigurator::ConstructL(TDesC& aIniFileName)
	{
	User::LeaveIfError(iFileServerSess.Connect());	
	User::LeaveIfError(iIniFile.Replace(iFileServerSess, aIniFileName, EFileWrite));  // Make sure the file is brand new.
	
	// Write a warning so that users don't confuse the file with the default Symbian one.
	User::LeaveIfError(iIniFile.Write(KTestingLabel));
	User::LeaveIfError(iIniFile.Write(KNewlineLit));
	}

