/*
* Copyright (c) 2002-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
* ** IMPORTANT ** PublishedPartner API's in this file are published to 3rd party developers via the 
* Symbian website. Changes to these API's should be treated as PublishedAll API changes and the Security TA should be consulted.
*
*/




/**
 @file
 @publishedPartner
 @released
*/

#ifndef __PBE_H__
#define __PBE_H__

#include <e32std.h>
#include "pbebase.h"

class CPBEncryptionData;
class CPBEncryptor;
class CPBDecryptor;

/** 
 * Password Based Encryption ciphers.
 *
 * Note that RC2 has an additional key parameter, the "effective key length".
 *
 * Used in the construction of CPBEncryptElement, CPBEncryptSet, CPBEncryptParms,
 * and CPBEncryptionData objects and in the CPBEncryptParms::Cipher() function.
 */
enum TPBECipher
	{
	/** AES cipher in CBC mode with a supplied key size of 128 bits. */
	ECipherAES_CBC_128,
	/** AES cipher in CBC mode with a supplied key size of 192 bits. */
	ECipherAES_CBC_192,
	/** AES cipher in CBC mode with a supplied key size of 256 bits. */
	ECipherAES_CBC_256,
	/** DES cipher in CBC mode (with a supplied key size of 56 bits). */
	ECipherDES_CBC,
	/** Triple-DES cipher in CBC mode. */
	ECipher3DES_CBC,
	/** 
	 * RC2 cipher in CBC mode with a supplied key length of 40 bits.
	 * 
	 * It has an effective key length of 1024 bits (128 bytes), which is compatible
	 * with OpenSSL RC2 encryption.
	 */
	ECipherRC2_CBC_40, 
	/**
	 * RC2 cipher in CBC mode with a supplied key length of 128 bits. 
	 * 
	 * It has an effective key length of 1024 bits (128 bytes), which is compatible
	 * with OpenSSL RC2 encryption.
	 */
	ECipherRC2_CBC_128,
	/**
	 * RC2 cipher in CBC mode with a supplied key length of 40 bits.
	 * 
	 * It has an effective key length of 128 bits (16 bytes), which is compatible 
	 * with the RC2 encryption used in PKCS#8 encryption keys generated by OpenSSL
	 */
	ECipherRC2_CBC_40_16,
	/**
	 * RC2 cipher in CBC mode with a supplied key length of 128 bits. 
	 * 
	 * It has an effective key length of 128 bits (16 bytes), which is compatible 
	 * with the RC2 encryption used in PKCS#8 encryption keys generated by OpenSSL
	 */
	ECipherRC2_CBC_128_16,
	/**
	 * ARC4 cipher with a supplied key length of 128 bits. 
	 * PKCS#12 PBE encryption algorithm 
	 */
	ECipherARC4_128,
	/**
	 * ARC4 cipher with a supplied key length of 40 bits. 
	 * PKCS#12 PBE encryption algorithm 
	 */
    ECipherARC4_40,    
	/**
	 * 2_KeyTriple-DES cipher in CBC mode. 
	 * PKCS#12 PBE encryption algorithm
	 */
    ECipher2Key3DES_CBC,
	/** 
	 *	RC2 Cipher in CBC mode with a supplied & effective key length of 40 bits. 
	 *  PKCS#12 PBE encryption algorithm
	 */
    ECipherRC2_CBC_40_5,
    };

/** 
 * Allows the password based encryption and decryption of elements.
 * Contains the encryption key and its associated encryption data.
 * See the Cryptography api-guide documentation for more information 
 * and sample code.
 */
class CPBEncryptElement : public CPBEncryptionBase
	{
public:
	/**
	 * Creates a new CPBEncryptElement object for encryption of new data.
	 *
	 * If strong cryptography is present, a 128 bit AES cipher is used; 
	 * otherwise, for weak cryptography, a 56 bit DES cipher is used.
	 *
	 * The symmetric key is derived from the password and a random salt using TPKCS5KDF::DeriveKeyL().
	 * 
	 * @param aPassword	The user supplied password
	 * @return			The new CPBEncryptElement object
	 */
	IMPORT_C static CPBEncryptElement* NewL(const TPBPassword& aPassword);

	/**
	 * Creates a new CPBEncryptElement object for encryption of new data.
	 * 
	 * If strong cryptography is present, a 128 bit AES cipher is used; 
	 * otherwise, for weak cryptography, a 56 bit DES cipher is used.
	 *
	 * The symmetric key is derived from the password and a random salt using TPKCS5KDF::DeriveKeyL().
	 *
	 * A pointer to the returned object is put onto the cleanup stack.
	 *
	 * @param aPassword	The user supplied password
	 * @return			The new CPBEncryptElement object
	 */
	IMPORT_C static CPBEncryptElement* NewLC(const TPBPassword& aPassword);

	/**
	 * Creates a new CPBEncryptElement object for encryption of new data.
	 *
	 * The symmetric key is derived from the password and a random salt using TPKCS5KDF::DeriveKeyL().
	 * 
	 * @param aPassword	The user supplied password
	 * @param aCipher	The cipher to use
	 * @return			The new CPBEncryptElement object
	 */
	IMPORT_C static CPBEncryptElement* NewL(const TPBPassword& aPassword, 
		TPBECipher aCipher);

	/**
	 * Creates a new CPBEncryptElement object for encryption of new data.
	 *
	 * The symmetric key is derived from the password and a random salt using TPKCS5KDF::DeriveKeyL().
	 * 
	 * A pointer to the returned object is put onto the cleanup stack.
	 *
	 * @param aPassword	The user supplied password
	 * @param aCipher	The cipher to use
	 * @return			The new CPBEncryptElement object
	 */
	IMPORT_C static CPBEncryptElement* NewLC(const TPBPassword& aPassword, 
		TPBECipher aCipher);

	/**
	 * Creates a new CPBEncryptElement object for encryption of new data.
	 *
	 * The symmetric key is derived from the password using TPKCS5KDF::DeriveKeyL().
	 * 
	 * @param aPassword	The user supplied password
	 * @param aParms	An encryption parameter object comprising the cipher,
	 *					salt, IV, and iteration count value. 
	 * @return			The new CPBEncryptElement object
	 */
	IMPORT_C static CPBEncryptElement* NewL(const TPBPassword& aPassword, 
		const CPBEncryptParms& aParms);

	/**
	 * Creates a new CPBEncryptElement object for encryption of new data.
	 *
	 * The symmetric key is derived from the password using TPKCS5KDF::DeriveKeyL().
	 * 
	 * A pointer to the returned object is put onto the cleanup stack.
	 *
	 * @param aPassword	The user supplied password
	 * @param aParms	An encryption parameter object comprising the cipher,
	 *					salt, IV, and iteration count value. 
	 * @return			The new CPBEncryptElement object
	 */
	IMPORT_C static CPBEncryptElement* NewLC(const TPBPassword& aPassword, 
		const CPBEncryptParms& aParms);

	/**
	 * Creates a new CPBEncryptElement object for decryption of existing data.
	 *
	 * If the specified password is valid, the function regenerates the encryption key;
	 * otherwise, it leaves with KErrBadPassphrase.
	 *
	 * @param aData				The encryption data object
	 * @param aPassword			The user supplied password
	 * @return					The new CPBEncryptElement object
	 * @leave KErrBadPassphrase	If the specified password is incorrect
	 */
	IMPORT_C static CPBEncryptElement* NewL(const CPBEncryptionData& aData,
		const TPBPassword& aPassword);

	/**
	 * Creates a new CPBEncryptElement object for decryption of existing data.
	 *
	 * If the specified password is valid, the function regenerates the encryption key;
	 * otherwise, it leaves with KErrBadPassphrase.
	 *
	 * A pointer to the returned object is put onto the cleanup stack.
	 *
	 * @param aData				The encryption data object
	 * @param aPassword			The user supplied password
	 * @return					The new CPBEncryptElement object
	 * @leave KErrBadPassphrase	If the specified password is incorrect
	 */
	IMPORT_C static CPBEncryptElement* NewLC(const CPBEncryptionData& aData,
		const TPBPassword& aPassword);

	/** 
	 * Gets the parameters allowing one to re-create the object with the
	 * same state at another point in the future.  
	 * 
	 * In order to decrypt any information previously encrypted with this object, 
	 * you <B><I>must</I></B> store this encryption data along with it. Failure 
	 * to do this will result in the permanent loss of the encrypted information.
	 * 
	 * @return The data allowing one to re-create this object at a later time.					
	 */
	const CPBEncryptionData& EncryptionData(void) const;

	/** 
	 * Constructs a CPBEncryptor object allowing the encryption of data.
	 * 
	 * @return	A pointer to a CPBEncryptor object.
	 *			The caller assumes ownership of the returned object.
	 */
	CPBEncryptor* NewEncryptL(void) const;

	/** 
	 * Constructs a CPBEncryptor object allowing the encryption of data.
	 * 
	 * @return	A pointer to a CPBEncryptor object.
	 *			The caller assumes ownership of the returned object.
	 *			The returned pointer is left on the cleanup stack.
	 */
	CPBEncryptor* NewEncryptLC(void) const;

	/** 
	 * Constructs a CPBDecryptor object allowing the decryption of data.
	 * 
	 * @return	A pointer to a CPBDecryptor object.
	 *			The caller assumes ownership of the returned object.
	 */
	CPBDecryptor* NewDecryptL(void) const;

	/** 
	 * Constructs a CPBDecryptor object allowing the decryption of data.
	 * 
	 * @return	A pointer to a CPBDecryptor object.
	 *			The caller assumes ownership of the returned object.
	 *			The returned pointer is left on the cleanup stack.
	 */
	CPBDecryptor* NewDecryptLC(void) const;

	/** 
	 * Gets the maximum output ciphertext length given a specified input plaintext length.  
	 * 
	 * @param aPlaintextLength	The plaintext length 
	 * @return					The maximum ciphertext length given a plaintext length.
	 */
	TInt MaxCiphertextLength(TInt aPlaintextLength) const;

	/** 
	 * Gets the maximum output plaintext length given a specified input ciphertext length.
	 *
	 * @param aCiphertextLength	The ciphertext length
	 * @return					The maximum plaintext length given a ciphertext length.
	 */
	TInt MaxPlaintextLength(TInt aCiphertextLength) const;

	/** Destructor */
	virtual ~CPBEncryptElement(void);
protected:
	/** @internalAll */
	void ConstructL(const TDesC8& aPassword);
	/** @internalAll */
	void ConstructL(const TDesC8& aPassword, const TPBECipher aCipher);
	/** @internalAll */
	void ConstructL(const TDesC8& aPassword, const CPBEncryptParms& aParms);
	/** @internalAll */
	void ConstructL(const CPBEncryptionData& aData, const TPBPassword& aPassword);
	/** @internalAll */
	TBool AuthenticateL(const TPBPassword& aPassword);
	/** @internalAll */
	void MakeEncryptKeyL(TUint aKeySize, const TDesC8& aPassword);
	/** @internalAll */
	CPBEncryptElement(void);
protected:
	/** The encryption data */
	CPBEncryptionData* iData;
	/** The derived encryption key */
	HBufC8* iEncryptKey;
private:
	CPBEncryptElement(const CPBEncryptElement&);
	CPBEncryptElement& operator= (const CPBEncryptElement&);
	};

/** 
 * Derived class to allow the efficient password based encryption and
 * decryption of multiple elements.
 * 
 * This is useful if one wants random access to an encrypted source consisting 
 * of multiple independent elements, for example, a database or a store. 
 * 
 * Since it is unreasonable to force the decryption of an entire set to allow 
 * access to just a tiny portion of it, and since it is too costly to derive separate 
 * keys for each element within the set, a single randomly generated <I>master</I> 
 * key is used.  This master key is encrypted with the password provided by the 
 * user of the class. Known plaintext attacks against the ciphertext are prevented 
 * by using a randomly chosen Initialisation Vector (IV) for each element.  
 * 
 * Contains the master encryption key.
 *
 * See the Cryptography api-guide documentation for more information and sample code.
 *
 * @see CPBEncryptElement
 * 
 * @since v8.0
 */
class CPBEncryptSet : public CPBEncryptElement
	{
public:
	/**
	 * Creates a new CPBEncryptSet object for encryption of new data 
	 * (and generates an encrypted master key).
	 *
	 * If strong cryptography is present, a 128 bit AES cipher is used; 
	 * otherwise, for weak cryptography, a 56 bit DES cipher is used.
	 *
	 * The symmetric key is derived from the password and a random salt using TPKCS5KDF::DeriveKeyL().
	 *
	 * @param aPassword	The users password.
	 * @return			A new CPBEncryptSet object
	 */
	IMPORT_C static CPBEncryptSet* NewL(const TPBPassword& aPassword);

	/**
	 * Creates a new CPBEncryptSet object for encryption of new data 
	 * (and generates an encrypted master key).
	 *
	 * The returned pointer is put onto the cleanup stack.
	 *
	 * If strong cryptography is present, a 128 bit AES cipher is used; 
	 * otherwise, for weak cryptography, a 56 bit DES cipher is used.
	 *
	 * The symmetric key is derived from the password and a random salt using TPKCS5KDF::DeriveKeyL().
	 *
	 * @param aPassword	The user supplied password
	 * @return			The new CPBEncryptSet object
	 */
	IMPORT_C static CPBEncryptSet* NewLC(const TPBPassword& aPassword);

	/**
	 * Creates a new CPBEncryptSet object for encryption of new data 
	 * (and generates an encrypted master key).
	 *
	 * The symmetric key is derived from the password and a random salt using TPKCS5KDF::DeriveKeyL().
	 * 
	 * @param aPassword	The user supplied password
	 * @param aCipher	The cipher to use
	 * @return			The new CPBEncryptSet object
	 */
	IMPORT_C static CPBEncryptSet* NewL(const TPBPassword& aPassword, 
		TPBECipher aCipher);

	/**
	 * Creates a new CPBEncryptSet object for encryption of new data 
	 * (and generates an encrypted master key).
	 *
	 * The returned pointer is put onto the cleanup stack.
	 *
	 * The symmetric key is derived from the password and a random salt using TPKCS5KDF::DeriveKeyL().
	 * 
	 * @param aPassword	The user supplied password
	 * @param aCipher	The cipher to use
	 * @return			The new CPBEncryptSet object
	 */
	IMPORT_C static CPBEncryptSet* NewLC(const TPBPassword& aPassword, 
		TPBECipher aCipher);

	/**
	 * Creates a new CPBEncryptSet object for encryption of new data 
	 * (and generates an encrypted master key).
	 *
	 * The symmetric key is derived from the password using TPKCS5KDF::DeriveKeyL().
	 * 
	 * @param aPassword	The user supplied password
	 * @param aParms	An encryption parameter object comprising the cipher,
	 *					salt, IV, and iteration count value. 
	 * @return			The new CPBEncryptSet object
	 */
	IMPORT_C static CPBEncryptSet* NewL(const TPBPassword& aPassword, 
		const CPBEncryptParms& aParms);

	/**
	 * Creates a new CPBEncryptSet object for encryption of new data 
	 * (and generates an encrypted master key).
	 *
	 * The returned pointer is put onto the cleanup stack.
	 *
	 * The symmetric key is derived from the password using TPKCS5KDF::DeriveKeyL().
	 * 
	 * @param aPassword	The user supplied password
	 * @param aParms	An encryption parameter object comprising the cipher,
	 *					salt, IV, and iteration count value. 
	 * @return			The new CPBEncryptSet object
	 */
	IMPORT_C static CPBEncryptSet* NewLC(const TPBPassword& aPassword, 
		const CPBEncryptParms& aParms);

	/**
	 * Creates a new CPBEncryptSet object for encryption of new data 
	 * (and generates an encrypted master key).
	 *
	 * If the specified password is valid, the function regenerates the encryption key;
	 * otherwise, it leaves with KErrBadPassphrase.
	 *
	 * @param aData					The encryption data object to copy 
	 * @param aEncryptedMasterKey	On return, the encrypted master key
	 * @param aPassword				The user supplied password
	 * @return						The new CPBEncryptSet object
	 * @leave KErrBadPassphrase		If the specified password is incorrect
	 */
	IMPORT_C static CPBEncryptSet* NewL(const CPBEncryptionData& aData,
		const TDesC8& aEncryptedMasterKey, const TPBPassword& aPassword);

	/**
	 * Creates a new CPBEncryptSet object for encryption of new data 
	 * (and generates an encrypted master key).
	 *
	 * The returned pointer is put onto the cleanup stack.
	 *
	 * If the specified password is valid, the function regenerates the encryption key;
	 * otherwise, it leaves with KErrBadPassphrase.
	 *
	 * @param aData					The encryption data object to copy 
	 * @param aEncryptedMasterKey	On return, the encrypted master key
	 * @param aPassword				The user supplied password
	 * @return						The new CPBEncryptSet object
	 * @leave KErrBadPassphrase		If the specified password is incorrect
	 */
	IMPORT_C static CPBEncryptSet* NewLC(const CPBEncryptionData& aData,
		const TDesC8& aEncryptedMasterKey, const TPBPassword& aPassword);
	
	/** 
	 * Gets the encrypted form of the master key.  
	 *
	 * This must be stored along with the object returned by CPBEncryptElement::EncryptionData() 
	 * in order for the object to be reconstructed with the same state at
     * some time in the future. Failure to do so will result in the permanent
     * loss of any information encrypted with this object.
     * 
     * @return		The encrypted master key.
     */
	IMPORT_C const TDesC8& EncryptedMasterKey(void) const;
	
	/** 
	 * Constructs a CPBEncryptor object based on the state of this object
	 * (i.e., the cipher and master key) allowing the encryption of data.
	 * 
	 * @return	A pointer to a CPBEncryptor object.
	 *			The caller assumes ownership of the returned object.
	 */
	CPBEncryptor* NewEncryptL(void) const;

	/** 
	 * Constructs a CPBEncryptor object based on the state of this object
	 * (i.e., the cipher and master key) allowing the encryption of data.
	 * 
	 * @return	A pointer to a CPBEncryptor object.
	 *			The caller assumes ownership of the returned object.
	 *			The returned pointer is left on the cleanup stack.
	 */
	CPBEncryptor* NewEncryptLC(void) const;

	/** 
	 * Constructs a CPBDecryptor object based on the state of this object
	 * (i.e., the cipher and master key) allowing the decryption of data.
	 * 
	 * @return	A pointer to a CPBDecryptor object.
	 *			The caller assumes ownership of the returned object.
	 */
	CPBDecryptor* NewDecryptL(void) const;

	/** 
	 * Constructs a CPBDecryptor object based on the state of this object
	 * (i.e., the cipher and master key) allowing the decryption of data.
	 * 
	 * @return	A pointer to a CPBDecryptor object.
	 *			The caller assumes ownership of the returned object.
	 *			The returned pointer is left on the cleanup stack.
	 */
	CPBDecryptor* NewDecryptLC(void) const;

	/** 
     * Re-encrypts the master key with the specified new password.
     *
     * @param aNewPassword	The new password
     */
	IMPORT_C void ChangePasswordL(const TPBPassword& aNewPassword);

	/** 
	 * Gets the maximum output ciphertext length given a specified input plaintext length.  
	 * 
	 * @param aPlaintextLength	The plaintext length 
	 * @return					The maximum ciphertext length given a plaintext length.
	 */
	TInt MaxCiphertextLength(TInt aPlaintextLength) const;

	/** 
	 * Gets the maximum output plaintext length given a specified input ciphertext length.
	 *
	 * @param aCiphertextLength	The ciphertext length
	 * @return					The maximum plaintext length given a ciphertext length.
	 */
	TInt MaxPlaintextLength(TInt aCiphertextLength) const;
	
	/** Destructor */
	virtual ~CPBEncryptSet(void);
protected:
	/** @internalAll */
	void ConstructL(const TDesC8& aPassword);
	/** @internalAll */
	void ConstructL(const TDesC8& aPassword, TPBECipher aCipher);
	/** @internalAll */
	void ConstructL(const TDesC8& aPassword, const CPBEncryptParms& aParms);
	/** @internalAll */
	void ConstructMasterKeyL(void);
	/** @internalAll */
	void ConstructL(const CPBEncryptionData& aData, 
		const TDesC8& aEncryptedMasterKey, const TPBPassword& aPassword);
	/** @internalAll */
	void DecryptMasterKeyL(TDes8& aMasterKey) const;
	/** @internalAll */
	void EncryptMasterKeyL(const TDesC8& aMasterKey);
protected:
	/** @internalAll */
	CPBEncryptSet(void);
	/** The derived encrypted master key*/
	HBufC8* iEncryptedMasterKey;
private:
	CPBEncryptSet(const CPBEncryptSet&);
	CPBEncryptSet& operator= (const CPBEncryptSet&);
	};

/** 
 * Class representing both 8 and 16 bit descriptor passwords.
 * Internally these are stored as 8 bit passwords.
 */
class TPBPassword
	{
public:
	/** 
	 * Sets the password.
	 * 
	 * Constructs a TPBPassword object with an 8 bit descriptor.
	 * 
	 * Internally this is represented as an octet byte sequence 
	 * (aka 8 bit TPtrC8 descriptor).
	 * 
	 * @param aPassword	A const reference to an 8 bit descriptor.
	 * 					representing the users initial password.
	 */
	IMPORT_C TPBPassword(const TDesC8& aPassword);
	
	/** 
	 * Sets the password.
	 * 
	 * Constructs a TPBPassword object with a 16 bit descriptor.
	 *
	 * Internally this is represented as an octet byte sequence
	 * (aka 8 bit TPtrC8 descriptor).
	 * 
	 * @param aPassword	A const reference to a 16 bit descriptor
	 * 					representing the users initial password.
	 */
	IMPORT_C TPBPassword(const TDesC16& aPassword);
	
	/**
	 * Gets the password.
	 * 
	 * Gets a const reference to an 8 bit descriptor representing the users
	 * initial password (which could have been either 8 or 16 bit).
	 * 
	 * @return		A const reference to an 8 bit descriptor.
	 */
	IMPORT_C const TDesC8& Password(void) const;
private:
	TPtrC8 iPassword;
	};

#endif
