/*
* Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
* Implements client interface to the KMS LDD.  The KMS uses
* the API defined in this file to control the LDD.
*
*/


/**
 @file
*/

#include "kmsldd.h"
#include "kmslddcommon.h"

using namespace KmsLddImpl;


EXPORT_C TInt RKmsChannel::Open()
/**
	Open a channel with the KMS LDD.  This function can only be used by
	the key management server process.
	
	@return					KErrNone on success, any other error code on failure.
 */
	{
	// device driver factory name.  This is not the LDD filename.
	const TDesC& name = RKmsChannel::FactoryName();
	
	// the driver must have been built with at least the same version as this channel.
	const TVersion reqVer = KmsLddImpl::Version();

	return DoCreate(
		name,
		reqVer,
		KNullUnit,			// aUnit: units not supported (KDeviceAllowUnit flag not set)
		NULL,				// aDriver: PDD name, not supported (KDeviceAllocPhysicalDevice flag not set)
		NULL,				// anInfo: allow info not supported (KDeviceAllowInfo flag not set)
		EOwnerThread,		// aType: handle is owned by thread, not process
		EFalse);			// aProtected: channel cannot be shared with other processes
	}

EXPORT_C void RKmsChannel::GenerateKey(
	TInt aLength, RProcess aOwner, TKeyHandle& aHandle, TRequestStatus& aStatus)
/**
	Generate a key and store it in the hardware.  The generated key is owned by
	the identified process, which must delete it when it is no longer in use.
	
	@param	aLength			Length of new key in bytes.
	@param	aOwner			Identifies process which will own this key.
	@param	aHandle			On success this is set to the newly-created key's handle.
							Its value is undefined on failure.
	@param	aStatus			This object is completed when the request has finished.
							On completion it will contain KErrNone on success, or any other
							error code on failure.  KErrArgument is returned if the store
							cannot support the supplied key length.
	@return					
	@see DeleteKey
 */
	{
	TGenKeyArgs gka = {aLength, aOwner.SecureId(), &aHandle};
	TGenKeyArgsPckg gkaPckg(gka);
	DoRequest(EGenerateKey, aStatus, (TAny*) &gkaPckg);
	}

EXPORT_C void RKmsChannel::StoreKey(const TDesC8& aData, RProcess aOwner, TKeyHandle& aHandle, TRequestStatus& aStatus)
/**
	Store a key in the hardware store.  This API is provided to transfer
	keys to hardware if required to use accelerator hardware.  The stored key
	is owned by the identified process, which must delete it when it is no
	longer in use.
	
	@param	aData			Key contents.
	@param	aOwner			Identifies process which will own this key.
	@param	aHandle			On success this is set to the newly-created key's handle.
							Its value is undefined on failure.
	@param	aStatus			This object is completed when the request has finished.
							On completion it will contain KErrNone on success, or any other
							error code on failure.  KErrArgument is returned if the store
							cannot support the supplied key length.
	@see DeleteKey
 */
	{
	TStoreKeyArgs ska = {aOwner.SecureId(), &aHandle};
	TStoreKeyArgsPckg skaPckg(ska);
	DoRequest(EStoreKey, aStatus, (TAny*) &aData, (TAny*) &skaPckg);
	}

EXPORT_C void RKmsChannel::DeleteKey(RProcess aClaimedOwner, TKeyHandle aHandle, TRequestStatus& aStatus)
/**
	Delete a key from the hardware store.  The key must have been
	created with GenerateKey or StoreKey.
	
	@param	aClaimedOwner	This process is claiming ownership of the key and attempting
							to delete it.
	@param	aHandle			Handle generated by GenerateKey or StoreKey.
	@param	aStatus			This object is completed when the request has finished.
							On completion it will contain KErrNone on success, or any other
							error code on failure.
	@see GenerateKey
	@see StoreKey
 */
	{
	TSecureId sid = aClaimedOwner.SecureId();
	TUint32 handleValue = aHandle.iValue;
	DoRequest(EDeleteKey, aStatus, (TAny*) sid.iId, (TAny*) handleValue);
	}

EXPORT_C void RKmsChannel::AddUsage(RProcess aClaimedOwner, TKeyHandle aHandle, TInt aOperation, const TSecurityPolicy& aSecPol, TRequestStatus& aStatus)
/**
	Add a usage operation and policy for the supplied key.
	
	@param	aClaimedOwner	This process is claiming ownership of the key, and therefore the
							right to add a usage.
	@param	aHandle			Identifies key in hardware store.
	@param	aOperation		Numeric value identifies operation.
	@param	aSecPol			Security policy which should be applied to clients wishing to use
							the supplied key for the supplied operation.
	@param	aStatus			This object is completed when the request has finished.
							On completion it will contain KErrNone on success, or any other
							error code on failure.
 */
	{
	TSecureId sid = aClaimedOwner.SecureId();
	TUsageIdentifierArgs aia = {sid, aHandle, aOperation};
	TUsageIdentifierArgsPckg aiaPckg(aia);
	TPckg<TSecurityPolicy> secPolPckg(aSecPol);
	DoRequest(EAddUsage, aStatus, (TAny*) &aiaPckg, (TAny*) &secPolPckg);
	}

EXPORT_C void RKmsChannel::DeleteUsage(RProcess aDeletor, TKeyHandle aHandle, TInt aOperation, TRequestStatus& aStatus)
/**
	Delete a usage for the supplied key.

 	@param	aDeletor		This process is claiming ownership of the key, and therefore the
							right to delete a usage.
	@param	aHandle			Identifies key in hardware store.
	@param	aOperation		Numeric value identifies operation.
	@param	aStatus			This object is completed when the request has finished.
							On completion it will contain KErrNone on success, or any other
							error code on failure.
 */
	{
	TSecureId sid = aDeletor.SecureId();
	TUsageIdentifierArgs uia = {sid, aHandle, aOperation};
	TUsageIdentifierArgsPckg uiaPckg(uia);
	DoRequest(EDeleteUsage, aStatus, (TAny*) &uiaPckg);
	}

EXPORT_C void RKmsChannel::UsageAllowed(RProcess aUserProc, TKeyHandle aHandle, TInt aOperation, TBool& aAllowed, TRequestStatus& aStatus)
/**
	Predicate function tests whether the supplied process can use the supplied key for
	the supplied operation.  This API is defined for informational purposes only and is
	not used by the key management server.  It's primary purpose is to test the key store
	drivers.

	@param	aUserProc		Process which wants to use the key.
	@param	aHandle			Identifies existing key in store.
	@param	aOperation		Numeric identifier for operation.  Example operations
							include signing, but the actual meaning is not defined here.
	@param	aAllowed		On success this is set to whether or not the process can
							perform the requested operation with the key.  Its value
							is undefined on failure.
	@param	aStatus			This object is completed when the request has finished.
							On completion it will contain KErrNone on success, any other error
							code on failure. In this context, "success" means can determine whether
							or not the operation is allowed.  It does not necessarily mean the
							operation <em>is</em> allowed.
 */
	{
	TUsageAllowedArgs uaa = {aUserProc.Handle(), aHandle, aOperation};
	TUsageAllowedArgsPckg uaaPckg(uaa);
	DoRequest(EUsageAllowed, aStatus, (TAny*) &uaaPckg, (TAny*) &aAllowed);
	}


