use strict;
use warnings;
use File::Spec::Functions;
use Getopt::Long;
use Pod::Usage;

# Version of the script - just use the date
$main::VERSION = '18-Oct-08';

# Get arguments
my ( $version, $help, $verbose, @dirs, @files, $debug, $drive, $epocroot, $configini, $lproj );
GetOptions("lproj|l" => \$lproj, "config|c=s" => \$configini, "epocroot|e=s" => \$epocroot, "destinationdrive|dd:s" => \$drive, "debug" => \$debug, "version|ver" => \$version, "help" => \$help, "verbose|v" => \$verbose, "dir|d=s" => \@dirs, "file|f=s" => \@files) or pod2usage('Invalid parameters');

# Handle help and version
pod2usage({ verbose => 1, exitval => 0}) if $help;
version() if $version;

# Check arguments
$verbose = 1 if $debug;
$verbose = $verbose ? '-v' : '';
$epocroot = $ENV{EPOCROOT} if !$epocroot;
$lproj = '' if !$lproj;

if ($configini)
	{
	# There shouldn't be any need for any other arguments
	pod2usage('Unnecessary arguments supplied.') if @dirs or @files or $drive or @ARGV;
	
	# Now do the installation
	my $installer = WidgetInstaller->new('verbose' => $verbose, 'debug' => $debug, 'epocroot' => $epocroot, 'lproj' => $lproj);
	$installer->installConfig($configini);
	}
else
	{
	# Check for destination drive
	pod2usage('Intended destination drive (-dd) not specified on command line.') if !$drive;
	pod2usage('Invalid drive format: $drive.') if $drive !~ /^([a-zA-Z]):?$/;
	$drive = uc($1).':';

	# Issue a warning about UID clashes
	warn "WARNING: UID\'s may clash across drives. Use a config file (-c) to safely install Widgets on multiple drives.\n";

	# Pull from the rest of the arguments if nothing specified
	if (!@files and !@dirs)
		{
		foreach my $entry ( @ARGV )
			{
			push @dirs, $entry if -d $entry;
			push @files, $entry if -f $entry;
			}
		}
	
	# Find files in the given directories
	foreach my $dir ( @dirs )
		{
		opendir DIR, $dir or die "Failed to open directory $dir: $!";
		foreach my $file ( grep /\.(wdgt\.zip|wgz)$/i, readdir DIR )
			{
			# Add the file to the list
			push @files, catfile($dir, $file);
			}
		closedir DIR;
		}

	# Give an error if we haven't got any files to process
	pod2usage('No files or directories found') if !@files;

	# Now do the installation
	my $installer = WidgetInstaller->new('verbose' => $verbose, 'debug' => $debug, 'epocroot' => $epocroot, 'lproj' => $lproj);
	$installer->installFiles($drive, \@files);

	# Make IBY files
	$installer->makeIBY($drive);
	$installer->makeIBY($drive, 'localised');
	}

# ***
# Package for installing Widgets
#
package WidgetInstaller;

use File::Spec::Functions;
use File::Path;
use File::Basename;
use Unicode::String qw(utf8 utf16);
use File::Temp qw/tempdir/;
use File::Copy;
use XML::Parser;
use Data::Dumper;
use XML::Simple;

# CONSTANTS
use constant ICON_SIZES => "88,32,24"; 			# Size of icons to generate in the MBM file
use constant INTERNAL_UID_LOWER => 0x2000DAD2;	# Lower UID bound for midlets installed to internal drives and rom
use constant EXTERNAL_UID_LOWER => 0x2000DCC6;	# Lower UID bound for midlets installed to removable cards
use constant WIDGET_UI_UID => 0x10282822;		# UID of the widget app

# Folder paths
use constant DESTINATION => 'private/10003a3f/import/apps/NonNative/Resource';
use constant ROM_DEST => 'epoc32/release/winscw/udeb/Z/';
use constant DRIVE_DEST => 'epoc32/winscw/%s';
use constant WIDGET_REGISTRY => 'private/10282f06/WidgetEntryStore.xml';

# Create a new object
sub new
	{
	my $invocant = shift;
	my $self = bless({}, ref $invocant || $invocant);

	my %args = @_;
	$self->{'args'} = \%args;
	$self->{'args'}->{'verbose'} = $self->{'args'}->{'verbose'} ? '-v' : '';

	$self->{'installedFiles'} = ();
	$self->{'freeuidint'} = INTERNAL_UID_LOWER;
	$self->{'freeuidext'} = EXTERNAL_UID_LOWER;
	$self->{'langmapping'} = $self->getLangMapping($self->{'args'}->{'lproj'});

	return $self;
	}

# Gets the language mapping from the Widget_lproj.xml file
sub getLangMapping
	{
	my ( $self, $lproj ) = @_;
	
	# Get the LPROJ file which specifies lang id mappings
	if (!$lproj)
		{
		$lproj = WidgetInstaller::fixFilename("${epocroot}epoc32/data/Z/private/10282f06/Widget_lproj.xml");
		$lproj = WidgetInstaller::fixFilename("${epocroot}S60/mw/web/WebEngine/WidgetRegistry/Data/Widget_lproj.xml") if !-e $lproj;
		$lproj = WidgetInstaller::findCmd('Widget_lproj.xml') if !-e $lproj;
		undef $lproj if !-e $lproj;
	
		# Display a warning if localisation can't be performed
		warn "WARNING: Can't find Widget_lproj.xml - localisation is disabled.\n" if !$lproj;
		}

	# Load the mapping file
	if ($lproj)
		{
		die "Can't find $lproj file." if !-e $lproj;
		print "Localisation support enabled using config: $lproj\n" if $verbose;
		my $mapping = XMLin($lproj);
		print "Found ", scalar(@{ $mapping->{'LangID'} }), " language mappings.\n" if $verbose;
		return $mapping;
		}
	}

# Install Widgets listed in config file
# Format is as follows where "drive-z" specifies widgets for drive z: etc...
# Comments are okay - they start with a # character :-)
# If the file doesn't exist as given then EPOCROOT is prepended to the filename to try and find it
#
# [drive-z]
# \path\widget1.wgz
# widget2.wdgt.zip
#
# [drive-e]
# widget3.wgz

sub installConfig
	{
	my ( $self, $file ) = @_;
	$self->{'installedFiles'} = ();
	
	my ( $drive, %installData );
	open CONFIG, $file or die "Failed to open config file $file: $!";
	while(my $line = <CONFIG>)
		{
		# Ignore comments
		$line =~ s/\#.*$//;
		if ($line =~ /^\[/)
			{
			undef $drive;

			# Have we found a new section
			if ($line =~ /^\[drive-([a-z])\]$/i)
				{
				$drive = uc($1);
				next;
				}
			}
		# Add to the list of Widget files
		if ($drive && $line =~ /^(.+?\.(?:wdgt\.zip|wgz))\s*$/i)
			{
			my $widget = $1;
			$widget = fixFilename(catfile($self->{'args'}->{'epocroot'}, $1)) if !-e $widget;
			die "Can't find widget $1" if !-e $widget;

			print "Widget for drive $drive: $widget\n" if $self->{'args'}->{'verbose'};
			push @{ $installData{"$drive:"} }, $widget;
			}
		}

	close CONFIG;

	# Now intall the widgets for each drive specified in the config file
	foreach my $drive ( keys %installData )
		{
		$self->installFiles($drive, $installData{$drive} );
		$self->makeIBY($drive);
		$self->makeIBY($drive, 'localised');
		}
	}

# ***
# Installs files to a drive
#
sub installFiles
	{
	my ( $self, $drive, $fileList ) = @_;
	print "Installing files for drive $drive\n" if $self->{'args'}->{'verbose'};
	
	# Unregister any existing widgets as otherwise when the registry is rewritten their icons will appear in the emulator but they won't work
	$self->unregisterWidgets($drive);

	# Process each widget in turn
	my ( @installedProps );
	foreach my $filename ( @$fileList )
		{
		# Check the file exists
		die "Can't find $filename" if !-e $filename;

		# Create a temporary folder
		print "\nInstalling $filename\n";
		my $tempdir = tempdir ( DIR => '.', CLEANUP => !$self->{args}->{'debug'});

		# Unzip the file
		my @extracted;
		my $size =0;
		die "Can't find unzip.exe tool on PATH." if !findCmd('unzip.exe');
		open UNZIP, "unzip.exe \"$filename\" -d $tempdir|" or die "Failed to unzip $filename: $!";
		while(<UNZIP>)
			{
			if (/(?:inflating|extracting):\s+$tempdir\/(.+?)\s*$/)
				{
				# Save extracted file details
				push @extracted, $1;
				print "Extracting: $1\n" if $self->{args}->{'debug'};
	
				# Remember total size for later
				$size += -s "$tempdir/$1";
				}
			}
		close UNZIP;
		die "No files extracted from $filename" if !@extracted;

		# Load the PLIST file that describes the widget
		my ( $root ) = $extracted[0] =~ /^([^\/]+)\//;
		my $plist = catfile($tempdir, $root, 'Info.plist');
		die "Can't find $root/Info.plist file" if !-e $plist;

		# Parse the XML file into a hash
		my $widgetdata = parsePList($plist);

		# Set widget package properties
		$widgetdata->{'FileName'} = $filename;
		$widgetdata->{'FileSize'} = $size;

		# Fix up some of the fields
		$widgetdata->{'PropertyListVersion'} = 1;
		$widgetdata->{'BundleName'} = $widgetdata->{'BundleDisplayName'} if !$widgetdata->{'BundleName'};
		$widgetdata->{'AllowNetworkAccess'} = $widgetdata->{'AllowFullAccess'} if !$widgetdata->{'AllowNetworkAccess'};
		$widgetdata->{'DriveName'} = $drive;
		$widgetdata->{'BasePath'} = sprintf('%s\\private\\%08x\\%s\\', $widgetdata->{'DriveName'}, WIDGET_UI_UID, $widgetdata->{'BundleIdentifier'});
		$widgetdata->{'MainHTML'} = "$widgetdata->{'BasePath'}$root\\$widgetdata->{'MainHTML'}";
		$widgetdata->{'IconPath'} = "$widgetdata->{'BasePath'}$root\\";

		# Load the language translations for BundleDisplayName
		$widgetdata->{'LocBundleDisplayName'} = $self->getLocalisedStrings(catfile($tempdir, $root), 'DisplayName');

		# Find the UID to use
		$widgetdata->{'Uid'} = $self->findfreeUid($widgetdata);
		print sprintf("Using UID for midlet: %08x\n", $widgetdata->{'Uid'}) if $self->{args}->{'verbose'};

		# Make sure the destination exists
		my $dest = $self->regFileName($drive);
		mkpath $dest;

		# Create the MBM file icon
		die "ERROR: Can't find PNG2MBM command in PATH." if !(my $cmd = findCmd('png2mbm.pl'));
		my $icon = catfile($tempdir, $root, 'Icon.png');
	        die "ERROR: Widget bundle must include an Icon.png file in $root directory.\n" unless -e $icon;
		my $mbm = $self->regFileName($drive, sprintf("[%08x].mbm", $widgetdata->{'Uid'}));
		print "Generating: $mbm\n";
		die "Failed to create MBM $mbm " if system("perl $cmd $self->{args}->{'verbose'} -in \"$icon\" -out $mbm -sizes ".ICON_SIZES) != 0;

		# Add the mbm to the list of files
		$self->addToRomList($drive, $mbm);

		# Create the INI file which defines the registry info
		my $ini = catfile($tempdir, 'reg.ini');
		$self->makeIni($widgetdata, $ini);
		unlink 'debug.ini'; copy($ini, 'debug.ini') if $debug;
		print "Generated INI file: $ini\n" if $self->{args}->{'verbose'};

		# Generate the registry files
		die "ERROR: Can't find WidgetRegFiles.exe command in PATH." if !($cmd = findCmd('WidgetRegFiles.exe'));
		die "Failed to generate registry files" if system("$cmd \"$ini\"") != 0;
		my ( $reg, $loc ) = ( catfile($dest, sprintf("%08x_reg.rsc", $widgetdata->{'Uid'})), catfile($dest, sprintf("%08x_loc.rsc", $widgetdata->{'Uid'})) );
		die "Failed to generate REG file: $!" if !-e $reg;
		$self->addToRomList($drive, $reg);
		die "Failed to generate LOC file: $!" if !-e $loc;
		$self->addToRomList($drive, $loc, 'localised');

		# Create install folder
		my $dir = $self->installDir($drive, $widgetdata->{'BundleIdentifier'});
		mkpath $dir;
	
		# Now copy the widget files to the right place
		print "Install Directory: $dir\n";
		foreach my $widgetfile ( @extracted )
			{
			my ( $sourceFile, $destFile ) = ( catfile($tempdir, $widgetfile), catfile($dir, $widgetfile) );
			print "Copying $sourceFile to $destFile\n" if $self->{args}->{'debug'};
	
			mkpath dirname($destFile);
			unlink $destFile;
			copy $sourceFile, $destFile or die "Failed to copy $sourceFile to $destFile: $!";
			$self->addToRomList($drive, $destFile);
			}

		# Copy the MBM file into the widget install directory
		# because native installation does and uses it after
		# installation to manage UID consistency.
		my $mbmName = sprintf("[%08x].mbm", $widgetdata->{'Uid'});
		my $destFile = catfile($dir, $root, $mbmName);
		print "Copying $mbm to $destFile\n" if $self->{args}->{'debug'};
		copy $mbm, $destFile or die "Failed to copy $mbm to $destFile: $!";
		$self->addToRomList($drive, $destFile);

		# Remember the data for the registry
		push @installedProps, $widgetdata;
		if (!$debug)
			{
			rmtree $tempdir or die "Failed to delete $tempdir: $!";
			}
		}

	# Generate the registry and IBY file
	$self->addToRomList($drive, $self->makeRegistry($drive, \@installedProps));
	}

# Get localised version of strings if they exist
sub getLocalisedStrings
	{
	my ( $self, $dir, $strName ) = @_;
	return if (!$self->{'langmapping'});

	# Iterate through all the languages we know about
	my %result;
	for(my $i = 0; $i < scalar(@{ $self->{'langmapping'}->{LangID} }); $i++)
		{
		my ( $langid, $langname ) = ( $self->{'langmapping'}->{LangID}->[$i], $self->{'langmapping'}->{LangDir}->[$i] );

		# Generate the name of the file containing localised strings for this language
		my $locfile = catfile($dir, "$langname.lproj", "InfoPlist.strings");
		if (-e $locfile)
			{
			# Open the file
			print "Found $langname language translations in $locfile\n" if $verbose;
			open LOC, $locfile or die "Failed to open $locfile: $!";

			# Get the byte order mark from the start of the file
			my $bom;
			$bom = unpack("S", $bom) if (read(LOC, $bom, 2) == 2);
			if ($bom)
				{
				seek LOC, 3, 0 if $bom == 0xEFBB; # Skip utf8 bom				
				}
			else
				{
				# go back to start of file if no bom
				seek LOC, 0, 0;
				}

			while(my $line = <LOC>)
				{
				# Do unicode conversion
				my $ustr;
				if ($bom)
					{
					$ustr = utf16($line);
					$ustr->byteswap if $bom != 0xFFFE;
					}
				else
					{
					$ustr = utf8($line);
					}

				# Find the string we're looking for
				if ($ustr->utf8 =~ /(?:^|\s)$strName\s*=\s*\"([^\"]*)\"/)
					{
					print "\t...$strName => $1\n" if $debug;
					$result{$langid} = utf8($1);
					last;
					}
				}
			close LOC;
			}
		}
	return \%result if keys %result;
	}

# Find and execute a command
sub findCmd
	{
	my $cmd = shift;
	return fixFilename("./$cmd") if -e $cmd;

	# Search each entry in the PATH
	my @paths = split /[;:]/, $ENV{PATH};
	push @paths, dirname($0);
	foreach my $path ( @paths )
		{
		my $fullcmd = fixFilename(catfile($path, $cmd));
		return $fullcmd if -e $fullcmd;
		}
	}

# Make INI file describing widget - this is passed to widgetregfiles.exe
sub makeIni
	{
	my ( $self, $data, $file ) = @_;
	open INI, ">$file" or die "Failed to open $file for writing: $!";
 
	# Get directory where mbm should go
	my $dir = $self->regFileName($data->{'DriveName'});

	print INI "[app_registration_info]\n";
	print INI sprintf("uid=%08x\n", $data->{'Uid'});
	print INI "app_file=$data->{'MainHTML'}\n";
	print INI "caption=$data->{'BundleDisplayName'}\n";
	print INI "drive_name=$data->{'DriveName'}\n";
	print INI "results_dir=$dir\n";

	# Add language stuff if we have the mapping
	if ($data->{'LocBundleDisplayName'})
		{
		my @langList;
		foreach my $langid ( sort { $a <=> $b } keys %{ $data->{'LocBundleDisplayName'} } )
			{
			my $symid = sprintf("%02d", $langid);
			push @langList, $symid;
			print INI "caption$symid=", $data->{'LocBundleDisplayName'}->{$langid}->utf8, "\n";
			}
		print INI "languages=", join(' ', @langList), "\n";
		}

	close INI;
	convert2Unicode($file);
	}

# Find a UID to use for the widget
sub findfreeUid
	{
	my ( $self, $data ) = @_;
	my $appfile = lc $data->{'MainHTML'};

	# pick the next free UID
	my $uid = isInternal($data->{'DriveName'}) ? $self->{'freeuidint'}++ : $self->{'freeuidext'}++;
	return $uid;
	}

# Fix slash problems in a filename
sub fixFilename
	{
	my $filename = shift;
	$filename =~ s/([\\\/])[\\\/]/$1/g;
	return catfile(split(/[\\\/]/, $filename));
	}

# Get the destination for REG/LOC/MBM file
sub regFileName
	{
	my ( $self, $drive, $filename ) = @_;
	my $result = catfile($self->{'args'}->{'epocroot'}, $drive =~ /^[zZ]/ ? ROM_DEST : sprintf(DRIVE_DEST, substr($drive, 0, 1)), DESTINATION, $filename);
	return fixFilename($result);
	}

# Widget install directory
sub installDir
	{
	my ( $self, $drive, $id ) = @_;
	my $result = catfile($self->{'args'}->{'epocroot'}, $drive =~ /^[zZ]/ ? ROM_DEST : sprintf(DRIVE_DEST, substr($drive, 0, 1)), 'private', sprintf("%08x", WIDGET_UI_UID), $id);
	return fixFilename($result);	
	}

# Determines whether a drive should be considered internal or not
sub isInternal
	{
	my $drive = shift;
	die "Invalid drive format: $drive" if $drive !~ /^[a-zA-Z]:/;
	return 1 if $drive =~ /^[zcZC]/;
	}

# Parse these awful PLIST files
sub parsePList
	{
	my $file = shift;

	# Create parser object
	our ($key, $val, $plisthash ) = ('', '', {});
	my $parser = new XML::Parser;
	$parser->setHandlers('Doctype' => \&docT, 'Start' => \&startH, 'End' => \&endH, 'Char' => \&dataH);

	# Parse the file
	open XML, $file or die "Couldn't open $file";
	$parser->parse(*XML);
	close XML;

	# Check required fields exist
	die "Widget MainHTML unknown" if !$plisthash->{'MainHTML'};
	die "Widget BundleIdentifier unknown" if !$plisthash->{'BundleIdentifier'};
	die "Widget BundleDisplayName unknown" if !$plisthash->{'BundleDisplayName'};

	# Return result
	return $plisthash;

	# Called on a start tag
	sub startH
		{
		my ($p, $el, %atts) = @_;
		undef $key if ($el =~ /^key$/i);
		$val = '';
		}
	
	# Receives document type
	sub docT
		{
		my ($expat, $name, $sysid, $pubid, $internal ) = @_;
		die "PLIST format looks wrong!" if lc($name) ne 'plist';
		$plisthash->{'NokiaWidget'} = ( $pubid =~ m[^-//Nokia//DTD PLIST]i ) ? 1 : 0;
		}

	# Receives character data
	sub dataH
		{
		my ($p, $s) = @_;
		$val .= $s;
		}
	
	# Called on an end tag
	sub endH
		{
		my ($p, $el) = @_;
		if ($el =~ /^key$/i)
			{
			$key = $val;
			}
		elsif ($key)
			{
			$val = 1 if $el =~ /^true$/i;
			$val = 0 if $el =~ /^false$/i;

			# Fix stuff so it's in the correct format
			$key =~ s/^CF//;
			$key = 'BundleIdentifier' if $key =~ /^Identifier$/i;
			$key = 'BundleDisplayName' if $key =~ /^DisplayName$/i;
			$key = 'BundleVersion' if $key =~ /^Version$/i;

			$plisthash->{$key} = $val;
			undef $key;
			}
		$val = ''
		}
	}

# Stores the details of files to be added to "rom"
sub addToRomList
	{
	my ( $self, $drive, $file, $localised ) = @_;
	$file = fixFilename($file);

	# All files should be under epoc32 somewhere - need to drop a bit of the path for the rom destination
	my $localpath = $drive =~ /^[zZ]/ ? fixFilename(catfile($self->{'args'}->{'epocroot'}, ROM_DEST, '')) : fixFilename(catfile($self->{'args'}->{'epocroot'}, sprintf(DRIVE_DEST, substr($drive, 0, 1)), ''));

	my $dest = fixFilename($file);
	$dest =~ s/^\Q$localpath\E//i;

	# Add the file to the list for the rom
	# It may be localised - in which it'll be put in a different IBY file
	$localised = $localised ? '_rsc' : '';
	$self->{"installedFiles${localised}"}->{$drive}->{$file} = $dest;
	}

# Make the IBY file
sub makeIBY
	{
	my ( $self, $drive, $localised ) = @_;

	# Generate the file name for the IBY file
	$localised = $localised ? '_rsc' : '';
	my $name = $drive =~ /^[zZ]/ ? "preinstalledwidgets${localised}.iby" : sprintf("preinstalledwidgets_drive%s${localised}.iby", substr($drive, 0, 1));
	my $iby = fixFilename(catfile($self->{'args'}->{'epocroot'}, 'epoc32', 'rom', 'include', $name));
	print "Generating: $iby\n";

	mkpath dirname($iby);
	open IBY, ">$iby" or die "Failed to open $iby for writing: $!";
	$name =~ s/\./_/g; $name = uc($name);
	print IBY "// GENERATED FILE: EDIT WITH CARE\n\#ifndef __${name}__\n\#define __${name}__\n\n";
	foreach my $file ( sort keys %{ $self->{"installedFiles${localised}"}->{$drive} } )
		{
		my $dest = $self->{"installedFiles${localised}"}->{$drive}->{$file};
		print IBY "data=$file\t$dest\n";
		}
	print IBY "\#endif\n";
	close IBY;
	}

# Unregister (with Apparc) existing Widgets
sub unregisterWidgets
	{
	my ( $self, $drive ) = @_;
	my $registry = fixFilename(catfile($self->{'args'}->{'epocroot'}, $drive =~ /^[zZ]/ ? ROM_DEST : sprintf(DRIVE_DEST, substr($drive, 0, 1)), WIDGET_REGISTRY));

	# If the registry already exists, remove apparc registry info for those widgets
	# This should avoid problems with unregistered widget icons in the emulator?
	if (-e $registry)
		{
  		my $ref = XMLin($registry, 'forcearray' => [ 'entry' ], 'keyattr' => { 'prop' => 'content' } );
		foreach my $entry ( @{ $ref->{entry} } )
			{
			my $uid = $entry->{prop}->{Uid}->{val}->{content};

			print "Unregistering existing Widget: $entry->{prop}->{BundleIdentifier}->{val}->{content}\n" if $verbose;
			my $dest = $self->regFileName($drive);
			my $mbm = catfile($dest, sprintf("[%08x].mbm", $uid));
			my ( $reg, $loc ) = ( catfile($dest, sprintf("%08x_reg.rsc", $uid)), catfile($dest, sprintf("%08x_loc.rsc", $uid)) );
			unlink $mbm, $reg, $loc;

			# We also have to delete the widget directory otherwise it'll be re-registered
			my $id = $entry->{prop}->{BundleIdentifier}->{val}->{content};
			my $dir = $self->installDir($drive, $id);
			rmtree $dir;		
			}
		}
	}

# Make the registry
sub makeRegistry
	{
	my ($self, $drive, $installed) = @_;
	my $registry = fixFilename(catfile($self->{'args'}->{'epocroot'}, $drive =~ /^[zZ]/ ? ROM_DEST : sprintf(DRIVE_DEST, substr($drive, 0, 1)), WIDGET_REGISTRY));
	print "\nGenerating: $registry\n";

	# Write the file
	mkpath dirname($registry);
	open OUT, ">$registry" or die "Failed to open WidgetEntryStore.xml: $!";
 
	print OUT "<?xml version=\"1.0\" encoding=\"UTF-16\" standalone=\"yes\" ?>\n<widgetregistry>\n";
	foreach my $pList ( @$installed )
		{
		dumpPList(\*OUT, $pList);
		}
	print OUT "</widgetregistry>\n";
	close OUT;

	# Convert the file to unicode
	convert2Unicode($registry);

	# Return the generated file
	return $registry;
	}

# Converts a file to Unicode
sub convert2Unicode
	{
	my $file = shift;
	
	my @lines;
	open IN, $file or die "Failed to open $file: $!";
	binmode IN;
	while(<IN>)
		{
		my $u = utf8($_);
		$u->byteswap;
		push @lines, $u->utf16;
		}
	close IN;

	open OUT, ">$file" or die "Failed to open $file for writing: $!";
	binmode OUT;
	print OUT pack("CC", 0xff, 0xfe);
	print OUT @lines;
	close OUT;
	}

# Dumps a single PList hash object
sub dumpPList
	{
	my ( $fh, $data ) = @_;
	my @regProperties = (
		[ 'PropertyListVersion', 'int' ],
		[ 'BundleIdentifier', 'string' ],
		[ 'BundleName', 'string' ],
		[ 'BundleDisplayName', 'string' ],
		[ 'MainHTML', 'string' ],
		[ 'BundleVersion', 'string' ],
		[ 'Height', 'int' ],
		[ 'Width', 'int' ],
		[ 'AllowNetworkAccess', 'int' ],
		[ 'DriveName', 'string' ],
		[ 'BasePath', 'string' ],
		[ 'IconPath', 'string' ],
		[ 'FileSize', 'int' ],
		[ 'Uid', 'int' ],
		[ 'NokiaWidget', 'int' ],
		[ 'MiniViewEnabled', 'int' ]
	);
	print $fh "<entry>\n";
	foreach my $prop ( @regProperties )
		{
		my ( $key, $type ) = @$prop;
		print $fh "<prop>$key<val>$data->{$key}<type>$type</type></val></prop>\n" if defined $data->{$key};
		}
	print $fh "</entry>\n";
	}

# New getopt::long provides this - but old version doesn't?
sub version
	{
	print sprintf("$0: $main::VERSION\n$^X: %vd\nos: $^O", $^V);
	exit;
	}

__END__

=head1 NAME

installwidgets.pl - A script for generating all the files needed to install Widgets

=head1 SYNOPSIS

There are two ways to run this tool...

 a) installwidgets.pl [-h] [-ver] [-v] [-e <dir>] [-l <lproj.xml>] -c <config file>
 b) installwidgets.pl [-h] [-ver] [-v] [-e <dir>] [-l <lproj.xml>] -dd <z|c|e>: <dir or files>

 Options:
   -config|c <config file>        Config INI file specifing the Widgets to intall
   -destinationdrive|dd <z|c|e>:  Destination Drive on which Widget will be installed
   -epocroot|e                    Override value of EPOCROOT
   -lproj|l                       Location of Widget_lproj.xml file specifying language mappings
   -verbose|v                     Show verbose output
   -version|ver                   Show version number
   -help|h                        Show this help

A script for generating all the files needed to preinstall Widgets.

 Examples:
   perl installwidgets.pl -c config.ini       Install widgets listed in config.ini
   perl installwidgets.pl -dd c: x:\my.wgz    Installs just my.wgz for c:
   perl installwidgets.pl -dd e: x:\my.wgz    Install Widget for drive e:

 Author:
   peter.harper@symbian.com

=head1 DESCRIPTION

This tool can be used to pre-generate all the files needed to install Widgets. The tool and its dependencies can be placed anywhere on your PATH.
It generates the results in the epoc32 folder - in the appropriate locations for the emulator.
It finds the epoc32 folder using the EPOCROOT environment variable which can be overridden via the -e command line option.

=head2 CONFIG FILE

Use a config file if you want to install Widgets to more than one drive. This will ensure there are no UID clashes.
The format of the config file is similar to an INI file. There's a separate section [drive-<char>] for each drive letter.
The widgets to install for that drive are then listed. You can specify the exact location of the Widget, otherwise it will try and find the Widget via EPOCROOT.

    # Widgets to be pre-installed for the ROM
    [drive-z]
    \somepath\foo.wgz
    \somepath\bar.wgz

    # Widgets for the internal disk
    [drive-c]
    \somepath\widget1.wdgt.zip

    # Widgets for the removable disk
    [drive-e]
    \somepath\widget2.wdgt.zip

=head2 DEPENDENCIES

The tool has some dependencies which must exist for it to work.

=over

=item 1

png2mbm.pl - A script to generate an MBM file from a PNG

=item 2

WidgetRegFiles.exe - an EXE which can generate Symbian REG and LOC files for registering non-native Widget apps.
This tool is built with "SymPort" a native tools port of basic Symbian OS services.

=item 3

unzip - For extracting files from the Widget archive.

=item 4

GD.pm - Perl support for the GD graphics library for PNG support, see http://www.libgd.org .

=back

=head3 INSTALLING GD

You can install GD automatically with a simple command - however the command you need to use differs depending on the version of Perl you have installed.
At the time of writing Symbian requires Perl version 5.6 - although in my experience Perl 5.8 works okay. To find out which version of Perl you have type "perl -v" on the command line.

To install the GD library:

=over

=item *

For Perl v5.6: "ppm install http://theoryx5.uwinnipeg.ca/ppmpackages/GD.ppd "

=item *

For Perl v5.8: "ppm install http://theoryx5.uwinnipeg.ca/ppms/GD.ppd "

=back

=head2 WIDGET INSTALL PROCESS

Here's a detailed breakdown of what the script does.

=over

=item 1

If a directory is specified on the command line it searches for Widget files with the extension "*.wdgt.zip" or "*.wgz".
Each Widget is processed in turn.

If a config.ini file is passed on the command line then it gets the lists of Widgets from here.
This process is repeated for all Widgets and all drives listed in the config file.

=item 2

Any existing Widgets listed in "private\10282f06\WidgetEntryStore.xml" are deleted from the epoc32 tree.
This ensures that there are no problems when testing Widgets in the emulator.

=item 3

All the compressed files in the Widget are extracted to a temporary folder.

=item 4

The details for the Widget are loaded from its "Info.plist" file.

=item 5

A UID is chosen for the widget. This differs depending on whether installation is for an internal drive (z: or c:) or an external drive (e: etc).
Note that if you aren't careful it's possible for UID's to clash when Widgets are installed to different drives.
To avoid this you should use a config file (-config or -c command line option) to preinstall all the Widgets needed on all drives at the same time.

=item 5

A Symbian MBM file is generated from the "Icon.png" file supplied by the Widgets.
Three different sized icons are generated "88x88", "32x32" and "24x24".
The MBM file is placed in "private/10003a3f/import/apps/NonNative/Resource/[<UID>].mbm".

=item 6

"WidgetRegFiles.exe" is executed to generate REG and LOC resource files used to register the Widget as an app in Symbian OS.
These files are placed in "private/10003a3f/import/apps/NonNative/Resource".

=item 7

All the widgets files are copied to a folder under "private\10282822".
The Widget's bundle identifier is used to create a unique folder under here for the Widget.

=item 8

The Widget registry is generated in "private\10282f06\WidgetEntryStore.xml"

=item 9

If Widgets are being preinstalled for ROM an IBY file is created in "epoc32\rom\include\preinstalledwidgets.iby".
A separate IBY file is generated for the localised parts of a Widget "epoc32\rom\include\preinstalledwidgets_rsc.iby".
Separate IBY files (per drive) are generated for Widgets preinstalled to UDA, e.g. preinstalledwidgets_driveC.iby and preinstalledwidgets_driveC_rsc.iby.
These IBY files can be used to add all the Widgets to ROM, ROFS or UDA.

=back

=head3 LOCALISATION

Widgets can specify localised caption names. These are stored in language specific versions of the LOC resource files, e.g. <uid>_loc.r01 for English and <uid>_loc.r02.
The appropriate file should be picked up automatically at runtime depending on the device language.

Localisation support is controlled by a file usually called Widget_lproj.xml. This contains a list of supported languages and a mapping from Symbian language id to language name.
You can tell the tool where to find this file with the -l command option. If this option is unspecified it tries to find the file in /epoc32/data/Z/private/10282f06 /S60/mw/web/WebEngine/WidgetRegistry/Data/Widget_lproj.xml.
If it still can't find the file it looks for the file in the current directory or somewhere else in your PATH. If it can't find the file Widgets will be unlocalised.

=head3 NOTES

=over

=item 1

The location of the private folder is in the appropriate place for the files to appear in the emulator.
This is different depending on the intended destination drive (see -dd command line option) for the Widget.
e.g. "epoc32/release/winscw/udeb/z/", "epoc32/winscw/c" or "epoc32/winscw/e"

=item 2

Files are extracted to epoc32 on the current drive relative to the EPOCROOT environment variable or the value given for -epocroot (-e) on the command line.

=item 3

A different IBY file is generated for each drive.

=over

=item *

Z: - \epoc32\rom\include\preinstalledwidgets.iby

=item *

C: - \epoc32\rom\include\preinstalledwidgets_driveC.iby

=item *

E: - \epoc32\rom\include\preinstalledwidgets_driveE.iby

=back

There are separate resource files for localised resources e.g. preinstalledwidgets_rsc.iby.

=cut
