use strict;
use warnings;
use File::Spec::Functions;
use Getopt::Long;
use Pod::Usage;
use File::Basename;

# Version of the script - just use the date
$main::VERSION = '07-Oct-08';

# Get arguments
my ( $version, $help, $verbose, $in, $out, $sizes, $debug);
GetOptions("debug|deb" => \$debug, "version|ver" => \$version, "help" => \$help, "verbose|v" => \$verbose, "in|i=s" => \$in, "out|o=s" => \$out, "sizes|s=s" => \$sizes) or pod2usage('Invalid parameters');
$verbose = 1 if $debug;

# Handle help and version
pod2usage({ verbose => 2, exitval => 0}) if $help;
version() if $version;
pod2usage("Invalid arguments") if !$in || !$out | !$sizes;

# Do the work
print "Creating $out from $in\n" if $verbose;
my $mbm = SymMbm->createFromPng($in, verbose => $verbose);
foreach my $size ( split(/,/, $sizes) )
	{
	print "Adding bitmap of size $size pixels\n" if $verbose;
	$mbm->addBitmap($size, $size);
	}
$mbm->dumpMbm($out);

# New getopt::long provides this - but old version doesn't?
sub version
	{
	print sprintf("$0: $main::VERSION\n$^X: %vd\nos: $^O", $^V);
	exit;
	}

# ***
# Package for creating MBM files
#
package SymMbm;

use GD;
use File::Temp qw/mktemp/;
use File::Copy;

# Create a new object from a PNG file
sub createFromPng
	{
	my $invocant = shift;
	my $self = bless({}, ref $invocant || $invocant);

	$self->{bitmaps} = [];
	$self->{masks} = [];
	$self->{in_filename} = shift;

	my %args = @_;
	$self->{args} = \%args;

	GD::Image->trueColor(1);

	$self->loadPng();

	return $self;
	}

# Delete the temporary files
sub DESTROY
	{
	my $self = shift;
	unlink @{$self->{bitmaps}};
	unlink @{$self->{masks}};
	}

# Generate the MBM file
# It should be possible to do this in code - but for now just run bmconv
sub dumpMbm
	{
	my $self = shift;
	my $filename = shift;

	unlink $filename;

	checkBmconv();
	my $cmd = "bmconv $filename";
	for (my $i = 0; $i < scalar(@{$self->{bitmaps}}); $i++)
		{
		$cmd .= " /c24$self->{bitmaps}[$i] /8$self->{masks}[$i]";
		}
	$cmd .= " >nul 2>&1" unless $self->{args}->{verbose};

	print "Executing: $cmd\n" if $self->{args}->{verbose};
	system($cmd);
	die "Failed to create MBM file $filename" if !-e $filename;
	}

# Check the bmconv tool exists
sub checkBmconv
	{
	my $found;
	open BMCONV, "bmconv 2>&1|" or die "Failed to execute command: $!";
	while(<BMCONV>)
		{
		if (/Symbian OS multiple bitmap file\/rom store conversion program./)
			{
			$found = 1;
			last;
			}
		}
	close BMCONV;
	die "ERROR: Can't find BMCONV command in PATH. This is usually found in \\epoc32\\tools" if (!$found);
	}

# Add an bitmap of the required dimensions
sub addBitmap
	{
	my $self = shift;
	my ( $width, $height ) = ( shift, shift );

	my ( $bmp, $mask ) = $self->dumpBitmaps($width, $height);
	push @{ $self->{bitmaps} }, $bmp;
	push @{ $self->{masks} }, $mask;
	}

# Load an image from a PNG
sub loadPng
	{
	my $self = shift;
	
	$self->{image} = GD::Image->new($self->{in_filename}) or die "Failed to load $self->{in_filename}: $!";
	}

# Save BMP image and mask to temp files
sub dumpBitmaps
	{
	my $self = shift;
	my ( $out_width, $out_height ) = ( shift, shift );
	my ( $out_filename, $out_filename_mask ) = ( mktemp(sprintf("bmp%d_XXXX", scalar @{ $self->{bitmaps} })), mktemp(sprintf("bmp%dmask_XXXX", scalar @{ $self->{bitmaps} })) );

	# Get input width and height
	my ( $in_width, $in_height ) = ( $self->{image}->width, $self->{image}->height );

	# Create a new image of the required dimensions
	my $out_img = GD::Image->new($out_width, $out_height);

	# Make sure background is white
	my $white = $out_img->colorClosest(255,255,255);
	$out_img->filledRectangle(0, 0, $out_width, $out_height, $white);

	# Resize the image
	$out_img->copyResampled($self->{image},0, 0, 0, 0, $out_width, $out_height, $in_width, $in_height);

	# Create bitmap
	open BMP, ">$out_filename" or die "Failed to open $out_filename: $!";
	binmode BMP;	
	printBmpHeader(\*BMP, $out_img);
	printBmpBody(\*BMP, $out_img);
	close BMP;

	# Generate the mask from the alpha channel
	my $mask_img = GD::Image->new($in_width + $in_width%2, $in_height + $in_height%2);
	$mask_img->filledRectangle(0, 0, $in_width, $in_height, $white);
	for(my $y = 0; $y < $in_height; $y++)
		{
		for(my $x = 0; $x < $in_width; $x++)
			{
			my $val = $self->{image}->getPixel($x, $y) >> 24;
			$val += 0x80 if $val > 0; # Transparency is 0->127
			$val = 0xff - $val; # Invert
			$mask_img->setPixel($x, $y, ($val << 16) + ($val << 8) + $val);
			}
		}
	
	# Make sure the background is white
	$out_img->filledRectangle(0, 0, $out_width, $out_height, $white);

	# Resize the mask
	$out_img->copyResampled($mask_img, 0, 0, 0, 0, $out_width, $out_height, $in_width, $in_height);

	# Create mask
	open BMPMASK, ">$out_filename_mask" or die "Failed to open $out_filename_mask: $!";
	binmode BMPMASK;
	printBmpHeader(\*BMPMASK, $out_img);
	printBmpBody(\*BMPMASK, $out_img);
	close BMPMASK;

	if ($debug)
		{
		unlink "bmp_${out_width}x$out_height.bmp";
		copy($out_filename, "bmp_${out_width}x$out_height.bmp");

		unlink "bmp_mask_${out_width}x$out_height.bmp";
		copy($out_filename_mask, "bmp_mask_${out_width}x$out_height.bmp");
		}

	return ( $out_filename, $out_filename_mask );
	}

# Print the header for a BMP file
sub printBmpHeader
	{
	my ( $file, $img ) = ( shift, shift );
	my ( $width, $height ) = ( $img->width, $img->height );

	printf $file '%s', pack 'A2ISSI', 'BM', ($width * $height * 3 ) + 54, 0, 0, 54;
	printf $file '%s', pack 'IiiSSIIiiII', 40, $width, $height, 1, 24, 0, $width * $height, 11811, 11811, 0, 0;
	}

# Print the body of a BMP file
sub printBmpBody
	{
	my ( $file, $img) = ( shift, shift );
	my ( $width, $height ) = ( $img->width, $img->height );
	for my $y ( reverse 0 .. $height - 1 )
		{
    	for my $x ( 0 .. $width - 1 )
			{
			my $real = $img->getPixel( $x, $y );
			my ( $r, $g, $b ) = $img->rgb($real);
        	printf $file '%s', pack 'CCC', ( $b, $g, $r );
    		}
		}
	}

__END__

=head1 NAME

png2mbm.pl - A script for generating mbm files from a png image

=head1 SYNOPSIS

png2mbm.pl -in <png file> -out <mbm file> -sizes <icon sizes>

 Options:
   -in <png file>        - PNG file to read
   -out <mbm file>       - Where to save the mbm file
   -sizes <icon sizes>   - Comma separated list of icon sizes
   -help                 - Show this help
   -version              - Show version of this script
   -verbose              - Verbose output

Reads a PNG file and generates bitmap images of the required sizes
These are then stored in an Symbian (multi-bitmap) MBM file

 Example:
   png2mbm.pl -i icon.png -o icon.mbm -s 88,32,24

 Author:
   peter.harper@symbian.com

=cut