// main.cpp
//
// Copyright (c) Symbian Software Ltd 1997-2008. All rights reserved.
//

#include <e32std.h>
#include <e32debug.h>
#include <f32file.h>
#include <s32file.h>
#include <s32mem.h>
#include <barsc2.h>
#include <barsread2.h>
#include <BACLINE.H>
#include <inifile.h>
#include <stdlib.h>

#define KMaxCaption 256
#define KAppMaxGroupName 16

const TUint KUidPrefixedNonNativeRegistrationResourceFile=0x1027289d;	
const TUint KUidAppRegistrationFile = 0x101F8021;
const TUint KUidWidgetLauncher = 0x10282821;

struct SAppDetails
	{
	TFileName iAppFile;
	TUid iUid;
	TBuf<KMaxCaption> iCaption;
	TBuf<1> iDrive;
	TUid iAppTypeUid;
	TBuf<KAppMaxGroupName> iGroupName;
	TFileName iResultsDir;
	TBuf<5> iLanguage; // Enough to hold 16bit decimal language code
	};

// This code is mostly ported from APPARC source code.
// The APPARC code was obviously not designed very well and so can't be reused - I tried!
// This code generates uncompressed (rather than Unicode compressed) resource files - it's easier.
// The files contain application registration data.
// We're using this to generate registration data for a non-native application (widgets).
//
// In the OS resource data (a REG file) is generated via:
// RApaLsSession::RegisterNonNativeApplicationL =>
//  CApaRegistrationResourceFileWriter::GenerateFileContentsL =>
//  CApaResourceFileWriterBase::DoGenerateFileContentsL =>
//  CApaRegistrationResourceFileWriter::MainResourceInCompiledFormatL
//
// Similarly localised resource data (a LOC file) is generated via:
// RApaLsSession::RegisterNonNativeApplicationL =>
//  CApaLocalisableResourceFileWriter::GenerateFileContentsL =>
//  CApaResourceFileWriterBase::DoGenerateFileContentsL =>
//  CApaLocalisableResourceFileWriter::MainResourceInCompiledFormatL
//
// The actual resource files are saved to file in:
// CApsNonNativeApplicationsManager::DoRegisterNonNativeApplicationL =>
//  CApsRegisterNonNativeApplication::WriteResourceFileL
//
// Note: Non-native apps TYPES are themselves registered and this information is kept in the following STORE file...
// c:\private\10003a3f\NonNativeTypes.dat
// It's quite trivial to pre-generate this information but it doesn't seem necessary in this case.
// Non-native app types are registered via RApaLsSession::RegisterNonNativeApplicationTypeL

void DumpRegFileL(const TDesC& aRegName, const TDesC& aLocName)
	{
	// SOURCE: CApaAppInfoReaderV2::Read
	RDebug::Print(_L("\n[app_registration_info]\nfile=%S]\n"), &aRegName);

	RFs fs;
	User::LeaveIfError(fs.Connect());
	CleanupClosePushL(fs);
	
	// SOURCE: CApaAppInfoReaderV2::ReadMandatoryInfoL
	
	TEntry entry;
	User::LeaveIfError(fs.Entry(aRegName, entry));
	RDebug::Print(_L("uids=%08x,%08x,%08x\n"), entry[0], entry[1], entry[2]);
	
	TInt fileOffset = sizeof(TCheckedUid);
	TInt fileLength = entry.iSize - fileOffset;

	CResourceFile* regResFile = CResourceFile::NewLC(fs, aRegName, fileOffset, fileLength);
	
	const TInt KAppRegistrationInfoResourceId = 1;

	RResourceReader reader;
	reader.OpenLC(regResFile, KAppRegistrationInfoResourceId);
	
	reader.ReadUint32L(); // skip over LONG reserved_long
	reader.ReadUint32L(); // skip over LLINK reserved_llink

	// read LTEXT app_file
	const TPtrC appFile(reader.ReadTPtrCL());
	RDebug::Print(_L("app_file=%S\n"), &appFile);
	
	// read LONG attributes
	TUint attributes = reader.ReadUint32L();
	RDebug::Print(_L("attributes=%08x\n"), attributes);

	// SOURCE: CApaAppInfoReaderV2::ReadNonLocalisableInfoL
	
	// read LTEXT localisable_resource_file
	TPtrC locResFileName(reader.ReadTPtrCL());
	RDebug::Print(_L("localisable_resource_file=%S\n"), &locResFileName);

	CResourceFile* locResFile = NULL;
	if (locResFileName.Length() > 0)
		{
		// expecting opaque data to be in the localisable resource file
		TParsePtrC parse(locResFileName);
	
		// open the localisable resource file	
		locResFile = CResourceFile::NewL(fs, aLocName, 0, 0);
		}
	CleanupStack::PushL(locResFile);
	
	// read LONG localisable_resource_id
	TUint locResId = reader.ReadUint32L();
	RDebug::Print(_L("localisable_resource_id=%d\n"), locResId);

	TInt appIsHidden = reader.ReadInt8L();
	RDebug::Print(_L("hidden=%d\n"), appIsHidden);

	TInt embeddability = reader.ReadInt8L();
	RDebug::Print(_L("embeddability=%d\n"), embeddability);
	
	TInt supportsNewFile = reader.ReadInt8L();
	RDebug::Print(_L("supports_new_file=%d\n"), supportsNewFile);

	TInt launchInBackground = reader.ReadInt8L();
	RDebug::Print(_L("launch_in_background=%d\n"), launchInBackground);

	TPtrC groupName = reader.ReadTPtrCL();
	RDebug::Print(_L("group_name=%S\n"), &groupName);

	// read BYTE default_screen_number
	TInt screen = reader.ReadUint8L();
	RDebug::Print(_L("screen=%d\n"), screen);

	// SOURCE: CApaAppInfoReaderV2::ReadDataTypesL

	// read LEN WORD STRUCT datatype_list[]
	const TInt dataTypeCount = reader.ReadInt16L();
	RDebug::Print(_L("datatype_count=%d\n"), dataTypeCount);

	for (TInt i=0; i < dataTypeCount; i++)
		{
		TInt priority = reader.ReadInt32L();
		TPtrC8 dataTypePtr = reader.ReadTPtrC8L();
		RDebug::Print(_L("datatype%d=%S\n"), i, &dataTypePtr);
		RDebug::Print(_L("datatype%d_priotity=%d\n"), i, priority);
		}
		
	const TInt fileOwnershipCount = reader.ReadInt16L();
	RDebug::Print(_L("file_ownership_count=%d\n"), fileOwnershipCount);

	for (TInt i=0; i < fileOwnershipCount; i++)
		{
		TPtrC fileNamePtr = reader.ReadTPtrCL();
		RDebug::Print(_L("file%d=%S\n"), i, &fileNamePtr);
		}

	// SOURCE: CApaAppInfoReaderV2::ReadNonLocalisableOptionalInfoL
	TInt serviceCount = 0;
	TRAPD(err, serviceCount = reader.ReadInt16L());
	if (err != KErrNone)
		{
		if (err != KErrEof)
			User::Leave(err);
		}
	else
		{
		RDebug::Print(_L("service_count=%d\n"), serviceCount);
		for (TInt i=0; i < serviceCount; i++)
			{
			const TUid serviceUid = {reader.ReadUint32L()};
			RDebug::Print(_L("service%d_uid=%08x\n"), i, serviceUid);
			
			// SOURCE: CApaAppInfoReaderV2::ReadDataTypesL

			// read LEN WORD STRUCT datatype_list[]
			const TInt dataTypeCount = reader.ReadInt16L();
			RDebug::Print(_L("service%d_data_type_count: %d\n"), i, dataTypeCount);

			for (TInt j=0; j < dataTypeCount; j++)
				{
				TInt priority = reader.ReadInt32L();
				TPtrC8 dataTypePtr = reader.ReadTPtrC8L();
				RDebug::Print(_L("service%d_datatype%d=%S\n"), i, j, &dataTypePtr);
				RDebug::Print(_L("service%d_datatype%d_priority=%d\n"), i, j, priority);
				}
			
			const TUint opaqueResId = reader.ReadUint32L();
			RDebug::Print(_L("service%d_opaque_resource_id=%d\n"), i, opaqueResId);

			// SOURCE: CApaAppInfoReaderV2::ReadOpaqueDataL
			if (opaqueResId > 0)
				{
				const TUint KResourceOffsetMask = 0xFFFFF000;
				HBufC8* opaqueData = NULL;
				if (opaqueResId & KResourceOffsetMask)
					{
					opaqueData = locResFile->AllocReadLC(opaqueResId);
					RDebug::Print(_L("service%d_opaque_loc_len=%d\n"), i, opaqueData->Length());
					}
				else
					{
					opaqueData = regResFile->AllocReadLC(opaqueResId);
					RDebug::Print(_L("service%d_opaque_reg_len=%d\n"), i, opaqueData->Length());
					}
				RDebug::Printf("service%d_opaque_data=%S\n", i, opaqueData); // Narrow
				CleanupStack::PopAndDestroy(opaqueData);
				}
			}
		
		// read LLINK opaque_data
		const TUint opaqueResId = reader.ReadUint32L();
		RDebug::Print(_L("opaque_resource_id=%d\n"), opaqueResId);

		if (opaqueResId > 0)
			{
			const TUint KResourceOffsetMask = 0xFFFFF000;
			HBufC8* opaqueData = NULL;
			if (opaqueResId & KResourceOffsetMask)
				{
				opaqueData = locResFile->AllocReadLC(opaqueResId);
				RDebug::Print(_L("opaque_loc_len=%d\n"), opaqueData->Length());
				}
			else
				{
				opaqueData = regResFile->AllocReadLC(opaqueResId);
				RDebug::Print(_L("opaque_loc_len=%d\n"), opaqueData->Length());
				}		
			RDebug::Printf("opaque_data=%S\n", opaqueData);	
			CleanupStack::PopAndDestroy(opaqueData);
			}
		}

	if (locResFile)
		{
		TParsePtrC parse(locResFileName);
		TPtrC name(parse.NameAndExt());
		RDebug::Print(_L("\n[localisable_app_info]\nfile=%S]\n"), &name);

		// SOURCE: CApaAppInfoReaderV2::ReadLocalisableInfoL
		
		RResourceReader locReader;
		locReader.OpenLC(locResFile, locResId);

		locReader.ReadUint32L(); // skip over LONG reserved_long
		locReader.ReadUint32L(); // skip over LLINK reserved_llink

		// read LTEXT short_caption
		TPtrC shortCaption = locReader.ReadTPtrCL();
		RDebug::Print(_L("short_caption=%S\n"), &shortCaption);

		locReader.ReadUint32L(); // skip over LONG reserved_long
		locReader.ReadUint32L(); // skip over LLINK reserved_llink

		// read LTEXT caption
		TPtrC caption = locReader.ReadTPtrCL();
		RDebug::Print(_L("caption=%S\n"), &caption);

		// read WORD number_of_icons
		const TInt numOfIcons = locReader.ReadInt16L();
		RDebug::Print(_L("icon_count=%d\n"), numOfIcons);
		
		// read LTEXT icon_file
		TPtrC iconFile = locReader.ReadTPtrCL();
		RDebug::Print(_L("icon_file=%S\n"), &iconFile);

		// read LEN WORD STRUCT view_list[]
		const TInt numOfViews = locReader.ReadInt16L();
		RDebug::Print(_L("views_count=%d\n"), numOfViews);

		for(TInt view = 0; view < numOfViews; ++view)
			{
			locReader.ReadUint32L(); // skip over LONG reserved_long
			locReader.ReadUint32L(); // skip over LLINK reserved_llink

			// read LONG uid
			const TUid viewUid = {locReader.ReadInt32L()};
			RDebug::Print(_L("view%d_uid=%08x\n"), view, viewUid.iUid);

			// read LONG screen_mode
			const TInt screenMode = {locReader.ReadInt32L()};
			RDebug::Print(_L("view%d_screen_mode=%d\n"), view, screenMode);

			locReader.ReadUint32L(); // skip over LONG reserved_long
			locReader.ReadUint32L(); // skip over LLINK reserved_llink

			// read LTEXT caption
			TPtrC viewCaption = locReader.ReadTPtrCL();
			RDebug::Print(_L("view%d_caption=%S\n"), view, &viewCaption);

			// read WORD number_of_icons
			const TInt numOfViewIcons = locReader.ReadInt16L();
			RDebug::Print(_L("view%d_icon_count=%d\n"), view, numOfViewIcons);
			
			// read LTEXT icon_file
			TPtrC viewIconFile = locReader.ReadTPtrCL();
			RDebug::Print(_L("view%d_icon_file=%S\n"), view, &viewIconFile);
			}
		
		// Read LTEXT group_name
		TBuf<0x10> groupName;
		TRAPD(ret, (groupName = locReader.ReadTPtrCL()));
		RDebug::Print(_L("group_name=%S\n"), &groupName);

		CleanupStack::PopAndDestroy(&locReader);
		}
	CleanupStack::PopAndDestroy(4, &fs);
	}

// SOURCE: \src\common\generic\app-framework\apparc\apgrfx\APGICNFL.CPP
class MDataSink
	{
public:
	MDataSink(RBuf8& aBuf) : iBuffer(aBuf), iNumberOfBytes(0) {};
	//
	TInt Length() { return iNumberOfBytes; }
	void DoWriteBufferL(const TDesC8& aBuffer);
	//
private:
	RBuf8& iBuffer;
	TInt iNumberOfBytes;
	};

void MDataSink::DoWriteBufferL(const TDesC8& aBuffer)
	{
	if (iBuffer.MaxLength() - iBuffer.Length() < aBuffer.Length())
		iBuffer.ReAllocL(iBuffer.Length() + aBuffer.Length() + 20);
	
	iBuffer.Append(aBuffer);
	iNumberOfBytes += aBuffer.Length();
	}

void WriteBufferL(MDataSink& aDataSink, const TDesC8& aBuffer)
	{
	if (aBuffer.Length()>0)
		aDataSink.DoWriteBufferL(aBuffer);
	}

void WriteLittleEndianUint32L(MDataSink& aDataSink, TUint aUint32)
	{
	TBuf8<4> buffer;
	buffer.Append(aUint32&0xff);
	buffer.Append((aUint32>>8)&0xff);
	buffer.Append((aUint32>>16)&0xff);
	buffer.Append((aUint32>>24)&0xff);
	WriteBufferL(aDataSink, buffer);
	}

void WriteUint8L(MDataSink& aDataSink, TUint aUint8)
	{
	TBuf8<1> buffer;
	buffer.Append(aUint8&0xff);
	WriteBufferL(aDataSink, buffer);
	}

void WriteTextL(MDataSink& aDataSink, const TDesC& aText)
	{
	// LTEXT
	WriteUint8L(aDataSink, aText.Length());

	if ((aDataSink.Length() % 2) != 0 && aText.Length() > 0)
		WriteUint8L(aDataSink, 0xab);

	WriteBufferL(aDataSink, TPtrC8(reinterpret_cast<const TUint8*>(aText.Ptr()), aText.Size()));
	}

void WriteLittleEndianUint16L(MDataSink& aDataSink, TUint aUint16)
	{
	TBuf8<2> buffer;
	buffer.Append(aUint16&0xff);
	buffer.Append((aUint16>>8)&0xff);
	WriteBufferL(aDataSink, buffer);
	}

void WriteUidTypeL(MDataSink& aDataSink, TUid aUid2, TUid aUid3)
	{
	WriteBufferL(aDataSink, TCheckedUid(TUidType(TUid::Uid(0x101f4a6b), aUid2, aUid3)).Des());
	}

// SOURCE: CApaRegistrationResourceFileWriter::MainResourceInCompiledFormatL
void MainResourceRegL(const SAppDetails& aAppDetails, MDataSink& aDataSink)
	{
	// LONG reserved_long = 0
	WriteLittleEndianUint32L(aDataSink, 0);

	// LLINK reserved_llink = 0
	WriteLittleEndianUint32L(aDataSink, 0);
	
	// LTEXT app_file(KMaxFileNameLength) = ""
	WriteTextL(aDataSink, aAppDetails.iAppFile);

	// LONG attributes = 0
	const TUint ENonNative = 0x4;
	WriteLittleEndianUint32L(aDataSink, ENonNative);

	// LTEXT localisable_resource_file(KMaxFileNameLength) = ""
	TFileName locResName;
	_LIT(KLocResNameFormat, "%S:\\private\\10003a3f\\import\\apps\\NonNative\\Resource\\%08x_loc.rsc");
	locResName.Format(KLocResNameFormat, &aAppDetails.iDrive, aAppDetails.iUid);
	WriteTextL(aDataSink, locResName);

	// LONG localisable_resource_id = 1
	WriteLittleEndianUint32L(aDataSink, 1);

	// BYTE hidden = KAppNotHidden
	WriteUint8L(aDataSink, 0);

	// BYTE embeddability = KAppNotEmbeddable
	WriteUint8L(aDataSink, 0);

	// BYTE newfile = KAppDoesNotSupportNewFile
	WriteUint8L(aDataSink, 0);

	// BYTE launch = KAppLaunchInForeground
	WriteUint8L(aDataSink, 0);

	// LTEXT group_name(KAppMaxGroupName) = ""
	WriteTextL(aDataSink, KNullDesC);

	// BYTE default_screen_number = 0
	WriteUint8L(aDataSink, 0);

	// LEN WORD STRUCT datatype_list[]
	WriteLittleEndianUint16L(aDataSink, 0);

	// LEN WORD STRUCT file_ownership_list[]
	WriteLittleEndianUint16L(aDataSink, 0);

	// LEN WORD STRUCT service_list[]
	WriteLittleEndianUint16L(aDataSink, 0);

	// LLINK opaque_data = 0
	WriteLittleEndianUint32L(aDataSink, 2);
	}

void MainResourceLocL(const SAppDetails& aAppDetails, MDataSink& aDataSink)
	{
	// LOCALISABLE_APP_INFO

	// LONG reserved_long = 0
	WriteLittleEndianUint32L(aDataSink, 0);

	// LLINK reserved_llink = 0
	WriteLittleEndianUint32L(aDataSink, 0);

	// LTEXT short_caption(KMaxCaption) = ""
	WriteTextL(aDataSink, KNullDesC);

	// STRUCT caption_and_icon

	// LONG reserved_long = 0
	WriteLittleEndianUint32L(aDataSink, 0);

	// LLINK reserved_llink = 0
	WriteLittleEndianUint32L(aDataSink, 0);

	// LTEXT caption(KMaxCaption) = ""
	WriteTextL(aDataSink, aAppDetails.iCaption);

	// WORD number_of_icons = 0
	WriteLittleEndianUint16L(aDataSink, 1);

	// LTEXT icon_file(KMaxFileNameLength) = ""
	TFileName iconFile;
	_LIT(KIconFileFormat, "%S:\\private\\10003a3f\\import\\apps\\NonNative\\Resource\\[%08x].mbm");
	iconFile.Format(KIconFileFormat, &aAppDetails.iDrive, aAppDetails.iUid);
	WriteTextL(aDataSink, iconFile);
	
	// LEN WORD STRUCT view_list[]
	WriteLittleEndianUint16L(aDataSink, 0);

	//	LTEXT group_name(KAppMaxGroupName) = ""
	WriteTextL(aDataSink, aAppDetails.iGroupName);
	}

void CreateResourceL(RFs& aFs, const SAppDetails& aAppDetails, TBool aRegFile)
	{
	// SOURCE: CWidgetRegistrationManager::RegisterWidgetL
    TBuf8<KMaxFileName> opaqueData;
    RDesWriteStream writeStream( opaqueData );
    writeStream.WriteUint32L(aAppDetails.iUid.iUid);
    writeStream.WriteUint32L(aAppDetails.iAppFile.Length());
    writeStream.WriteL(aAppDetails.iAppFile);
    writeStream.CommitL();
	// End of widget specific stuff
    
	// SOURCE: CApsRegisterNonNativeApplication::WriteResourceFileL
	RFileWriteStream targetStream;
	CleanupClosePushL(targetStream);

	// Generate filename
	TFileName filename;
	if (aRegFile)
		filename.Format(_L("%S%08x_reg.rsc"), &aAppDetails.iResultsDir, aAppDetails.iUid);
	else
		filename.Format(_L("%S%08x_loc.r%S"), &aAppDetails.iResultsDir, aAppDetails.iUid, &(aAppDetails.iLanguage));
	RDebug::Print(_L("Generating: %S\n"), &filename);
		
	RFile file;
	User::LeaveIfError(file.Replace(aFs, filename, EFileShareAny|EFileWrite));
	targetStream.Attach(file);
	
	// SOURCE: CApsNonNativeApplicationsManager::DoRegisterNonNativeApplicationL
	if (aRegFile)
		{
		const TUidType uidPrefix(TUid::Uid(KUidPrefixedNonNativeRegistrationResourceFile), aAppDetails.iAppTypeUid, aAppDetails.iUid);
		targetStream.WriteL(TCheckedUid(uidPrefix).Des());
		}

	RBuf8 buf;
	buf.CreateL(20);
	CleanupClosePushL(buf);

	MDataSink sink(buf);
	
	// SOURCE: CApaResourceFileWriterBase::DoGenerateFileContentsL
	if (aRegFile)
		WriteUidTypeL(sink, TUid::Uid(KUidAppRegistrationFile), aAppDetails.iUid);
	else
		WriteUidTypeL(sink, TUid::Null(), TUid::Null());
		
	WriteUint8L(sink, 0); // flags
	
	// Create a "sink" to contain the "main" resource data
	// We need to know the size of this before it's written into the resource file
	RBuf8 res;
	res.CreateL(20);
	CleanupClosePushL(res);
	MDataSink mainSink(res);
	
	if (aRegFile)
		MainResourceRegL(aAppDetails, mainSink);
	else
		MainResourceLocL(aAppDetails, mainSink);
		
	WriteLittleEndianUint16L(sink, Max(mainSink.Length(), opaqueData.Length())); // size of larges resource
	WriteUint8L(sink, 0); // compression bit flags

	// Write the main data followed by opaque data
	WriteBufferL(sink, res);
	CleanupStack::PopAndDestroy(&res);
	if (aRegFile)
		WriteBufferL(sink, opaqueData);
	
	TInt filePosition = 16 + 1 + 2 + 1;
	WriteLittleEndianUint16L(sink, filePosition);
	filePosition += mainSink.Length();
	WriteLittleEndianUint16L(sink, filePosition);
	if (aRegFile)
		{
		filePosition += opaqueData.Length();
		WriteLittleEndianUint16L(sink, filePosition);
		}
	
	targetStream.WriteL(buf);
	targetStream.CommitL();

	CleanupStack::PopAndDestroy(2, &targetStream);
	}

TInt CharToHex(TChar c)
	{
	if (c>='0' && c<='9')
		return c - TChar('0');
	else if (c>='A' && c<='F')
		return c - TChar('A') + TChar(10);
	else if (c>='a' && c<='f')
		return c - TChar('a') + TChar(10);
	return -1;
	}

TUint ReadHex(const TDesC& aStr)
	{
	TUint x = 0;
	for (TInt pos = 0; pos < aStr.Length(); pos++)
		{
		TInt digit = CharToHex(aStr[pos]);
		if (digit < 0)
			break;
		x = (x<<4) | ((TUint)digit);
		}
	return x;
	}

void doShowHelp()
	{
	_LIT(KHelp, "\
		\nWidgetRegFiles.exe version 18-10-2008.\
		\nMake Symbian non-native application registration files.\
		\nCopyright (c) 1998-2008 Symbian Software Ltd. All rights reserved.");
	_LIT(KUsage, "\
		\n\nUsage:\
		\n WidgetRegFiles.exe <ini file specifying details> - generate registration files\
		\n WidgetRegFiles.exe <reg file> <loc file> - dump details of registration files");
	_LIT(KIniFormat, "\
		\n\nIni File Format Specifying Details:\n\
		\n [app_registration_info]\
		\n uid=12345678\
		\n app_file=app.exe\
		\n caption=MyApp\
		\n drive=Z\n");

	RDebug::Print(KHelp);
	RDebug::Print(KUsage);
	RDebug::Print(KIniFormat);
	}

void doMainL()
	{
	CCommandLineArguments* cmd = CCommandLineArguments::NewLC();

	// ini file supplied?
	if (cmd->Count() == 2)
		{
		RFs fs;
		User::LeaveIfError(fs.Connect());
		CleanupClosePushL(fs);

		// Open the ini file
		TPtrC iniFile(cmd->Arg(1));
		BSUL::CIniFile16* ini;
		TRAPD(err, ini = BSUL::CIniFile16::NewL(fs, iniFile, ETrue));
		if (err == KErrNotFound)
			{
			RDebug::Print(_L("ERROR: failed to open %S\n"), &iniFile);
			exit(2);
			}
		User::LeaveIfError(err);
		CleanupStack::PushL(ini);
		
		_LIT(KSection, "app_registration_info");		
		SAppDetails appDetails;
		
		// Get app_file
		TPtrC value;		
		if (ini->FindVar(KSection(), _L("app_file"), value) != KErrNone)
			{
			RDebug::Print(_L("ERROR: app_file not specified in %S\n"), &iniFile);
			exit(2);			
			}
		appDetails.iAppFile = value;

		// Get uid - must be specified
		if (ini->FindVar(KSection(), _L("uid"), value) != KErrNone || value.Length() == 0)
			{
			RDebug::Print(_L("ERROR: uid not specified in %S\n"), &iniFile);
			exit(2);			
			}
		appDetails.iUid = TUid::Uid(ReadHex(value));
		
		// Get caption
		if (ini->FindVar(KSection(), _L("caption"), value) != KErrNone)
			{
			RDebug::Print(_L("ERROR: caption not specified in %S\n"), &iniFile);
			exit(2);			
			}
		appDetails.iCaption = value;
		
		// Get drive - or assume ROM
		if (ini->FindVar(KSection(), _L("drive_name"), value) == KErrNone && value.Length() > 0)
			appDetails.iDrive = value.Left(1);
		else
			appDetails.iDrive = _L("Z");
		
		// Get app_type - or else assume widgets
		if (ini->FindVar(KSection(), _L("app_type"), value) == KErrNone && value.Length() > 0)
			appDetails.iAppTypeUid = TUid::Uid(ReadHex(value));
		else
			appDetails.iAppTypeUid = TUid::Uid(KUidWidgetLauncher);

		// Get group name - can be blank
		if (ini->FindVar(KSection(), _L("group"), value) == KErrNone)
			appDetails.iGroupName = value;

		// Get the directory where files should be generated
		if (ini->FindVar(KSection(), _L("results_dir"), value) == KErrNone)
			{
			appDetails.iResultsDir = value;
			if (appDetails.iResultsDir[value.Length() - 1] != '\\')
				appDetails.iResultsDir.Append('\\');
			}
		
		// Generate default version of resource
		appDetails.iLanguage = _L("sc");

		// Now create the files
		CreateResourceL(fs, appDetails, ETrue);
		CreateResourceL(fs, appDetails, EFalse);

		// Generate the language versions of the files if required
		if (ini->FindVar(KSection(), _L("languages"), value) == KErrNone)
			{
			HBufC* languages = value.AllocLC();
			TLex lex(*languages);
			while(!lex.Eos())
				{
				// Get the language code
				lex.SkipCharacters();			
				appDetails.iLanguage = lex.MarkedToken();
				
				// Get the localised caption
				TBuf<32> key;
				key.Format(_L("caption%S"), &(appDetails.iLanguage));
				if (ini->FindVar(KSection(), key, value) == KErrNone)
					{
					// Generate localised resource file
					appDetails.iCaption = value;
					CreateResourceL(fs, appDetails, EFalse);
					}
				lex.SkipSpace();
				lex.Mark();
				}
			CleanupStack::PopAndDestroy(languages);
			}
		CleanupStack::PopAndDestroy(2, &fs);
		}
	// Files to analyse?
	else if (cmd->Count() == 3)
		{
		TPtrC reg(cmd->Arg(1));
		TPtrC loc(cmd->Arg(2));
		
		RDebug::Print(_L("Dumping reg %S and loc %S\n"), &reg, &loc);
		DumpRegFileL(reg, loc);
		}
	// Incorrect arguments - show help
	else
		{
		RDebug::Print(_L("ERROR: Invalid arguments\n"));
		doShowHelp();
		exit(2);
		}
	
	CleanupStack::PopAndDestroy(cmd);
	}

int E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* theCleanup = CTrapCleanup::New();
	TRAPD(ret, doMainL());
	if (ret < KErrNone)
		User::Panic(_L("LEAVE"), ret);
	delete theCleanup;
	__UHEAP_MARKEND;
	return ret;
	}
