/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  This file contains Hardcoded module implementation.
*
*/


#include <Stiftestinterface.h>
#include <SettingServerClient.h>
#include "cmmgrbc.h"


// -----------------------------------------------------------------------------
// CCmmgrBc::CCmmgrBc
// C++ default constructor can NOT contain any code, that
// might leave.
// -----------------------------------------------------------------------------
//
CCmmgrBc::CCmmgrBc()
    {
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::ConstructL
// Symbian 2nd phase constructor can leave.
//
// Note: If OOM test case uses STIF Logger, then STIF Logger must be created
// with static buffer size parameter (aStaticBufferSize). Otherwise Logger
// allocates memory from heap and therefore causes error situations with OOM
// testing. For more information about STIF Logger construction, see STIF Users
// Guide.
// -----------------------------------------------------------------------------
//
void CCmmgrBc::ConstructL()
    {
    // Read logger settings to check whether test case name is to be appended
    // to log file name.
    RSettingServer settingServer;
    CleanupClosePushL( settingServer );

    TInt ret = settingServer.Connect();
    if ( ret != KErrNone )
        {
        User::Leave( ret );
        }

    // Parse StifLogger defaults from STIF initialization file.
    TLoggerSettings loggerSettings;
    ret = settingServer.GetLoggerSettings( loggerSettings );
    if ( ret != KErrNone )
        {
        User::Leave( ret );
        }

    CleanupStack::Pop( &settingServer );
    settingServer.Close(); // Close Setting server session.
    iAddTestCaseTitleToLogName = loggerSettings.iAddTestCaseTitle;
    iVersionLogged = EFalse;

    iStdLog = CStifLogger::NewL( KCmmgrBcLogPath, KCmmgrBcLogFile );
    iLog = iStdLog;

    _LIT( KLogStr, "cmmgrbc.dll - Connection Settings API BC tests" );
    iLog->Log( KLogStr );
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::NewL
// Two-phased constructor.
// -----------------------------------------------------------------------------
//
CCmmgrBc* CCmmgrBc::NewL()
    {
    CCmmgrBc* self = new (ELeave) CCmmgrBc;

    CleanupStack::PushL( self );
    self->ConstructL();
    CleanupStack::Pop( self );

    return self;
    }

// Destructor
CCmmgrBc::~CCmmgrBc()
    {
    iLog = NULL;

    delete iStdLog;
    iStdLog = NULL;

    delete iTCLog;
    iTCLog = NULL;
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::InitL
// InitL is used to initialize the Test Module.
// -----------------------------------------------------------------------------
//
TInt CCmmgrBc::InitL(
    TFileName& /*aIniFile*/,
    TBool /*aFirstTime*/ )
    {
    _LIT( KLogStr, "cmmgrbc.dll - Initialized" );
    iLog->Log( KLogStr );
    return KErrNone;
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::GetTestCasesL
// GetTestCases is used to inquire test cases from the Test Module. Test cases
// are stored to array of test cases. The Test Framework will be the owner of
// the data in the RPointerArray after GetTestCases return and it does the
// memory deallocation.
// -----------------------------------------------------------------------------
//
TInt CCmmgrBc::GetTestCasesL(
        const TFileName& /*aConfig*/,
        RPointerArray<TTestCaseInfo>& aTestCases )
    {
    // Loop through all test cases and create new TTestCaseInfo items and
    // append items to aTestCase array.
    for ( TInt i = 0; Case(i).iMethod; i++ )
        {
        // Allocate new TTestCaseInfo from heap for a testcase definition.
        TTestCaseInfo* newCase = new (ELeave) TTestCaseInfo();

        // PushL TTestCaseInfo to CleanupStack.
        CleanupStack::PushL( newCase );

        // Set number for the testcase.
        // When the testcase is run, this comes as a parameter to RunTestCaseL.
        newCase->iCaseNumber = i;

        // Set title for the test case. This is shown in UI to user.
        newCase->iTitle.Copy( Case(i).iCaseName );

        // Append TTestCaseInfo to the testcase array. After appended
        // successfully the TTestCaseInfo object is owned (and freed)
        // by the TestServer.
        User::LeaveIfError( aTestCases.Append( newCase ) );

        // Pop TTestCaseInfo from the CleanupStack.
        CleanupStack::Pop( newCase );
        }

    _LIT( KLogStr, "cmmgrbc.dll - Loaded %d test cases" );
    iLog->Log( KLogStr, aTestCases.Count() );
    return KErrNone;
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::RunTestCaseL
// RunTestCaseL is used to run an individual test case specified by aCaseNumber.
// Test cases that can be run may be requested from Test Module by GetTestCases
// method before calling RunTestCase.
// -----------------------------------------------------------------------------
//
TInt CCmmgrBc::RunTestCaseL(
        const TInt aCaseNumber,
        const TFileName& /*aConfig*/,
        TTestResult& aResult )
    {
    if ( !iVersionLogged )
        {
        SendTestModuleVersion();
        iVersionLogged = ETrue;
        }

    // Return value
    TInt execStatus = KErrNone;

    // Get the pointer to test case function
    TCaseInfo tmp = Case( aCaseNumber );

    _LIT( KLogStartTC, "Starting testcase [%S]" );
    iLog->Log( KLogEmptyLine );
    iLog->Log( KLogStartTC, &tmp.iCaseName );

    // Check that case number was valid
    if ( tmp.iMethod )
        {
        // Open new log file with test case title in file name
        if ( iAddTestCaseTitleToLogName )
            {
            // Delete test case logger if exists
            if ( iTCLog )
                {
                delete iTCLog;
                iTCLog = NULL;
                }

            TFileName logFileName;
            TName title;
            TestModuleIf().GetTestCaseTitleL( title );

            logFileName.Format( KCmmgrBcLogFileWithTitle, &title );

            iTCLog = CStifLogger::NewL( KCmmgrBcLogPath, logFileName );
            iLog = iTCLog;
            }

        // Valid case was found, call it via function pointer
        iMethod = tmp.iMethod;
        TRAPD( err, execStatus = ( this->*iMethod )( aResult ) );
        if ( iAddTestCaseTitleToLogName )
            {
            // Restore standard log and destroy test case logger
            iLog = iStdLog;
            delete iTCLog; //Close test case log
            iTCLog = NULL;
            }
        User::LeaveIfError( err );
        }
    else
        {
        // Valid case was not found, return error.
        execStatus = KErrNotFound;
        }

    // Return case execution status (not the result of the case execution).
    return execStatus;
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::OOMTestQueryL
// Used to check if a particular test case should be run in OOM conditions and
// which memory allocations should fail.
// -----------------------------------------------------------------------------
//
TBool CCmmgrBc::OOMTestQueryL(
        const TFileName& /*aTestCaseFile*/,
        const TInt aCaseNumber,
        TOOMFailureType& /*aFailureType*/,
        TInt& aFirstMemFailure,
        TInt& aLastMemFailure )
    {
    aFirstMemFailure = Case( aCaseNumber ).iFirstMemoryAllocation;
    aLastMemFailure = Case( aCaseNumber ).iLastMemoryAllocation;
    return Case( aCaseNumber ).iIsOOMTest;
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::OOMTestInitializeL
// Used to perform the test environment setup for a particular OOM test case.
// Test Modules may use the initialization file to read parameters for Test
// Module initialization but they can also have their own configure file or
// some other routine to initialize themselves.
//
// NOTE: User may add implementation for OOM test environment initialization.
// Usually no implementation is required.
// -----------------------------------------------------------------------------
//
void CCmmgrBc::OOMTestInitializeL(
        const TFileName& /*aTestCaseFile*/,
        const TInt /*aCaseNumber*/ )
    {
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::OOMHandleWarningL
// In some cases the heap memory allocation should be skipped, either due to
// problems in the OS code or components used by the code being tested, or even
// inside the tested components which are implemented this way on purpose (by
// design), so it is important to give the tester a way to bypass allocation
// failures.
//
// NOTE: User may add implementation for OOM test warning handling. Usually no
// implementation is required.
// -----------------------------------------------------------------------------
//
void CCmmgrBc::OOMHandleWarningL(
        const TFileName& /*aTestCaseFile*/,
        const TInt /*aCaseNumber*/,
        TInt& /*aFailNextValue*/ )
    {
    }

// -----------------------------------------------------------------------------
// CCmmgrBc::OOMTestFinalizeL
// Used to perform the test environment cleanup for a particular OOM test case.
//
// NOTE: User may add implementation for OOM test environment finalization.
// Usually no implementation is required.
// -----------------------------------------------------------------------------
//
void CCmmgrBc::OOMTestFinalizeL(
        const TFileName& /*aTestCaseFile*/,
        const TInt /*aCaseNumber*/ )
    {
    }

//-----------------------------------------------------------------------------
// CCmmgrBc::SendTestModuleVersion
// Method used to send version of test module
//-----------------------------------------------------------------------------
//
void CCmmgrBc::SendTestModuleVersion()
    {
    #ifdef CMMGRBC_S60_032_SUPPORT
    TVersion moduleVersion;
    moduleVersion.iMajor = TEST_MODULE_VERSION_MAJOR;
    moduleVersion.iMinor = TEST_MODULE_VERSION_MINOR;
    moduleVersion.iBuild = TEST_MODULE_VERSION_BUILD;

    _LIT( KTestDllName, "cmmgrbc.dll" );
    TFileName moduleName;
    moduleName = KTestDllName;

    TBool newVersionOfMethod = ETrue;
    TestModuleIf().SendTestModuleVersion( moduleVersion, moduleName, newVersionOfMethod );
    #endif // CMMGRBC_S60_032_SUPPORT
    }

//-----------------------------------------------------------------------------
// CCmmgrBc::SetTestCaseResult
// Method used to set test case result according to error count.
//-----------------------------------------------------------------------------
//
TInt CCmmgrBc::SetTestCaseResult( TTestResult& aResult, const TInt aErrorCount )
    {
    _LIT( KLogCaseEnd, "Done, error count %d" );
    iLog->Log( KLogCaseEnd, aErrorCount );

    if ( aErrorCount != 0 )
        {
        _LIT( KDescription, "Errors detected" );
        aResult.SetResult( KErrGeneral, KDescription );
        }
    else
        {
        _LIT( KDescription, "All OK" );
        aResult.SetResult( KErrNone, KDescription );
        }
    return KErrNone;
    }

//-----------------------------------------------------------------------------
// CCmmgrBc::MyDelay
// A simple wait. Used to avoid possible CommsDat locked situations.
// If no argument is given, waits according to the value defined in
// KCmmgrbcDefaultDelayInMs
//-----------------------------------------------------------------------------
//
void CCmmgrBc::MyDelay( const TInt aDelay )
    {
#ifdef CMMGRBC_ENABLE_DELAYS
    User::After( aDelay * KCmmgrbcMicroSecondsInMilliSecond );
#endif
    }

// ========================== OTHER EXPORTED FUNCTIONS =========================

// -----------------------------------------------------------------------------
// LibEntryL is a polymorphic Dll entry point
// Returns: CTestModuleBase*: Pointer to Test Module object
// -----------------------------------------------------------------------------
//
EXPORT_C CTestModuleBase* LibEntryL()
    {
    return CCmmgrBc::NewL();
    }

// -----------------------------------------------------------------------------
// SetRequirements handles test module parameters (implements evolution version
// 1 for test module's heap and stack sizes configuring).
// Returns: TInt: Symbian error code.
// -----------------------------------------------------------------------------
//
EXPORT_C TInt SetRequirements(
        CTestModuleParam*& /*aTestModuleParam*/,
        TUint32& /*aParameterValid*/ )
    {
    /* --------------------------------- NOTE ---------------------------------
    USER PANICS occurs in test thread creation when:
    1) "The panic occurs when the value of the stack size is negative."
    2) "The panic occurs if the minimum heap size specified is less
       than KMinHeapSize".
       KMinHeapSize: "Functions that require a new heap to be allocated will
       either panic, or will reset the required heap size to this value if
       a smaller heap size is specified".
    3) "The panic occurs if the minimum heap size specified is greater than
       the maximum size to which the heap can grow".
    Other:
    1) Make sure that your hardware or Symbian OS is supporting given sizes.
       e.g. Hardware might support only sizes that are divisible by four.
    ------------------------------- NOTE end ------------------------------- */

    // Normally STIF uses default heap and stack sizes for test thread, see:
    // KTestThreadMinHeap, KTestThreadMaxHeap and KStackSize.
    // If needed heap and stack sizes can be configured here by user. Remove
    // comments and define sizes.

    /*
    aParameterValid = KStifTestModuleParameterChanged;

    CTestModuleParamVer01* param = CTestModuleParamVer01::NewL();
    // Stack size
    param->iTestThreadStackSize= 16384; // 16K stack
    // Heap sizes
    param->iTestThreadMinHeap = 4096;   // 4K heap min
    param->iTestThreadMaxHeap = 1048576;// 1M heap max

    aTestModuleParam = param;
     */
    return KErrNone;
    }

// End of file
