// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Demonstrates the Resource Manager's user side API, RBusDevResManUs

#include <e32cons.h>
#include <d32resmanus.h>

_LIT(KTxtExampleCode,"PRM Example");
_LIT(KTxtFailed,"\nFailed with error : %d \n");
_LIT(KTxtPressAnyKey," \nPress any key to continue\n");
_LIT(KEnterResourceIdValue,"\nEnter the resource ID - any number other than zero: ");
_LIT(KTxtStateChange,"\nState of the resource after the change: %d");
_LIT(KPrintNoOfClients,"\nNumber of clients is %d \n");
_LIT(KTxtOptionNotSupported,"\nThis option is not supported\n");
/**
Clients of the resource manager are identified by name.
The client name is passed to the RBusDevResManUs::Open() and RBusDevResManUs::GetNumResourcesInUseByClient() methods.
It is the clients responsibility to ensure the name is unique.
*/
_LIT8(KClient,"resourcemanager");
TClientName gName(KClient);

//Indicates whether kernel-side (as well as user-side) clients are included in the request.
const TBool gIsKernelClient = EFalse;

//Holds the resource ID selected by the user
TUint gResourceId;

/**
Resource Manager user-side class, Clients first need to open a channel with this
class in order to use it.
*/
RBusDevResManUs gChannel;

CConsoleBase* gConsole;

TUint gNumClients = 0; //The number of resource manager clients
TUint gNumResources = 0; //The number of available resources

/**
The resource state. This could be a binary value for a binary resource,
an integer level for a multilevel resource or some platform specific token
for a multi-property resource.
*/
TInt gReadValue = 0;

/**
Takes input from the console and returns the TUint value. 
*/
TUint GetUserInput()
	{
	TBufC<50> buf;
	TPtr ptr(buf.Des());
	TKeyCode ch = gConsole->Getch();
	while(ch != EKeyEnter)
		{
		_LIT(KChar, "%c");
		gConsole->Printf(KChar,ch);
		if(ch!= EKeyBackspace)
			{
			ptr.Append(ch);
			}
		ch=gConsole->Getch();   			
		}
	TUint value;

	TUint err = TLex(ptr).Val(value, EDecimal);
	
	if(value == 0)
		{
		gConsole->Printf(KTxtOptionNotSupported);
		return (TUint)KErrNotSupported;;
		}
	else
		{
		if (err == (TUint)KErrNone)
			{
			return value;
			}
		else 
			{
			return (TUint)KErrNotSupported;; // Error condition - could not parse input to number.
			}
		}
	}

/**
Gets and prints the number of resources.
*/
void GetNoOfResources()
	{
	_LIT(KTxtOption,"\nGet the number of resources \n");
	gConsole->Printf(KTxtOption);
	TInt err=gChannel.GetNoOfResources(gNumResources);
	if(err != KErrNone)
		{
		gConsole->Printf(KTxtFailed,err);
		}
	else
		{
		_LIT(KPrintNoOfResources,"\nNumber of resources = %d \n");
		gConsole->Printf(KPrintNoOfResources,gNumResources);
		}
	}

/**
Gets information about a resource selected by the user.
It prints the name and ID of the resource.
*/
void GetResourceInfo()
	{
	_LIT(KTxtOption,"\nGet information about a particular resource...\n");
	gConsole->Printf(KTxtOption);
	//Request the user to select a resource
	gConsole->Printf(KEnterResourceIdValue);
	gResourceId = GetUserInput(); 
	if(gResourceId != (TUint)KErrNotSupported)
		{
		/*
		The name of the resource is held in an 8 bit buffer. In order to print this value as a string, 
		it must be converted to a 16 bit buffer. One character is left to hold the 'Null' character to mark the end of
		the string.
		*/ 
		TBuf16<MAX_RESOURCE_NAME_LENGTH+1>name; //maximum resource length defined in d32resmanus.h 
		// retrieve information about the selected resource
		TResourceInfoBuf *info = new TResourceInfoBuf();
		TInt err=gChannel.GetResourceInfo(gResourceId,info);
		if(err!= KErrNone)
			{
			gConsole->Printf(KTxtFailed,err);
			}
		else
			{
			name.Copy((*info)().iName); //Copy the name of the resource into the Unicode buffer.
			name.PtrZ(); //Append a zero terminator to the buffer.
			_LIT(KPrintResourceInfo,"\nResource id = %d \t Resource name = %S \n");
			gConsole->Printf(KPrintResourceInfo,gResourceId ,&name);
			}
		delete info;
		}
	}
	
/**
Gets and prints the names and IDs of all the available resources.
*/
void GetAllResourcesInfoL()
	{
	_LIT(KTxtOption,"\nGet information about all the resources...\n");
	gConsole->Printf(KTxtOption);
	//Find the number of resources available
	GetNoOfResources();
	/*
	Construct an array of TResourceInfoBuf objects.
	TResourceInfo holds all the information about a resource.
	*/
	TInt err;
	RSimplePointerArray<TResourceInfoBuf> resPtrs(gNumResources);
	for(TUint i=0;i<gNumResources;i++)
		{
		TResourceInfoBuf* info = new TResourceInfoBuf;
		if((err=resPtrs.Insert(info, i))!=KErrNone)
			{
			gConsole->Printf(KTxtFailed,err);
			}
		}
	//Retrieve information about all the available resources
	if((err=gChannel.GetAllResourcesInfo(&resPtrs,gNumResources))!=KErrNone)
		{
		gConsole->Printf(KTxtFailed,err);
		}
	/*
	The name of the resource is held in an 8 bit buffer. In order to print this value as a string, 
	it must be converted to a 16 bit buffer. One character is left to hold the 'Null' character to mark the end of
	the string.
	*/ 
	TBuf16<MAX_RESOURCE_NAME_LENGTH+1>name;
	for(TUint i=0; i<gNumResources; i++)
		{
		TResourceInfoBuf* currentInfo = resPtrs[i];
		name.Copy((*currentInfo)().iName);//Copy the name of the resource into the Unicode buffer.
		name.PtrZ();//Append a zero terminator to the buffer.
		_LIT(KPrintResourceInfo,"Resource id = %d, name = %S \n");
		gConsole->Printf(KPrintResourceInfo,(*currentInfo)().iId,&name);
		if(i%5 == 0 && i!= 0) //Print 5 resources at a time.
			{
			gConsole->Printf(KTxtPressAnyKey);
			TChar tchar = gConsole->Getch();
			}
		delete currentInfo;
		}
	resPtrs.Close(); //Close the R class
	}

/**
Gets the current resource state.
*/
void GetResourceState()
	{
	_LIT(KTxtOption,"\nGet the state of a particular resource... \n");
	gConsole->Printf(KTxtOption);
	//Enter a resource value.
	gConsole->Printf(KEnterResourceIdValue);
	gResourceId = GetUserInput(); 
	if(gResourceId != (TUint)KErrNotSupported)
		{
		TRequestStatus getResourceStatus;
		/**
		Gets the value of the state of the resource.

		gResourceId - the ID of the resource.

		gReadValue - Updated with the resource state if the operation was successful.
	
		levelOwnerId - Will be updated with the current owner of the resource 
		if the operation was successful. If no client owns the resource then this will 
		contain -1.

		isCached - If ETrue, a cached value will be updated in gReadValue.
		*/
		TBool isCached = EFalse;
		TInt levelOwnerId = 0;
		gChannel.GetResourceState(getResourceStatus,gResourceId,isCached,&gReadValue,&levelOwnerId);
		User::WaitForRequest(getResourceStatus);
		if(getResourceStatus.Int() != KErrNone)
			{
			gConsole->Printf(KTxtFailed,getResourceStatus.Int());
			}
		else
			{
			_LIT(KPrintResourceState,"\nResource state is %d\n");
			gConsole->Printf(KPrintResourceState,gReadValue);
			_LIT(KTxtLevelOwnerId,"\nId of the client which owns this resource is %d");
			gConsole->Printf(KTxtLevelOwnerId,levelOwnerId);
			}
		}
	}

/**
Gets the number of clients using a resource
*/
void GetNumClientsUsingResource()
	{
	_LIT(KTxtOption,"\nGet the number of clients using a particular resource... \n");
	gConsole->Printf(KTxtOption);
	GetNoOfResources(); //Get the total number of resources available
	//Ask the user to select a resource.
	gConsole->Printf(KEnterResourceIdValue);
	gResourceId = GetUserInput(); 
	if(gResourceId != (TUint)KErrNotSupported)
		{
		_LIT(KPrintNoOfClientsUsingResource,"\nNumber of  clients using resource : %d\n");
		gNumClients = 0; //This should be initialised to '0' otherwise the API returns KErrArgument
		TInt err=gChannel.GetNumClientsUsingResource(gResourceId, gNumClients, gIsKernelClient);
		if((err == KErrArgument && gNumClients == 0) || err == KErrNone)
			{
			gConsole->Printf(KPrintNoOfClientsUsingResource,gNumClients);
			}
		else
			{
			gConsole->Printf(KTxtFailed,err);
			}
		}
	}

/**
Gets the number of resources in use by the client
*/
void GetNumResourcesInUseByClient()
	{
	_LIT(KTxtOption,"\nGet the number of resources in use by a particular client \n");
	gConsole->Printf(KTxtOption);
	_LIT(KTxtClientName,"Name of the client used in this option is resourcemanager ");
	gConsole->Printf(KTxtClientName);
	//Gets the number of resources used by the client, by passing in the client's name.
	TInt err=gChannel.GetNumResourcesInUseByClient(gName, gNumResources, gIsKernelClient);
	if(err!=KErrNone)
		{
		gConsole->Printf(KTxtFailed,err);
		}
	else
		{
		_LIT(KPrintNumResourcesInUseByClient,"\nNumber of resources in use by client :  %d\n");
		gConsole->Printf(KPrintNumResourcesInUseByClient,gNumResources);
		}
	}
	
/**
Gets information about all resources in use by a particular client
*/
void GetInfoOnResourcesInUseByClientL()
	{
	_LIT(KTxtOption,"\nGet information about all the resources in use by client... \n");
	gConsole->Printf(KTxtOption);

	//Get number of resources in use by the client.
	GetNumResourcesInUseByClient();

	if(gNumResources!= 0)
		{
		/*
		Construct an array of TResourceInfoBuf objects.
		TResourceInfo holds all the information about a resource.
		*/
		RSimplePointerArray<TResourceInfoBuf> resPtrs(gNumResources);

		for(TUint i=0;i<gNumResources;i++)
			{
			TResourceInfoBuf *info = new TResourceInfoBuf();
			TInt err=resPtrs.Insert(info, i);
			if(err!= KErrNone)
				{
				gConsole->Printf(KTxtFailed,err);
				}
			}
		//Gets information about all the resources
		TInt err=gChannel.GetInfoOnResourcesInUseByClient(gName, gNumResources, &resPtrs);
		if(err!= KErrNone)
			{
			gConsole->Printf(KTxtFailed,err);
			}
		else
			{
			/*
			The name of the resource is held in an 8 bit buffer. In order to print this value as a string, 
			it must be converted to a 16 bit buffer. One character is left to hold the 'Null' character to mark the end of
			the string.
			*/ 
			TBuf16<MAX_RESOURCE_NAME_LENGTH+1>name;
			_LIT(KPrintResourceInfo,"\nResource %d name = %S \n");
			for(TUint i=0; i<gNumResources; i++)
				{
				TResourceInfoBuf* currRes = resPtrs[i];
				name.Copy((*currRes)().iName);//Copy the name of the resource into the Unicode buffer.
				name.PtrZ();//Append a zero terminator to the buffer.
				gConsole->Printf(KPrintResourceInfo,i,&name);
				if(i%5 == 0) //Print 5 resources at a time.
					{
					gConsole->Printf(KTxtPressAnyKey);
					TChar tchar = gConsole->Getch();
					}
				}
			}
		resPtrs.Close();
		}
	}

/**
Gets information about all clients using a particular resource
*/
void GetInfoOnClientsUsingResourceL()
	{
	_LIT(KTxtOption,"\nGet information about all clients using a particular resource... \n");
	gConsole->Printf(KTxtOption);

	//Get the number of clients using a particular resource
	GetNumClientsUsingResource();

	if(gNumClients!= 0)
		{
		//Construct an array of TClientInfoBuf objects. TClientInfo consists of the client name and ID.
		RSimplePointerArray<TClientInfoBuf> resPtrs(gNumClients);
		for(TUint i=0;i<gNumClients;i++)
			{
			TClientInfoBuf *info = new TClientInfoBuf();
			TInt err=resPtrs.Insert(info, i);
			if(err!= KErrNone)
				{
				gConsole->Printf(KTxtFailed,err);
				}
			}
		TInt err=gChannel.GetInfoOnClientsUsingResource(gResourceId, gNumClients, &resPtrs, gIsKernelClient);
		if(err!= KErrNone)
			{
			gConsole->Printf(KTxtFailed,err);
			}
		else
			{
			//Print information about all the clients using this resource.
			_LIT(KPrintClientName,"Client name %d = %S, ID=%d\n");
			for(TUint i=0;i<gNumClients;i++)
				{
				TClientInfoBuf* currInfoBuf = resPtrs[i];
				TClientInfo currInfo=(*currInfoBuf)();
				TBuf16<sizeof(TClientName)> name;
				name.Copy(currInfo.iName);
				gConsole->Printf(KPrintClientName,i,&name,currInfo.iId);
				}
			}
		resPtrs.Close();
		}
	}

/**
Gets the names of all the clients
*/
void GetNamesOfAllClientsL()
	{
	_LIT(KTxtOption,"\nGet the names of all clients... \n");

	gConsole->Printf(KTxtOption);

	//Gets the number of clients
	TInt err=gChannel.GetNoOfClients(gNumClients, gIsKernelClient);

	//Creates an array of client names
	RSimplePointerArray<TClientName> resPtrs(gNumClients);
	if(err != KErrNone)
		{
		gConsole->Printf(KTxtFailed,err);
		}
	else
		{
		gConsole->Printf(KPrintNoOfClients,gNumClients);
		for(TUint i=0;i<gNumClients;i++)
			{
			TClientName *info = new TClientName();
			if((err=resPtrs.Insert(info, i))!= KErrNone)
				{
				gConsole->Printf(KTxtFailed,err);
				}
			}
		}

	// Gets the names of all clients
	if((err=gChannel.GetNamesAllClients(&resPtrs, gNumClients, gIsKernelClient)) != KErrNone)
		{
		gConsole->Printf(KTxtFailed,err);
		}
	else
		{
		_LIT(KPrintNamesOfAllClients,"\nNames of all clients\n");
		_LIT(KPrintClientName,"Client name : %S\n");
		gConsole->Printf(KPrintNamesOfAllClients);
		for(TUint i=0;i<gNumClients;i++)
			{
			TClientName *currName = resPtrs[i];
			TBuf16<sizeof(TClientName)> name;
			name.Copy(*currName);
			gConsole->Printf(KPrintClientName,&name);
			delete currName;
			}
		}
	resPtrs.Close();
	}

/**
Changes the state of a resource
*/
TInt ChangeStateOfResource()
	{
	_LIT(KTxtOption,"\nChange the state of a particular resource... \n");
	gConsole->Printf(KTxtOption);

	//Gets the current state of the resource
	GetResourceState();
	TInt newLevel = gReadValue + 2; //new (random) state value
	TRequestStatus changeResourceStatus;
	gChannel.ChangeResourceState(changeResourceStatus,gResourceId,newLevel);
	User::WaitForRequest(changeResourceStatus);
	if(changeResourceStatus.Int() != KErrNone)
		{
		_LIT(KTxtChangeOfStateFailed,"\nChange of state failed with return value %d\n");
		gConsole->Printf(KTxtChangeOfStateFailed,changeResourceStatus.Int());
		}
	else
		{
		gConsole->Printf(KTxtStateChange,newLevel);
		}
	return changeResourceStatus.Int();
	}

/**
Requests a notification when there is a change in the resource's state
*/
void RequestNotification()
	{
	_LIT(KTxtOption,"\nRequest resource manager to notify the client about changes to the state of a resource... \n");
	gConsole->Printf(KTxtOption);

	//Verify whether ChangeResourceState() is supported. If not, then notification cannot be issued
	TInt ret = ChangeStateOfResource();
	if(ret == KErrNone)
		{
		TRequestStatus notificationStatus;
		//Requests notification for any change in the resource state
		gChannel.RequestNotification(notificationStatus, gResourceId);
		TInt newLevel = gReadValue + 2;

		//Change the resource state.
		TRequestStatus changeResourceStatus;
		gChannel.ChangeResourceState(changeResourceStatus,gResourceId,newLevel);
		User::WaitForRequest(changeResourceStatus);
		gConsole->Printf(KTxtStateChange,newLevel);
		User::WaitForRequest(notificationStatus);		
		if(notificationStatus.Int() != KErrNone)
			{
			_LIT(KTxtNotificationFailed,"\nRequest Notification for any change in the state of a resource failed with %d");
			gConsole->Printf(KTxtNotificationFailed,notificationStatus.Int());
			}
		else
			{
			
			_LIT(KTxtNotificationSuccess,"\nRequest Notification for any change in the state of a resource is successful");
			gConsole->Printf(KTxtNotificationSuccess);
			}
		}
	else
		{
		_LIT(KTxtNotificationNotSupported,"User notification for this resource is not supported");
		gConsole->Printf(KTxtNotificationNotSupported);
		}
	}
	
void GetPowerResourceInfoL()
	{
	_LIT(KPrintSubMenu,"GetPowerResourceInfo was chosen\n");
	_LIT(KPrintSubMenu1," \nDemonstrate the Get Power Resource APIs\n \
		*****************************************************\n \
		Option 1:GetNoOfResources\n   \
		Option 2:GetResourceInfo\n   \
		Option 3:GetAllResourcesInfo\n \
		Option 4:GetResourceState\n ");
	_LIT(KPrintSubMenu2, "\t \tOption 5:GetNumClientsUsingResource \n \
		Option 6:GetNumResourcesInUseByClient\n \
		Option 7:GetInfoOnResourcesInUseByClient\n \
		Option 8:GetInfoOnClientsUsingResource\n  \
		Option 9:GetNamesOfAllClients\n \
		******************************************************\n " );
	gConsole->ClearScreen();
	gConsole->Printf(KPrintSubMenu);
	_LIT(KTxtPressMainESC," \nPress 'ESC' key to go back to the Main Menu\n");
	gConsole->Printf(KTxtPressMainESC);
	gConsole->Printf(KPrintSubMenu1);
	gConsole->Printf(KPrintSubMenu2);
	_LIT(KChooseSubMenuOption,"\nSelect an option 1 - 9 : ");
	gConsole->Printf(KChooseSubMenuOption);
	TChar tchar = gConsole->Getch();	
	while (tchar != EKeyEscape)
		{		
		switch(tchar)
			{		
			case '1':
				{
				GetNoOfResources();
				break;
				}
			case '2':
				{
				GetResourceInfo();
				break;
				}
			case '3':
				{
				GetAllResourcesInfoL();
				break;
				}
			case '4':
				{
				GetResourceState();
				break;
				}
			case '5':
				{
				GetNumClientsUsingResource();
				break;
				}
			case '6':
				{
				GetNumResourcesInUseByClient();
				break;
				}
			case '7':
				{
				GetInfoOnResourcesInUseByClientL();
				break;
				}
			case '8':
				{
				GetInfoOnClientsUsingResourceL();
				break;
				}
			case '9':
				{
				GetNamesOfAllClientsL();
				break;
				}
			default:
				{
				if(tchar == EKeyEscape)
					{
					tchar = 'a'; //Any value can be assigned, so that it goes back to the main menu
					}
				else
					{
					gConsole->Printf(KTxtOptionNotSupported);
					}
				break;
				}
			}
		gConsole->Printf(KTxtPressAnyKey);
		tchar = gConsole->Getch();	
		gConsole->Printf(KTxtPressMainESC);
		gConsole->Printf(KPrintSubMenu1);
		gConsole->Printf(KPrintSubMenu2);
		gConsole->Printf(KChooseSubMenuOption);	
		tchar = gConsole->Getch();		
		}
	}

void DoStartL()
	{
	TInt err= gChannel.Open(gName); // 'gName' contains the name of client
	_LIT(KPrintEnterPRMExample,"Power Resource Manager example\n");
	gConsole->Printf(KPrintEnterPRMExample);
	if (err!=KErrNone)
		{
		_LIT(KTxtCommonFailure,"\n Note: The application cannot demonstrate the features without a defined PSL \
		        (Platform Specific Layer) and the necessary ldd and pdd. \n");
		gConsole->Printf(KTxtCommonFailure);
		gConsole->Printf(KTxtFailed, err);
		_LIT(KTxtPressAnyKeyToCloseApp,"Press any key to close the application ");
		gConsole->Printf(KTxtPressAnyKeyToCloseApp);
		gConsole->Getch();
		}
	else
		{
		const TUint8 KNoOfGetStateRequests = 5; //The maximum number of get state asynchronous requests
		const TUint8 KNoOfSetStateRequests = 4; //The maximum number of set state asynchronous requests
		const TUint8 KNoOfNotifyRequests = 7;  //The maximum number of notification requests
		/**
		The maximum total number of state requests is at least one greater than the largest of 
		KNoOfGetStateRequests, KNoOfSetStateRequests, KNoOfNotifyRequests).
		
		KErrUnderFlow is returned if the number of State requests exceeds the maximum value defined 
		during initialisation.
		
		*/
		err=gChannel.Initialise(KNoOfGetStateRequests,KNoOfSetStateRequests,KNoOfNotifyRequests);
		if (err!=KErrNone)
			{
			gConsole->Printf(KTxtFailed, err);
			}
		_LIT(KTxtPressAnyKey,"Press any key to continue");
		gConsole->Printf(KTxtPressAnyKey);
		_LIT(KTxtPressESC," \nPress the 'ESC' key to exit\n");
		gConsole->Printf(KTxtPressESC);
		_LIT(KPrintMainMenu,"\nOptions for various Power Resource Manager Features \n \
			**************************************\n \
			1:GetPowerResourceInfo \n \
			2:ChangeStateOfResource \n \
			3:RequestNotification \n \
			**************************************\n \n");
		TChar tchar = gConsole->Getch();	
		//make a choice
		while (tchar != EKeyEscape)
			{
			gConsole->Printf(KPrintMainMenu);
			_LIT(KChooseMainMenuOption,"Select option 1 - 3 : ");
			gConsole->Printf(KChooseMainMenuOption);
			tchar = gConsole->Getch();			
			switch(tchar)
				{
				case '1':
					{	
					GetPowerResourceInfoL();
					break;
					}
				case '2':
					{
					ChangeStateOfResource();
					break;
					}
				case '3':
					{
					RequestNotification();
					break;
					}
				default:
					{
					if(tchar == EKeyEscape)
						{
						_LIT(KPrintExitMenu,"\nExit the Menu\n");
						gConsole->Printf(KPrintExitMenu);
						}
					else
						{
						gConsole->Printf(KTxtOptionNotSupported);
						}
					break;
					}
				}			
			}
		gChannel.Close();
		}
	}
void callExampleL() // Initialise and call example code under cleanup stack.
	{
	gConsole=Console::NewL(KTxtExampleCode,TSize(KConsFullScreen,KConsFullScreen));
	CleanupStack::PushL(gConsole);
	DoStartL(); // Perform example function.
	CleanupStack::PopAndDestroy(gConsole); // delete the gConsole.
	}

extern TInt E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* cleanup=CTrapCleanup::New(); // Create clean-up stack.
	if(cleanup!=NULL)
		{
		TRAPD (error,callExampleL());
		__ASSERT_ALWAYS(!error,User::Panic(KTxtExampleCode,error));
		}
	delete cleanup; // Delete clean-up stack.
	__UHEAP_MARKEND;
	return KErrNone;
	}
