/*
 * Copyright  2008 Nokia Corporation.
 */

// INCLUDE FILES
#include <AknQueryDialog.h>
#include <t32wld.h>
#include <badesca.h>
#include <aknlists.h> 
#include <eikmenup.h>
#include <aknnotewrappers.h>
#include "TZLocalizerAppUi.h"
#include "TZLocalizerDialog.h" 
#include "TZLocalizerEngine.h"
#include <TZLocalizer.rsg>
#include "tzlocalizer.hrh"


#include <avkon.hrh>

// constants

const TInt KBufSize = 30;
const TInt KTextBufferLength = 128;
const TInt KArrayGranularity = 5;
const TInt KTimeNumWidth     = 2;

// literals for the ui
_LIT( KDeleteCity , "Delete city:" );
_LIT( KNoMatchingCity, "No matching city found!");
_LIT( KCityName , "City name:" );
_LIT( KErrorAddingCity, "Error while adding new city!");
_LIT( KCityAdded, "City added.");
_LIT( KCityRemoved, "City removed.");
_LIT( KNewCityName , "New city name:" );
_LIT( KChooseCityGroupID , "Choose city group ID"); 
_LIT( KChooseTimeZoneID,"Choose Time Zone ID");
_LIT( KCityLocalTime, "\tCity local time:\t" );
_LIT( KCityGroupID, "\tCity group ID:\t");
_LIT( KCityTimeZoneID, "\tCity time zone ID:\t");
_LIT( KCityName2,"\tCity name:\t" );


// ---------------------------------------------------------
//  CTZLocalizerAppUi::ConstructL()
//  
// ---------------------------------------------------------
void CTZLocalizerAppUi::ConstructL()
    {
    BaseConstructL(EAknEnableSkin);

    // Create engine
    iTzEngine = CTZLocalizerEngine::NewL();

    // Start dialog
    iAppDialog = new (ELeave) CTZLocalizerDialog;
    iAppDialog->SetMopParent( this );
    iAppDialog->ExecuteLD( R_TZLOCALIZER_DIALOG );
    }

// ----------------------------------------------------
// CTZLocalizerAppUi::~CTZLocalizerAppUi()
// Destructor
// Frees reserved resources
// ----------------------------------------------------
CTZLocalizerAppUi::~CTZLocalizerAppUi()
    {    
    if (iAppDialog)
        {
        delete iAppDialog;
        }
    if(iTzEngine)
        {
        delete iTzEngine;
        }
    }
// ----------------------------------------------------
//  CTZLocalizerAppUi::DynInitMenuPaneL()
//  This function is called by the EIKON framework just before
//  it displays a menu pane. Its default implementation is empty, 
//  and by overriding it, the application can set the state of menu 
//  items dynamically according to the state of application data.
// ----------------------------------------------------
void CTZLocalizerAppUi::DynInitMenuPaneL( TInt /*aResourceId*/,CEikMenuPane* /*aMenuPane*/)
    {
    }

// ----------------------------------------------------
// CTZLocalizerAppUi::HandleKeyEventL()
// Standard key press information forwasion.
// ----------------------------------------------------
TKeyResponse CTZLocalizerAppUi::HandleKeyEventL( const TKeyEvent& aKeyEvent, 
                                              TEventCode aType )
    {
    if( aType!=EEventKey )
        {
        return EKeyWasNotConsumed;
        }

    switch( aKeyEvent.iCode )
        {
        case EKeyUpArrow:
        case EKeyDownArrow:
            {   
            if( iAppDialog != NULL )
                {
                TKeyResponse result = iAppDialog->OfferKeyEventL( aKeyEvent,
                    aType ); 
                return result;
                }
            } 
            break;

        default:
            break;
        }
    return EKeyWasNotConsumed;
    }

// ----------------------------------------------------
// CTZLocalizerAppUi::ShowInfoL()
// Briefly shows small information note which contain
// aText as message.
// ----------------------------------------------------
void CTZLocalizerAppUi::ShowInfoL( const TPtrC aText )
    {
    CAknInformationNote* infoNote = new (ELeave) CAknInformationNote();
    infoNote->ExecuteLD( aText );
    }

// ----------------------------------------------------
// CTZLocalizerAppUi::AskPopupChoiseL()
// Shows a popupmenu list containing aList information
// On return to aReturn is copied the chosen text and
// if aIndex != NULL the index # of the chosen item
// is copied to.
// ----------------------------------------------------
TBool CTZLocalizerAppUi::AskPopupChoiseL( const TPtrC aPrompt, 
                        CDesC16ArrayFlat* aList, TDes& aReturn )
    {
    TBool   rVal(EFalse);
    
    CEikTextListBox* list = new(ELeave) CAknSinglePopupMenuStyleListBox;
    CleanupStack::PushL(list);

   // Create popup list and PUSH it.
   CAknPopupList* popupList = CAknPopupList::NewL(
                                       list, R_AVKON_SOFTKEYS_OK_BACK,
                                       AknPopupLayouts::EMenuWindow);
   CleanupStack::PushL(popupList);

   // initialize listbox.
   
   list->ConstructL(popupList, CEikListBox::ELeftDownInViewRect);
   list->CreateScrollBarFrameL(ETrue);
   list->ScrollBarFrame()->SetScrollBarVisibilityL(
                               CEikScrollBarFrame::EOff,
                               CEikScrollBarFrame::EAuto);

    list->Model()->SetItemTextArray( aList );
    popupList->SetTitleL(aPrompt);

    //If the user enters ok on a selection
    if (popupList->ExecuteLD())
        {
        aReturn.Copy( (*aList)[list->CurrentItemIndex()] );
        
        rVal = ETrue;
    }
    CleanupStack::Pop(popupList);
    CleanupStack::PopAndDestroy(list);

    return rVal;
    }

// ----------------------------------------------------
// CTZLocalizerAppUi::QueryTextL()
// Shows a dialog with aPrompt note and waits for text 
// input which is then copied to aReturn 
// ----------------------------------------------------
TBool CTZLocalizerAppUi::QueryTextL( const TPtrC aPrompt, TDes& aReturn )
    {
    CAknTextQueryDialog* textQuery = CAknTextQueryDialog::NewL( aReturn );
    CleanupStack::PushL( textQuery );

    textQuery->SetPromptL( aPrompt );
    CleanupStack::Pop( textQuery );

    return textQuery->ExecuteLD( R_DATAQUERY_DATA_QUERY );
    }

// ----------------------------------------------------
// CTZLocalizerAppUi::HandleCommandL()
// Normal menu command handling
// ----------------------------------------------------
void CTZLocalizerAppUi::HandleCommandL( TInt aCommand )
    {
    switch ( aCommand )
        {
        case EAknSoftkeyExit:
        case EEikCmdExit:
            {
            Exit();
            break;
            }
    // ------------------------------------------------
    // Ask for (partial) city name and show the city
    // information in the lisbox.
        case ETZLocalizerCmdSearchCities:
            SearchCitiesL();
            break;

    // ------------------------------------------------
    // Add new city to world server database.
        case ETZLocalizerCmdAddCity:
            CreateNewCityL();
            break;
    // ------------------------------------------------
    // remove city from world server database
        case ETZLocalizerCmdRemoveCity:
            DeleteCityL();
            break;
            
    // Search Cities In Groups
        case ETZLocalizerCmdSearchCitiesInGroup:
            SearchCitiesInGroupsL();
            break;       
    // ------------------------------------------------
        default:
            break;      
        }
    }

// ---------------------------------------------------------
//  CTZLocalizerAppUi::FillListBoxL()
//  Fill main dialog listbox with city information
// ---------------------------------------------------------
void CTZLocalizerAppUi::FillListBoxL( CTzLocalizedCity* aLocalizedCity )
    {
    TBuf<KTextBufferLength>   formatBuffer;
    
    CDesC16ArrayFlat* cityInfo = 
        new (ELeave)CDesC16ArrayFlat( KArrayGranularity );
    CleanupStack::PushL( cityInfo );

    cityInfo->Reset();

    formatBuffer.Copy( KCityName2 );
    formatBuffer.Append( iTzEngine->GetCityName(aLocalizedCity) );
    cityInfo->AppendL( formatBuffer );

    formatBuffer.Copy( KCityTimeZoneID );
    TBuf<KBufSize> temp;
    temp.Num( iTzEngine->GetCityTimeZoneId(aLocalizedCity) );
    formatBuffer.Append( temp );
    cityInfo->AppendL( formatBuffer );

    formatBuffer.Copy( KCityGroupID );
    temp.Num( iTzEngine->GetCityGroupId(aLocalizedCity) );
    formatBuffer.Append( temp );
    cityInfo->AppendL( formatBuffer );
      
    formatBuffer.Copy( KCityLocalTime );
    HBufC* time;
    time = iTzEngine->GetCityLocalTimeL(aLocalizedCity);
    formatBuffer.Append( *time );
    
    cityInfo->AppendL( formatBuffer );

    if (iAppDialog)
        {
        iAppDialog->SetListBoxTextL( cityInfo );
        }
    
    delete time;
    CleanupStack::PopAndDestroy( cityInfo );
    }
// ---------------------------------------------------------
//  CTZLocalizerAppUi::FillListBoxL()
//  Fill main dialog listbox with city information
// ---------------------------------------------------------
void CTZLocalizerAppUi::FillListBoxL( CDesC16ArrayFlat* aArray )
    {
    if (iAppDialog)
        {
        iAppDialog->SetListBoxTextL( aArray );
        }
    }
    
// ---------------------------------------------------------
//  CTZLocalizerAppUi::ClearListBoxL()
//  Clear main dialog listbox
// ---------------------------------------------------------
void CTZLocalizerAppUi::ClearListBoxL()
    {
    CDesC16ArrayFlat* nothing = new (ELeave)CDesC16ArrayFlat(1);
    CleanupStack::PushL( nothing );    
    if (iAppDialog)
        {
        iAppDialog->SetListBoxTextL( nothing );
        }
    CleanupStack::PopAndDestroy(); // nothing       
    }

// ---------------------------------------------------------
//  CTZLocalizerAppUi::CreateCityUIL()
//  Ask all the information about new city and create it.
// ---------------------------------------------------------
void CTZLocalizerAppUi::CreateNewCityL()
    {
    TBuf<KBufSize> newCity;
    TBuf<KBufSize> timeZoneID;
    TBuf<KBufSize> groupID;
    TPtrC    text;

    // ask a new city name
    text.Set(KNewCityName);
    if( QueryTextL( text, newCity) == EFalse )
        {
        return;
        }
    
    CDesC16ArrayFlat* timeZones = iTzEngine->GetAllTimeZonesL();
    
    // choose time zone ID
    text.Set(KChooseTimeZoneID);
    if( AskPopupChoiseL( text, timeZones, timeZoneID ) == EFalse )
        {
        return;
        }
        
    CDesC16ArrayFlat* groupdIDs = iTzEngine->GetAllGroupIDL();
        
    // choose group zone ID
    text.Set(KChooseCityGroupID);
    if( AskPopupChoiseL( text, groupdIDs, groupID ) == EFalse )
        {
        return;
        }

    TInt tzID;
    TLex lexTimeZone(timeZoneID); 
    lexTimeZone.Val(tzID); 
    
    TInt grID;
    TLex lexGroup(groupID); 
    lexGroup.Val(grID); 

    CTzLocalizedCity* localizedCity = iTzEngine->AddCityL(tzID, newCity, grID);

    if( !localizedCity )
        {
        TPtrC msg(KErrorAddingCity);
        ShowInfoL( msg );
        }
    else
        {
        TPtrC msg(KCityAdded);
        ShowInfoL( msg );
        FillListBoxL( localizedCity );
        }
    delete localizedCity;
    }

// ---------------------------------------------------------
//  CTZLocalizerAppUi::SearchCitiesL()
//  Search a city and show it in the listbox
// ---------------------------------------------------------
void CTZLocalizerAppUi::SearchCitiesL()
    {
    
    TBuf<KTextBufferLength>    placeName;
              
    CTzLocalizedCity* localizedCity;  
    // ask for the city name
    TPtrC msg(KCityName);
    if( QueryTextL( msg, placeName) == false )
        {
        return;
        }
        
    // get the city matching the search string
    localizedCity = iTzEngine->FindCityL( placeName );
        
    if( !localizedCity )
        {
        TPtrC msg(KNoMatchingCity);
        ShowInfoL( msg );
        }
    else
        {
        FillListBoxL( localizedCity );
        }   
    delete localizedCity;
    }


// ---------------------------------------------------------
//  CTZLocalizerAppUi::DeleteCityL()
//  Delete city from world server database
// ---------------------------------------------------------
void CTZLocalizerAppUi::DeleteCityL()
    {
    CDesC16ArrayFlat* citiesList = iTzEngine->GetAllCitiesL();
    TBuf<KTextBufferLength> cityName;

    // ask for the city name with the list of all cities
    TPtrC msg(KDeleteCity);
    
    if( AskPopupChoiseL( msg, citiesList, cityName ) == EFalse )
        {
        return;
        }
        
    iTzEngine->RemoveCityL(cityName);
    
    // clear display and show "city removed" message
    ClearListBoxL();
    TPtrC msgRemoved(KCityRemoved);
    ShowInfoL( msgRemoved );
    }
    
// ---------------------------------------------------------
//  CTZLocalizerAppUi::SearchCitiesInGroupsL()
//  Search a city and show it in the listbox
// ---------------------------------------------------------
void CTZLocalizerAppUi::SearchCitiesInGroupsL()
    {
    TBuf<KBufSize> groupID;
    TPtrC    text;
    
    CDesC16ArrayFlat* groupIDs = iTzEngine->GetAllGroupIDL();
    
    // choose group zone ID
    text.Set(KChooseCityGroupID);
    if( AskPopupChoiseL( text, groupIDs, groupID ) == EFalse )
        {
        return;
        }
    
    TInt grID;
    TLex lexGroup(groupID); 
    lexGroup.Val(grID); 
    
    CDesC16ArrayFlat* citiest = iTzEngine->FindCitiesInGroupL(grID);
    FillListBoxL(citiest);
    
    citiest->Reset();
    delete citiest;
    }

void CTZLocalizerAppUi::HandleResourceChangeL( TInt aType )
    {
    CAknAppUi::HandleResourceChangeL(aType);
    
    if ( aType==KEikDynamicLayoutVariantSwitch )
        {
        if (iAppDialog)
            {
            TRect rect;
            AknLayoutUtils::LayoutMetricsRect(AknLayoutUtils::EMainPane,rect);
            iAppDialog->SetRect(rect);
            // NOTE: Remember to send KEikDynamicLayoutVariantSwitch also to dialog
            iAppDialog->HandleResourceChange( aType );
            }
        }   
    }
    
// End of File  
