/*
 * Copyright  2008 Nokia Corporation.
 */

#include <e32base.h>
#include <t32wld.h>
#include <badesca.h>

#include "TZLocalizerEngine.h"
const TInt KBufSize=30;

_LIT( KTab, "\t");
_LIT( KEmpty, " ");
_LIT(KDateTimeString, "%*E%*D%X%*N%*Y %1 %2 '%3 %H%:1%T%:2%S");

// ---------------------------------------------------------
//  CTZLocalizerEngine::NewL()
//  
// ---------------------------------------------------------
CTZLocalizerEngine* CTZLocalizerEngine::NewL()
    {
    CTZLocalizerEngine* self = CTZLocalizerEngine::NewLC();
    CleanupStack::Pop( self );
    return self;
    }

// ---------------------------------------------------------
//  CTZLocalizerEngine::NewLC()
//  
// ---------------------------------------------------------
CTZLocalizerEngine* CTZLocalizerEngine::NewLC()
    {
    CTZLocalizerEngine* self = new( ELeave ) CTZLocalizerEngine();
    CleanupStack::PushL( self );

    self->ConstructL();
    return self;
    }

// ---------------------------------------------------------
//  CTZLocalizerEngine::~CTZLocalizerEngine()
//  closes Time Zone server connection
// ---------------------------------------------------------
CTZLocalizerEngine::~CTZLocalizerEngine()
    {
    }

// ---------------------------------------------------------
//  CTZLocalizerEngine::CTZLocalizerEngine()
//  
// ---------------------------------------------------------
CTZLocalizerEngine::CTZLocalizerEngine()
    {
    }

// ---------------------------------------------------------
//  CTZLocalizerEngine::ConstructL()
//  Connect to time zone server. 
// ---------------------------------------------------------
void CTZLocalizerEngine::ConstructL()
    {
    }

// ---------------------------------------------------------
// TPtrC CTZLocalizerEngine::GetCityName(CTzLocalizedCity *aLocalizedCity)
// Get current city name.
// ---------------------------------------------------------
TPtrC CTZLocalizerEngine::GetCityName(CTzLocalizedCity *aLocalizedCity)
{
    return aLocalizedCity->Name();
}

// ---------------------------------------------------------
// CTZLocalizerEngine::GetCityTimeZoneId(CTzLocalizedCity *aLocalizedCity)
// Get current city time zone id.
// ---------------------------------------------------------
TUint16 CTZLocalizerEngine::GetCityTimeZoneId(CTzLocalizedCity *aLocalizedCity)
{
    return aLocalizedCity->TimeZoneId();
}

// ---------------------------------------------------------
// CTZLocalizerEngine::GetCityGroupId(CTzLocalizedCity *aLocalizedCity)
// Get current city group id.
// ---------------------------------------------------------
TUint8 CTZLocalizerEngine::GetCityGroupId(CTzLocalizedCity *aLocalizedCity)
{
    return aLocalizedCity->GroupId();
}

// ---------------------------------------------------------
// CTZLocalizerEngine::GetCityLocalTime()
// Convert universal time to local time.
// ---------------------------------------------------------
HBufC* CTZLocalizerEngine::GetCityLocalTimeL(CTzLocalizedCity *aLocalizedCity)
    {   
    RTz tzServer;
    User::LeaveIfError(tzServer.Connect());
    CleanupClosePushL(tzServer);
    
    TBuf<KBufSize> dateTimeString;
    TTime time;
    time.HomeTime();
    time.FormatL(dateTimeString, KDateTimeString);
    // this is a 2 stage process - first convert the date/time to UTC.
    // No need to specify the time zone because 
    // this function uses the current system time zone.
    TInt results=tzServer.ConvertToUniversalTime(time);
    
    //get the current time zone id
    CTzId* timezoneId = CTzId::NewL(GetCityTimeZoneId(aLocalizedCity));
    CleanupStack::PushL(timezoneId);
    // convert UTC time to local time
    results=tzServer.ConvertToLocalTime(time,*timezoneId);
    
    TBuf<KBufSize> timeZoneString;
    timeZoneString.Copy(timezoneId->TimeZoneNameID());
    time.FormatL(dateTimeString, KDateTimeString);
        
    CleanupStack::PopAndDestroy(timezoneId);
    
    HBufC* temp = HBufC::NewL(dateTimeString.Size());
    (*temp)= dateTimeString;
    
    CleanupStack::PopAndDestroy(1); // causes tzServer.Close() to be called

    return temp;        
    }

// ---------------------------------------------------------
// CTZLocalizerEngine::FindCityL()
// Find the specific city.
// return localizedcity
// ---------------------------------------------------------
CTzLocalizedCity* CTZLocalizerEngine::FindCityL(const TDesC& aCityName )
    {
    CTzLocalizer* loc = CTzLocalizer::NewL(); 
    CleanupStack::PushL(loc); 

    CTzLocalizedCity* localizedCity = loc->FindCityByNameL(aCityName);
    
    CleanupStack::PopAndDestroy(loc); 

    return localizedCity;   
    }
    
// ---------------------------------------------------------
//  CTZLocalizerEngine::GetAllTimeZonesL()
//  Fetch the valid time zone ID's
// ---------------------------------------------------------
CDesC16ArrayFlat* CTZLocalizerEngine::GetAllTimeZonesL()
    {
    CTzLocalizer* loc = CTzLocalizer::NewL(); 
    CleanupStack::PushL(loc); 

    CTzLocalizedCityArray* locCityArray = 
        loc->GetCitiesL(loc->ETzUTCAscending);
    CleanupStack::PushL(locCityArray); 
    
    CDesC16ArrayFlat* timeZones = 
        new (ELeave)CDesC16ArrayFlat( locCityArray->Count() );
        
    TBuf<KBufSize> temp;
    TInt result;
    TInt k = 0;
    for(TInt i=0; i<locCityArray->Count();i++)
        {
        result = 0;
        //get the time zone id
        temp.Num((locCityArray->At(i)).TimeZoneId());
        result = timeZones->Find(temp, k);
        //add if not allready exist
        if(result!=0)
            {
            timeZones->AppendL(temp);
            }           
        }

    timeZones->Sort();

    CleanupStack::PopAndDestroy(2); //loc and locCityArray  
    return timeZones;
    }    
    
// ---------------------------------------------------------
// CTZLocalizerEngine::GetAllGroupID()
// Fetch the valid city group ID's
// ---------------------------------------------------------
CDesC16ArrayFlat* CTZLocalizerEngine::GetAllGroupIDL()
    {
    CTzLocalizer* loc = CTzLocalizer::NewL(); 
    CleanupStack::PushL(loc); 

    CTzLocalizedCityArray* locCityArray = 
        loc->GetCitiesL(loc->ETzUTCAscending);
    CleanupStack::PushL(locCityArray); 
    
    CDesC16ArrayFlat* groupIDs = 
        new (ELeave)CDesC16ArrayFlat( locCityArray->Count() );
        
    TBuf<KBufSize> temp;
    TInt result;
    TInt k = 0;
    for(TInt i=0; i<locCityArray->Count();i++)
        {
        result = 0;
        //get the group id
        temp.Num((locCityArray->At(i)).GroupId());
        result = groupIDs->Find(temp, k);
        //add if not allready exist
        if(result!=0)
            {
            groupIDs->AppendL(temp);    
            }       
        }
        
    groupIDs->Sort();

    CleanupStack::PopAndDestroy(2); //loc and locCityArray  
    return groupIDs;
    }      

// ---------------------------------------------------------
// CDesC16ArrayFlat* CTZLocalizerEngine::GetAllCitiesL()
// Fetch the localized cities
// ---------------------------------------------------------
CDesC16ArrayFlat* CTZLocalizerEngine::GetAllCitiesL()
    {
    CTzLocalizer* loc = CTzLocalizer::NewL(); 
    CleanupStack::PushL(loc); 

    CTzLocalizedCityArray* locCityArray = 
        loc->GetCitiesL(loc->ETzAlphaNameAscending);
    CleanupStack::PushL(locCityArray); 
    
    CDesC16ArrayFlat* locCity = 
        new (ELeave)CDesC16ArrayFlat( locCityArray->Count() );

    for(TInt i=0; i<locCityArray->Count();i++)
        {
        locCity->AppendL((locCityArray->At(i)).Name()); 
        }   

    CleanupStack::PopAndDestroy(2); //loc and locCityArray

    return locCity;
    }


// ---------------------------------------------------------
//  CTZLocalizerEngine::AddCityL()
//  Add the city in database.
//  return localized city
// ---------------------------------------------------------    
CTzLocalizedCity* CTZLocalizerEngine::AddCityL(TInt aTimeZoneId, 
    const TDesC &aCityName, TInt aGroupId)
{
    CTzLocalizer* loc = CTzLocalizer::NewL(); 
    CleanupStack::PushL(loc);
    CTzLocalizedCity* localizedCity;
    TRAPD(error,
        localizedCity = loc->AddCityL(aTimeZoneId, aCityName, aGroupId));
    
    if(error!=KErrNone)
        {
        localizedCity = NULL;
        }
    CleanupStack::PopAndDestroy(loc); 
    
    return localizedCity;
}


// ---------------------------------------------------------
// CTZLocalizerEngine::FindCitiesInGroupL()
// Find the cities with same group id.
// ---------------------------------------------------------
CDesC16ArrayFlat* CTZLocalizerEngine::FindCitiesInGroupL( TInt aGroupID )
    {   
    CTzLocalizer* loc = CTzLocalizer::NewL(); 
    CleanupStack::PushL(loc); 
    
    CTzLocalizedCityArray* locCityArray
        = loc->GetCitiesInGroupL(aGroupID, loc->ETzAlphaNameAscending);
    CleanupStack::PushL(locCityArray); 
    
    CDesC16ArrayFlat* locCity = 
        new (ELeave)CDesC16ArrayFlat( locCityArray->Count() );

    TBuf<KBufSize> temp;

    for(TInt i=0; i<locCityArray->Count();i++)
        {
        temp.Copy( KTab );
        temp.Append((locCityArray->At(i)).Name());
        temp.Append( KTab );
        temp.Append( KEmpty );
        locCity->AppendL(temp);
        }   

    CleanupStack::PopAndDestroy(2); //loc and locCityArray

    return locCity;
    }

// ---------------------------------------------------------
// CTZLocalizerEngine::RemoveCityL()
// Removes a city from the user defined city collection 
// ---------------------------------------------------------
void CTZLocalizerEngine::RemoveCityL(const TDesC& aCityName )
    {
    CTzLocalizer* loc = CTzLocalizer::NewL(); 
    CleanupStack::PushL(loc); 
    CTzLocalizedCity* localizedCity = FindCityL(aCityName);
    //Remove a city from the user defined collection
    loc->RemoveCityL(localizedCity);
    CleanupStack::PopAndDestroy(loc);
    }

//EOF
