/*
 * Copyright  2008 Nokia Corporation.
 */

#include <eikrted.h>  // CEikRichTextEditor
#include <txtrich.h>  // CRichText

#include "SMSExampleRTEContainer.h"

// Rich text editor starting position
#define KEditorPosition TPoint(0,0)

// Number of controls inside this container
const TInt KNumberOfControls = 1;

const TInt KNumberOfLines = 0;
const TInt KTextLimit = 128;

_LIT(KEventLog, "Event log: \n" );
_LIT(KTextLine, "---------------------------");

// ----------------------------------------------------------------------------
// Symbian OS 2 phase constructor.
// ----------------------------------------------------------------------------
CSMSExampleRTEContainer* CSMSExampleRTEContainer::NewL(const TRect& aRect)
    {
    CSMSExampleRTEContainer* self = CSMSExampleRTEContainer::NewLC(aRect);
    CleanupStack::Pop(self);
    return self;
    }

// ----------------------------------------------------------------------------
// Symbian OS 2 phase constructor.
// ----------------------------------------------------------------------------
CSMSExampleRTEContainer* CSMSExampleRTEContainer::NewLC(const TRect& aRect)
    {
    CSMSExampleRTEContainer* self = new (ELeave) CSMSExampleRTEContainer();
    CleanupStack::PushL(self);
    self->ConstructL(aRect);
    return self;
    }

// ---------------------------------------------------------
// Symbian OS two phased constructor.
// ---------------------------------------------------------
void CSMSExampleRTEContainer::ConstructL(const TRect& aRect)
    {
    CreateWindowL();

    iRTE = new (ELeave) CEikRichTextEditor();
    iRTE->SetContainerWindowL(*this);
    iRTE->ConstructL(this, KNumberOfLines, KTextLimit,
        CEikEdwin::EReadOnly | CEikEdwin::EAvkonDisableCursor,
        EGulFontControlAll, EGulNoSymbolFonts);
    iRTE->CreateScrollBarFrameL()->SetScrollBarVisibilityL(
      CEikScrollBarFrame::EOff, CEikScrollBarFrame::EOn);

    SetRect(aRect);
    ActivateL();

    DrawTextL(KEventLog);
    }

// ----------------------------------------------------------------------------
// Destructor.
// ----------------------------------------------------------------------------
CSMSExampleRTEContainer::~CSMSExampleRTEContainer()
    {
    delete iRTE;
    iRTE = 0;
    }

// ----------------------------------------------------------------------------
// Draw text without drawing a line break at the end.
// ----------------------------------------------------------------------------
void CSMSExampleRTEContainer::DrawTextWithoutCarriageL(const TDesC& aText)
    {
    // Set color to black
    iCharacterFormatMask.SetAttrib(EAttColor);
    iCharacterFormat.iFontPresentation.iTextColor = KRgbBlack;
    
    CRichText* text = iRTE->RichText();
    TInt textSize = text->DocumentLength();

    text->InsertL(textSize, aText);
    // Apply formatting
    text->ApplyCharFormatL(iCharacterFormat, iCharacterFormatMask,
                           textSize, aText.Length());
    iRTE->HandleTextChangedL();
    }

// ----------------------------------------------------------------------------
// Add text using a RichTextEditor
// ----------------------------------------------------------------------------
void CSMSExampleRTEContainer::DrawTextL(const TDesC& aText)
    {
    DrawTextWithoutCarriageL(aText);
    AddCarriageReturnL();
    iRTE->HandleTextChangedL();

    //Keep displaying the bottom of the screen
    iRTE->MoveCursorL(TCursorPosition::EFPageDown, EFalse);
    }

// ----------------------------------------------------------------------------
// Add one line break.
// ----------------------------------------------------------------------------
void CSMSExampleRTEContainer::AddCarriageReturnL()
    {
    CRichText* text = iRTE->RichText();
    text->InsertL(text->DocumentLength(), CEditableText::ELineBreak);
    }
    
// ----------------------------------------------------------------------------
// Draw using a RichtTextEditor. Set Underline off when finished drawing.
// ----------------------------------------------------------------------------
void CSMSExampleRTEContainer::DrawUnderlinedTextL(const TDesC& aText)
    {
    iCharacterFormatMask.SetAttrib(EAttFontUnderline);

    iCharacterFormat.iFontPresentation.iUnderline = EUnderlineOn;
    DrawTextWithoutCarriageL(aText);
    iCharacterFormat.iFontPresentation.iUnderline = EUnderlineOff;
    }

// ----------------------------------------------------------------------------
// Draw one line.
// ----------------------------------------------------------------------------
void CSMSExampleRTEContainer::DrawLineL()
    {
    DrawTextL(KTextLine);
    }

// ---------------------------------------------------------
// Called by framework when the view size is changed
// ---------------------------------------------------------
void CSMSExampleRTEContainer::SizeChanged()
    {
    TRect rect = Rect();
    TInt scrollbarWidth = GetScrollbarWidth();
    iRTE->SetExtent(KEditorPosition, TSize(rect.Width() - scrollbarWidth,
        rect.Height()));
    }

// ----------------------------------------------------------------------------
// Returns the width (breadth) of the scrollbar for the editor
// ----------------------------------------------------------------------------
TInt CSMSExampleRTEContainer::GetScrollbarWidth() const
    {
    TInt scrollbarWidth = iRTE->ScrollBarFrame()->
        ScrollBarBreadth(CEikScrollBar::EVertical);
    // If scrollbars are not drawn yet, the width remains zero. In this
    // case, an intentionally magical number is returned.
    if (scrollbarWidth == 0)
        {
        scrollbarWidth = 8;
        }
    return scrollbarWidth;
    }
    
TInt CSMSExampleRTEContainer::CountComponentControls() const
    {
    return KNumberOfControls;
    }

CCoeControl* CSMSExampleRTEContainer::ComponentControl(TInt aIndex) const
    {
    switch (aIndex)
        {
        case 0:
            return iRTE;
        default:
            return NULL;
        }
    }

void CSMSExampleRTEContainer::Draw(const TRect& aRect) const
    {
    CWindowGc& gc = SystemGc();
    gc.SetPenStyle(CGraphicsContext::ENullPen);
    gc.SetBrushColor(KRgbWhite);
    gc.SetBrushStyle(CGraphicsContext::ESolidBrush);
    gc.DrawRect(aRect);
    }

TKeyResponse CSMSExampleRTEContainer::OfferKeyEventL(const TKeyEvent& aKeyEvent,
    TEventCode aType)
    {
    if (iRTE)
        {
        return iRTE->OfferKeyEventL(aKeyEvent, aType);
        }
    else
        {
        return CCoeControl::OfferKeyEventL(aKeyEvent, aType);
        }
    }

// End of file
