/** 
 *  @file MsgQLib.h
 *  Description: Header file for MsgQLib
 *  Copyright (c) 2007 Nokia Corporation.
 *  This material, including documentation and any related
 *  computer programs, is protected by copyright controlled by
 *  Nokia Corporation.
 */

#ifndef MSGQLIB_H
#define MSGQLIB_H

#include <_ansi.h>

#ifdef __cplusplus
extern "C" {
#endif

/* MsgQLib sizes */
#define  MAX_MSG_LEN	   2048 

/* hash table parameters - table size must be a prime number  */
#define  MSGQ_TBL_SZ	   101

/* message send states */
#define MSG_Q_READY        0
#define MSG_Q_CLEANING	   1


#define NO_WAIT 			0
#define WAIT_FOREVER 		0xFFFF

#ifndef NULL
#define NULL 				0
#endif /* NULL */

/* errors */
#define KMsgQLibErr			 	0x100
#define KMsgQLibNoMemoryErr 	(KMsgQLibErr | 1) /* out of memory */
#define KMsgQLibQIdErr 			(KMsgQLibErr | 2) /* queue already in use or invalid */
#define KMsgQLibParamErr    	(KMsgQLibErr | 3) /* illegal parm(s)/val(s) */
#define KMsgQLibQFlushErr   	(KMsgQLibErr | 4) /* queue being flushed */
#define KMsgQLibQShortErr   	(KMsgQLibErr | 5) /* internal error:  inconsistency between #of msgs*/

#define  MSG_Q_FIFO       1
#define  MSG_Q_PRIORITY   2
#define  MSG_PRI_NORMAL   3
#define  MSG_PRI_URGENT   4
#define  SEM_Q_FIFO       5
#define  SEM_Q_PRIORITY   6

#define  OK               0
#define  ERROR           -1  

typedef  unsigned long    ULONG ;
typedef  unsigned short   ushort_t;

/**
 *
 * Creates a message queue with the argument passed
 *
 * @param qName   - queue name
 * @param maxMsgs - max messages in the queue
 * @param qOptions - message queue options
 *                        MSG_Q_FIFO
 *                        MSG_Q_PRIORITY
 * @param err - error code to be returned
 *
 * @return Returns status and error code
 */
IMPORT_C int MsgQCreate (ULONG qName, ULONG maxMsgs, ULONG qOptions, int* err);

/**
 *
 * This function deletes a message queue
 *
 * @param qName - queue name
 * @param err - error code to be returned
 *
 * @return Returns status and error code
 **/
IMPORT_C int MsgQDelete (ULONG qName, int* err);

/**
 *
 * This function sends a message with internal copy
 *
 * @param qName - queue name
 * @param msg - message to send
 * @param nBytes - length of message
 * @param priority - message priority
 * @param timeout - milliseconds to wait
 * @param err - error code to be returned
 *
 * @return Returns status and error code
 **/

IMPORT_C int MsgQSend (ULONG qName, char* msg, ULONG nBytes, ULONG priority, int timeout, int* err);

/**
 *
 * This function receives a message with internal copy
 *
 * @param qName - queue name
 * @param msg - buffer for received message
 * @param maxNBytes - length of buffer
 * @param timeout - milliseconds to wait
 * @param err - error code to be returned
 *
 * @return Returns status and error code
 **/

IMPORT_C int MsgQReceive (ULONG qName, char* msg, ULONG maxNBytes, int timeout, int* err);

/**
 *
 * This function checks how many messages are in a queue
 *
 * @param qName - queue name
 * @param err - error code to be returned
 *
 * @return Returns status and error code
 **/
IMPORT_C int MsgQCheck (ULONG qName, int* err);

/**
 * This function checks the maximum number of messages in a queue
 *
 * @param qName - queue name
 * @param err - error code to be returned
 *
 * @return Returns status and error code
 **/
IMPORT_C int MsgQMaxCheck (ULONG qName, int* err);

/**
 *
 * This function empties the specified queue
 *
 * @param qName - queue name
 * @param err - error code to be returned
 *
 * @return Returns status and error code
 **/
IMPORT_C int MsgQClean (ULONG qName, int* err);

#ifdef __cplusplus
}
#endif

#endif /* MSGQLIB_H */
