/** 
 *  @file MsgQInternals.cpp
 *  Description: Source file for MsgQLib
 *  Copyright (c) 2007 Nokia Corporation.
 *  This material, including documentation and any related
 *  computer programs, is protected by copyright controlled by
 *  Nokia Corporation.
 */

#include <stdlib.h>
#include "MsgQInternal.h"


/* Declaration and definition of Internal Functions and data structures */
MSGQ_INFO_PTR        MsgQInfo[MSGQ_TBL_SZ];
MSGQ_INFO_LIST_PTR   MsgQListHead = NULL;

inline unsigned int HashIndex(ULONG qName);

/*************** INTERNAL FUNCTIONS ******************************/

/***************************************************************************
*  InstallMsqQTable (qName, qid, semId, sendState, numMsgs, err)
*  Function: This function installs a queue into the hash table
****************************************************************************/

int InstallMsqQTable(ULONG qName, int qId, int semId, int* err) {

	MSGQ_INFO_PTR pMsgQInfo = NULL;
	unsigned int  index;

	if ((pMsgQInfo = MsgQTableLookup(qName)) == NULL) {
		pMsgQInfo = (MSGQ_INFO_PTR)malloc(sizeof(*pMsgQInfo));

		if(pMsgQInfo != NULL) {
			index = HashIndex(qName);

			pMsgQInfo->next  = MsgQInfo[index];
			MsgQInfo[index]   = pMsgQInfo;
			pMsgQInfo->qName = qName;
			pMsgQInfo->qId   = qId;
			pMsgQInfo->semId = semId;
			pMsgQInfo->sendState = MSG_Q_READY;
			pMsgQInfo->numMsgs   = 0;

			*err = OK;
			return (OK);
		}
		else
			*err = KMsgQLibNoMemoryErr;
	}
	else
		*err = KMsgQLibQIdErr;

	return(ERROR);

}


/******************************************************************************
*  HashIndex
*  Function: This function returns the hash index
*******************************************************************************/

inline unsigned int HashIndex(ULONG qName) {
	return(qName % MSGQ_TBL_SZ);
}


/************************************************************************
*  MsgQTableLookup (qName)
*  Function: This function finds the block pointer for each queue
*************************************************************************/

MSGQ_INFO* MsgQTableLookup(ULONG qName) {
	MSGQ_INFO_PTR pMsgQInfo = NULL;

	for (pMsgQInfo = MsgQInfo[HashIndex(qName)]; pMsgQInfo != NULL; pMsgQInfo = pMsgQInfo->next) {
		if (qName == pMsgQInfo->qName)
			return(pMsgQInfo);
	}

	return(NULL);
}


/*************************************************************************
*  RemoveFromMsqQTable (qName, err)
*  Function: This function removes a queue from the hash table
**************************************************************************/


int RemoveFromMsqQTable(ULONG qName, int* err) {
	unsigned int  index = 0;
	MSGQ_INFO_PTR prev = NULL;
	MSGQ_INFO_PTR pMsgQInfo = NULL;

	index = HashIndex(qName);
	for (pMsgQInfo = MsgQInfo[index]; pMsgQInfo != NULL; pMsgQInfo = pMsgQInfo->next) {
		if (qName == pMsgQInfo->qName)
			break;
		prev = pMsgQInfo;
	}

	if (pMsgQInfo != NULL) {
		if (prev == NULL)
			MsgQInfo[index] = pMsgQInfo->next;
		else
			prev->next = pMsgQInfo->next;

		free((void*)pMsgQInfo);
		*err = OK;
		return (OK);
	}
	else
		*err = KMsgQLibQIdErr;

	return(ERROR);
}


/************************************************************************
*  AddToMsgQTable (qName)
*  Function: Adding a queue to list
*************************************************************************/

void AddToMsgQTable(ULONG qName) {
	MSGQ_INFO_LIST_PTR tempNext;

	if (MsgQListHead != NULL) {
		/* subsequent entries */
		tempNext = MsgQListHead->next;

		if ((MsgQListHead->next = (MSGQ_INFO_LIST*)malloc(sizeof(MSGQ_INFO_LIST))) != NULL) 		{
			MsgQListHead->next->next = tempNext;
			MsgQListHead->next->qName = qName;
		}
		else 
			MsgQListHead->next = tempNext;
	}
	else {
		if ((MsgQListHead = (MSGQ_INFO_LIST*)malloc(sizeof(MSGQ_INFO_LIST))) != NULL) {
			MsgQListHead->next = NULL;
			MsgQListHead->qName = qName;
		}
	}
}

/************************************************************************
*  DeleteFromMsgQTable (qName)
*  Function:  removing a queu entry from  list
*************************************************************************/

void DeleteFromMsgQTable(ULONG qName) {
	MSGQ_INFO_LIST_PTR prev = NULL;
	MSGQ_INFO_LIST_PTR pMsgQInfo = NULL;

	for (pMsgQInfo = MsgQListHead; pMsgQInfo != NULL; pMsgQInfo = pMsgQInfo->next) {
		if (qName == pMsgQInfo->qName)
			break;
		prev = pMsgQInfo;
	}

	if (pMsgQInfo != NULL) {
		/* Check whether prev pointer is null or not. If it is Null, update Head pointer */
		if( prev == NULL )
			MsgQListHead = MsgQListHead->next ;

		/* Else update the linked list by removing present node and updating prev next pointer */
		else
			prev->next = pMsgQInfo->next;

		/* Now free up the memory used by the present node */
		free((void*) pMsgQInfo);
	}
}
