// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Demonstrates an OE hybrid application.
//



/**
 @file
*/
#include <stdio.h>
#include <f32file.h>
#include <stdlib.h>
#include <string.h>

/**
Converts a TDesC derived descriptor to a 'C' string.
@param aDes The descriptor to be converted.
@return A pointer to the character array that stores the converted string.
*/
char* DescToChar(const TDesC& aDes)
	{
	// Get the length of the descriptor.
	TInt length = aDes.Length();

	// Allocate memory to the array of characters.
	// An additional byte of memory is allocated to the array as 
	// Symbian descriptors do not end with the end of string '\0' character.
	char* ptr = (char*)malloc(sizeof(char)*(length + 1));
	if(ptr == NULL)
		{
		// Return NULL if memory was not allocated properly.
		return NULL;
		}
	// The loop index.
	TInt ix = 0;
	for(; ix < length; ix++)
		{
		// Copy the contents of the descriptor into the array of characters.
		ptr[ix] = aDes[ix];
		}
	// Append the end of string to the 'C' style string.
	ptr[ix] = '\0';

	// return the pointer to the array.
	return ptr;
	}

/**
Prints the memory information for a drive.
@param aFs A handle to the file server session.
@param aDriveNumber The drive number of the drive.
*/
void PrintDriveInfoL(const RFs& aFs, const TInt aDriveNumber)
	{
	// Get the drive information
	TDriveInfo driveInfo;
	TInt errDrive = aFs.Drive(driveInfo,aDriveNumber);
	if(errDrive != KErrNone)
		return;

	// Get the name of the drive.
	TBuf<KMaxFileName> driveName;
	TInt errName = aFs.GetDriveName(aDriveNumber,driveName);
	if(errName != KErrNone)
		return;

	// Convert the drive name from native Symbian descriptor string to a 'C' string.
	char* cptr = DescToChar(driveName);
	CleanupStack::PushL(cptr);
	// Check if a name has been specified for the drive.
	if(strlen(cptr))
		{
		// Print the name of the drive.
		printf("Drive Name :%s\n",cptr);
		}
	
	// Print the memory information for the drive.
	// The memory size is defined for a drive only if it is formattable.
	if((driveInfo.iMediaAtt & KMediaAttFormattable))
		{
		TChar driveLetter;
		// if drive-list entry non-zero, drive is available
		TInt errDLetter = aFs.DriveToChar(aDriveNumber,driveLetter);
		if(errDLetter != KErrNone)
			return;
		// The following line prints the drive letter followed by the hex value
		// of the integer indicating that drive's attributes
		printf("Drive Letter: %c\n",TUint(driveLetter));			
	
		// Get the volume information for the formatted device.
		TVolumeInfo volumeInfo;
		TInt errVol = aFs.Volume(volumeInfo,aDriveNumber);
		if(errVol != KErrNone)
			return;

		// Get the total size and the free space of the drive.
		TInt64 driveTotalSize = volumeInfo.iSize;
		TInt64 driveFreeSize =  volumeInfo.iFree;
		// Print the memory information for the drive.
		printf("Total size of the drive: %d\n", driveTotalSize);
		printf("Free space: %d\n", driveFreeSize);
		
		printf(" [press the enter key to continue]\n");
		// Wait for a key press.
		getchar();
		}
	
	CleanupStack::PopAndDestroy(cptr);
	}

LOCAL_C void MainL()
	{
	// The file server session.
	RFs fs;
	// Connect to the file server.
	User::LeaveIfError(fs.Connect());
	CleanupClosePushL(fs);
	printf("\nValid drives as characters (and as numbers) are:\n");

	// Initialise drive number to the first drive.
	TInt driveNumber=EDriveA;
	TChar driveLetter;

    for (; driveNumber<=EDriveZ; driveNumber++)
        {
        if (fs.IsValidDrive(driveNumber))
            {
			// Indicates that the drive exists and is valid.
            TInt errDrive = fs.DriveToChar(driveNumber,driveLetter);
            if(errDrive == KErrNone)
            	{
				// Print the drive letter.
	            printf("%c",TUint(driveLetter));
				// Convert the drive letter to the drive number.
	            TInt errChar = fs.CharToDrive(driveLetter, driveNumber);
	            if(errChar == KErrNone)
                	{
					// Print the drive number.
		            printf("(%d) ",driveNumber);
                	}
            	}
            }
        }

    printf("\n");

	// Get a list of the available drives.
    TDriveList drivelist;
    User::LeaveIfError(fs.DriveList(drivelist));

	// A TDriveList (the list of available drives), is an array of
    // 26 bytes. Each byte with a non zero value signifies that the
    // corresponding drive is available.

    printf("\nUsing DriveList(), available drives are: \n");
    for (driveNumber=EDriveA; driveNumber<=EDriveZ;driveNumber++)
        {
        if (drivelist[driveNumber])
            {
			// if drive-list entry non-zero, drive is available
            TInt errDrive = fs.DriveToChar(driveNumber,driveLetter);
            if(errDrive == KErrNone)
            	{
	            // The following line prints the drive letter followed by the hex value
	            // of the integer indicating that drive's attributes
	            printf("Drive Letter: %c\n",TUint(driveLetter));
				printf(" [press the enter key to continue]\n");
				// Wait for a key press.
				getchar();
            	}
            }
        }
    
	printf("Memory information for formattable drives:\n");
	printf(" [press the enter key to continue]\n");
	// Wait for a key press.
	getchar();
	for (driveNumber=EDriveA; driveNumber<=EDriveZ;driveNumber++)
        {
        if (drivelist[driveNumber])
            {
			PrintDriveInfoL(fs,driveNumber);
			}
		}	

	CleanupStack::PopAndDestroy(&fs);
	}


GLDEF_C TInt E32Main()
	{
	// Create cleanup stack
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();

	// Run application code inside TRAP harness, wait for key press when terminated
	TRAPD(mainError, MainL());
	if (mainError)
		{
		printf(" failed, leave code = %d", mainError);
		}

	printf(" [press the enter key to exit]\n");
	getchar();

	delete cleanup;
	__UHEAP_MARKEND;
	return KErrNone;
	}

