// Copyright (c) 2000-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

#include "filechangenotifier.h"

CChangeNotifier* CChangeNotifier::NewL(TInt aPriority)
    {
    CChangeNotifier* self=new(ELeave)CChangeNotifier(aPriority);
    CleanupStack::PushL(self);
    self->ConstructL();
    CleanupStack::Pop(self);
    return self;
    }

void CChangeNotifier::ConstructL()
    {
    //Create console
    _LIT(KWelcome, "Welcome to File Change Notifier Example\n");
    _LIT(KTextConsoleTitle, "Extended File Change Notifier");
    iConsole = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen));
    iConsole->Printf(KWelcome);
    
    // Initiate a session by connecting to the File Server
    User::LeaveIfError(iFs.Connect()); 
    // Buffer size for holding the results of completed notifications.
    const TInt KNotificationHeaderSize = (sizeof(TUint16)*2)+(sizeof(TUint));
    const TInt KMinNotificationBufferSize = 2*KNotificationHeaderSize + 2*KMaxFileName;
    
    // Get the system drive on which the file/folder operations can be done.
    int driveNumber = iFs.GetSystemDrive();
    iDrive = (TChar('A'+ driveNumber));
    
    // Create an object for 'CFsNotify' 
    iNotify= CFsNotify::NewL(iFs,KMinNotificationBufferSize);   
    }

CChangeNotifier::CChangeNotifier(TInt aPriority):CActive( aPriority )
    {   
    CActiveScheduler::Add(this);
    }

/**
Destructor
*/
CChangeNotifier::~CChangeNotifier()
    {
    if(iIsCreateFile)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\");
        path.Append(KPath);
                                
        TBuf<20> filename;
        _LIT(KFileName,"create.file");
        filename.Append(KFileName);
    
        TBuf<80> fullname;
        fullname.Append(path);
        fullname.Append(filename);                  
            
        iFs.Delete(fullname); 
        }
    if(iIsCreateFolder)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\createfolder\\");
        path.Append(KPath); 
        iFs.RmDir(path);
        }
    if(!iIsDeleteFile)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\deletefolder\\");
        path.Append(KPath);
                                    
        TBuf<20> filename;
        _LIT(KFileName,"file.txt");
        filename.Append(KFileName);
        
        TBuf<80> fullname;
        fullname.Append(path);
        fullname.Append(filename);                  
                
        iFs.Delete(fullname); 
        }
    if(iIsRenameFile)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\");
        path.Append(KPath);
                                    
        TBuf<20> filename;
        _LIT(KFileName,"NewFileName");
        filename.Append(KFileName);
        
        TBuf<80> fullname;
        fullname.Append(path);
        fullname.Append(filename);                  
                
        iFs.Delete(fullname); 
        }
    
    if(iIsRenameFolder)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\RenameFolder\\");
        path.Append(KPath);
        iFs.RmDir(path);
        }
    if(iIsDeleteFolder)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\deletefolder\\");
        path.Append(KPath);
        iFs.RmDir(path);
        }
    if(iIsVolumeNameChange)
        {
        TBuf<80> path;
        path.Append('U'); 
        path.Append(_L(":"));
        
        _LIT(KVolumeLabel,"MyDrive");  
        iFs.SetVolumeLabel(KVolumeLabel,iDriveNumber);
        }
    if(iIsMultipleNotifications)
        {      
        // delete file
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\");
        path.Append(KPath);                                
        TBuf<20> filename;
        filename.Append(_L("rename.file"));
        
        TBuf<80> fullname;
        fullname.Append(path);
        fullname.Append(filename);                  
                
        iFs.Delete(fullname); 
        }
    delete iConsole;
    delete iNotify;
    iFs.Close();
    }

// The RunL() is called by the Active scheduler.
void CChangeNotifier::RunL()
    {
    /*
     * When the user chooses an option which is not available in the list of supported features.
     * The Active scheduler is stopped and the application exits.
     */
    if(iOption)
        {
        _LIT(KNotSupported,"\nNot Supported\n");
        iConsole->Printf(KNotSupported);
        CActiveScheduler::Stop();
        }
    else
        {
        /*
         * When the user chooses a Sub-Menu option which is not among the listed options, then 
         * the list of Main options are once again displayed to the user.
         */
        if(iOptionContinue)
            {
            RequestCharacter();
            }
        if(iReqStatus == KErrNone)
            {
            _LIT(KSuccess,"\nSuccessfully completed the posted notifications\n");
            iConsole->Printf(KSuccess);
            if(!iDeleteFileContinue)
                {
                if(iIsMultipleNotifications)
                    {                           
                    //Get first notification
                    const TFsNotification* notification = iNotify->NextNotification();
                    TFsNotification::TFsNotificationType notificationType = ((TFsNotification*)notification)->NotificationType();

                    //Process Notification
                    ProcessMultipleChanges(notification);

                    //Get Next notification
                    notification = iNotify->NextNotification();

                    //The operation may not have occurred yet..
                    if(notification == NULL)
                        {
                        iNotify->RequestNotifications(iReqStatus);
                        User::WaitForRequest(iReqStatus); // We need wait for the corresponding file change to complete.
                        notification = iNotify->NextNotification();
                        }

                    //Process Notification
                    ProcessMultipleChanges(notification);            
                    }
                    RequestCharacter();
                }
            else
                {
                DeleteFolder();
                }
            }
        }
    }

/*
 * This method will process the multiple notifications which are completed.
 * Currently, the creation of the file and rename of the file are happening one after another in the DoChanges() Method
 * defined below. Hence the file to be renamed will be created first and then rename operation is done.
 */
void CChangeNotifier::ProcessMultipleChanges(const TFsNotification* aNotification)
    {
    TFsNotification::TFsNotificationType notificationType = ((TFsNotification*)aNotification)->NotificationType();

    if(notificationType == TFsNotification::ECreate) 
        { 
        _LIT(KNewFileCreated,"Notification of successful creation of a new file 'notify.file'\n");
        iConsole->Printf(KNewFileCreated);
        }
    else if (notificationType == TFsNotification::ERename)
        {
        _LIT(KFileRename,"Notification for notify.file being renamed to rename.file ");
        iConsole->Printf(KFileRename);
        }
    }
/*
 * User gets notified when there is a change in the file. Different kinds of notifications are posted by the 
 * user.
 * 
 * Notifications are recevied for following operations :
 * - Any change in the contents of the particular file.
 * - If any new file is created inside the particular folder.
 * - When a file rename is done.
 * - When the file attributes are changed.
 * - When a particular file is deleted.
 * 
 */
void CChangeNotifier::NotifyChangeToFile()
    {   
    _LIT(KMenuL1,"\nChoose an option:\n");
    _LIT(KMenuL2,"1.Get notified when a file is created\n");
    _LIT(KMenuL3,"2.Get notified when the file contents change\n");
    _LIT(KMenuL4,"3.Get notified when a file rename occurs\n");
    _LIT(KMenuL5,"4.Get notified when file attributes change\n");
    _LIT(KMenuL6,"5.Get notified when a file is deleted\n");
        
    iConsole->Printf(KMenuL1);
    iConsole->Printf(KMenuL2);
    iConsole->Printf(KMenuL3);
    iConsole->Printf(KMenuL4);
    iConsole->Printf(KMenuL5);
    iConsole->Printf(KMenuL6);

    TChar choice = iConsole->Getch();
    switch(choice)
        {
        case '1':
            {
            CreateFile();
            break;
            }
        case '2':
            {
            ChangeFileContents();
            break;
            }
        case '3':
            {
            RenameFile();
            break;
            }
        case '4':
            {
            ChangeFileAttributes ();
            break;
            }
        case '5':
            {
            DeleteFile();
            break;
            }
        default:
            {
            iOption = ETrue;
            _LIT(KPressAnyKey,"Press any key to Exit");
            iConsole->Printf(KPressAnyKey);
            iConsole->Read(iStatus);
            if(!IsActive())
                {
                SetActive();
                } 
            break;
            }
        }        
    }

/*
 * User gets notified when a new file is created with a particular name in a particular folder.
 */
void CChangeNotifier::CreateFile()
    {   
    if(!iIsCreateFile)
        {
        TBuf<40> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\");
        path.Append(KPath);
                            
        TBuf<20> filename;
        _LIT(KFileName,"create.file");
        filename.Append(KFileName);
                            
        TInt err = iNotify->AddNotification((TUint)TFsNotification::ECreate,path,filename);
        if(err == KErrNone)
            {
            _LIT(KSuccess,"\nAdded 'Create File' type notification to  CFsNotify object\n");
            iConsole->Printf(KSuccess);
            }
        else
            {
            _LIT(KSuccess,"Added Notification failed\n");
            iConsole->Printf(KSuccess);
            }
        err = iNotify->RequestNotifications(iReqStatus);
        if(err==KErrNone)
            {
            _LIT(KSuccess,"Request for the above notification is successful\n");
            iConsole->Printf(KSuccess);
            }                       

        DoChanges(ECreateFile);
        iStatus = iReqStatus; 
        if(!IsActive())
            {
            SetActive();
            }
        iIsCreateFile = ETrue;
        }
    else
        {
        _LIT(KResult,"File already created\n");
        iConsole->Printf(KResult);
        PressAnyKey();
        }
    }

/*
 * User gets notified when the contents of the file change.
 */
void CChangeNotifier::ChangeFileContents()
    {  
    TBuf<80> path;
    path.Append(iDrive);
    _LIT(KPath,":\\private\\e80000fb\\filechange\\");
    path.Append(KPath);
                            
    TBuf<20> filename;
    _LIT(KFileName,"file.txt");
    filename.Append(KFileName);
                            
    TInt err = iNotify->AddNotification((TUint)TFsNotification::EFileChange,path,filename);
    if(err == KErrNone)
        {
        _LIT(KSuccess,"\nAdded 'File contents change' type notification to  CFsNotify object\n");
        iConsole->Printf(KSuccess);
        }
    else
        {
        _LIT(KSuccess,"Added Notification failed\n");
        iConsole->Printf(KSuccess);
        }
    err = iNotify->RequestNotifications(iReqStatus);
    if(err==KErrNone)
        {
        _LIT(KSuccess,"Request for the above notification is successful\n");
        iConsole->Printf(KSuccess);
        } 
    DoChanges(EChangeFileContents);
    iStatus = iReqStatus; 
    if(!IsActive())
        {
        SetActive();
        } 
    }

/*
 * User gets notified when a file name is changed.
 */
void CChangeNotifier::RenameFile()
    {
    if(!iIsCreateFile)
        {
        _LIT(KCreateFile,"\nCreating a file so that you can try changing it attributes later\n");
        iConsole->Printf(KCreateFile);
        CreateFile();
        }
    else
        {
        if(!iIsRenameFile)
            {
            TBuf<80> path;
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\");
            path.Append(KPath);
            
            TBuf<20> filename;
            _LIT(KFileName,"create.file");
            filename.Append(KFileName);
                            
            TInt err = iNotify->AddNotification((TUint)TFsNotification::ERename,path,filename);
            if(err == KErrNone)
                {
                _LIT(KSuccess,"\nAdded 'File rename' type notification to  CFsNotify object\n");
                iConsole->Printf(KSuccess);
                }
            else
                {
                _LIT(KSuccess,"Added notification failed\n");
                iConsole->Printf(KSuccess);
                }
            err = iNotify->RequestNotifications(iReqStatus);
            if(err==KErrNone)
                {
                _LIT(KSuccess,"Request for the above notification is successful\n");
                iConsole->Printf(KSuccess);
                }
            DoChanges(ERenameFile);          
            iStatus = iReqStatus; 
            if(!IsActive())
                {
                SetActive();
                } 
            iIsRenameFile = ETrue;
            }
        else
            {
            _LIT(KResult,"File rename is already performed\n");
            iConsole->Printf(KResult);
            PressAnyKey();
            } 
        }
    }

/*
 * User gets notified when the file attributes change
 */
void CChangeNotifier::ChangeFileAttributes()
    { 
    if(!iIsCreateFile)
        {
        _LIT(KCreateFile,"\nCreating a file so that you can try changing it attributes later\n");
        iConsole->Printf(KCreateFile);
        CreateFile();
        }
    else
        {
        if(!iIsChangeFileAttributes )
            {
            TBuf<80> path;
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\");
            path.Append(KPath);
                            
            TBuf<20> filename;
            _LIT(KFileName,"create.file");
            filename.Append(KFileName);
                            
            TInt err = iNotify->AddNotification((TUint)TFsNotification::EAttribute,path,filename);
            if(err == KErrNone)
                {
                _LIT(KSuccess,"\nAdded 'File attributes change' type notification to  CFsNotify object\n");
                iConsole->Printf(KSuccess);
                }
            else
                {
                _LIT(KSuccess,"Added Notification failed\n");
                iConsole->Printf(KSuccess);
                }
            err = iNotify->RequestNotifications(iReqStatus);
            if(err==KErrNone)
                {
                _LIT(KSuccess,"Request for the above notification is successful\n");
                iConsole->Printf(KSuccess);
                }
            DoChanges(EChangeFileAttributes );                
            iStatus = iReqStatus; 
            if(!IsActive())
                {
                SetActive();
                } 
            iIsChangeFileAttributes  = ETrue;
            }
        else
            {
            _LIT(KResult,"File attributes already changed\n");
            iConsole->Printf(KResult);
            PressAnyKey();
            }    
        }
    }

/*
 * User gets notified when a file is deleted.
 */
void CChangeNotifier::DeleteFile()
    { 
    if(!iIsDeleteFile)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\deletefolder\\");
        path.Append(KPath);
                            
        TBuf<20> filename;
        _LIT(KFileName,"file.txt");
        filename.Append(KFileName);
                            
        TInt err = iNotify->AddNotification((TUint)TFsNotification::EDelete,path,filename);
        if(err == KErrNone)
            {
            _LIT(KSuccess,"\nAdded 'File Delete' type notification to  CFsNotify object\n");
            iConsole->Printf(KSuccess);
            }
        else
            {
            _LIT(KSuccess,"Add Notification failed\n");
            iConsole->Printf(KSuccess);
            }
        err = iNotify->RequestNotifications(iReqStatus);
        if(err==KErrNone)
            {
            _LIT(KSuccess,"Request for the above notification is successful\n");
            iConsole->Printf(KSuccess);
            }
                        
        DoChanges(EDeleteFile);
        iStatus = iReqStatus; 
        if(!IsActive())
            {
            SetActive();
            }  
        iIsDeleteFile = ETrue;
        }
    else
        {
        _LIT(KResult,"File already deleted\n");
        iConsole->Printf(KResult);
        PressAnyKey();
        }    
    }

/*
 * User gets notified when a folder is created.
 */
void CChangeNotifier::CreateFolder()
    {   
    if(!iIsCreateFolder)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\createfolder\\");
        path.Append(KPath);
                            
        TBuf<10> filename;
        _LIT(KFileName,"*");
        filename.Append(KFileName);
        
        TInt err = iNotify->AddNotification((TUint)TFsNotification::ECreate,path,filename);
        if(err == KErrNone)
            {
            _LIT(KSuccess,"\nAdded 'Create Folder' type notification to  CFsNotify object\n");
            iConsole->Printf(KSuccess);
            }
        else
            {
            _LIT(KSuccess,"Add Notification failed\n");
            iConsole->Printf(KSuccess);
            }
        err = iNotify->RequestNotifications(iReqStatus);
        if(err==KErrNone)
            {
            _LIT(KSuccess,"Request for the above notification is successful\n");
            iConsole->Printf(KSuccess);
            }
                        
        DoChanges(ECreateFolder);
        iStatus = iReqStatus; 
        if(!IsActive())
            {
            SetActive();
            }  
        iIsCreateFolder = ETrue;
        }
        else
            {
            _LIT(KResult,"Folder already created\n");
            iConsole->Printf(KResult);
            PressAnyKey();
            }  
        }

/*
 * User gets notified when a folder rename is done.
 */
void CChangeNotifier::RenameFolder()
    {   
    if(!iIsRenameFolder)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\newfolder\\");
        path.Append(KPath);
        
        TBuf<80> newPath;
        newPath.Append('c');
        _LIT(KPath1,":\\private\\e80000fb\\filechange\\RenameFolder\\");
        newPath.Append(KPath1);
                            
        TBuf<20> filename;
        _LIT(KFileName,"*");
        filename.Append(KFileName);
                            
        TInt err = iNotify->AddNotification((TUint)TFsNotification::ERename,path,filename);
        if(err == KErrNone)
            {
            _LIT(KSuccess,"\nAdded 'Rename Folder' type notification to  CFsNotify object\n");
            iConsole->Printf(KSuccess);
            }
        else
            {
            _LIT(KSuccess,"Add Notification failed\n");
            iConsole->Printf(KSuccess);
            }
        err = iNotify->RequestNotifications(iReqStatus);
        if(err==KErrNone)
            {
            _LIT(KSuccess,"Request for the above notification is successful\n");
            iConsole->Printf(KSuccess);
            }
        DoChanges(ERenameFolder);                
        iStatus = iReqStatus; 
        if(!IsActive())
            {
            SetActive();
            }  
        iIsRenameFolder = ETrue;
        }
    else
        {
        _LIT(KResult,"Folder rename is performed already\n");
        iConsole->Printf(KResult);
        PressAnyKey();
        }  
    }

void CChangeNotifier::DeleteFolder()
    { 
    if(!iIsDeleteFolder)
        {
        TBuf<80> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\deletefolder\\");
        path.Append(KPath);
        
        TBuf<20> filename;
        _LIT(KFileName,"*");
        filename.Append(KFileName);
                            
        TInt err = iNotify->AddNotification((TUint)TFsNotification::EDelete ,path,filename);
        if(err == KErrNone)
            {
            _LIT(KSuccess,"\nAdded 'Delete Folder' type notification to  CFsNotify object\n");
            iConsole->Printf(KSuccess);
            }
        else
            {
            _LIT(KSuccess,"Add Notification failed\n");
            iConsole->Printf(KSuccess);
            }

        err = iNotify->RequestNotifications(iReqStatus);
        if(err==KErrNone)
            {
            _LIT(KSuccess,"Request for the above notification is successful\n");
            iConsole->Printf(KSuccess);
            }
        DoChanges(EDeleteFolder);
        iStatus = iReqStatus; 
        if(!IsActive())
            {
            SetActive();
            } 
        iIsDeleteFolder = ETrue;
        }
    else
        {
        _LIT(KResult,"Folder is already deleted\n");
        iConsole->Printf(KResult);
        PressAnyKey();
        }
    if(iDeleteFileContinue )
        {
        iDeleteFileContinue = EFalse;
        }
    }

/*
 * User gets notified when a folder is deleted.
 */
void CChangeNotifier::DeleteFileandFolder()
    { 
        if((!iIsDeleteFile)){
            DeleteFile();
            iDeleteFileContinue = ETrue;
            }
        else{
            DeleteFolder();
            }
    }

/*
 * User gets notified when there is change in Folder.
 */
void CChangeNotifier::NotifyChangeToFolder()
    {              
    _LIT(KMenuL1,"\nChoose an option:\n");
    _LIT(KMenuL2,"1.Create a Folder\n");
    _LIT(KMenuL3,"2.Rename the Folder\n");
    _LIT(KMenuL4,"3.Delete the Folder\n");
        
    iConsole->Printf(KMenuL1);
    iConsole->Printf(KMenuL2);
    iConsole->Printf(KMenuL3);
    iConsole->Printf(KMenuL4);

    TChar choice = iConsole->Getch();
    switch(choice)
        {
        case '1':
            {
            CreateFolder();
            break;
            }
        case '2':
            {
            RenameFolder();
            break;
            }
        case '3':
            {
           DeleteFileandFolder();
            break;
            }
        default:
            {
            iOption = ETrue;
            _LIT(KPressAnyKey,"Press any key to Exit");
            iConsole->Printf(KPressAnyKey);
            iConsole->Read(iStatus);
            if(!IsActive())
                {
                SetActive();
                } 
            break;
            }
        }        
    }
/*
 * User gets notified when there is a change in the Volume name of a Drive.
 */
void CChangeNotifier::VolumeNameChange()
    {   
    if(!iIsVolumeNameChange)
        {
        TBuf<5> path;
        path.Append('U'); 
        path.Append(_L(":"));
        TBuf<20> filename;
                            
        TInt err = iNotify->AddNotification((TUint)TFsNotification::EVolumeName,path,filename);
        if(err == KErrNone)
            {
            _LIT(KSuccess,"\nAdded 'Volume name change' type notification to  CFsNotify object\n");
            iConsole->Printf(KSuccess);
            }
        else
            {
            _LIT(KSuccess,"Add Notification failed\n");
            iConsole->Printf(KSuccess);
            }
        err = iNotify->RequestNotifications(iReqStatus);
        if(err==KErrNone)
            {
            _LIT(KSuccess,"Request for the above notification is successful\n");
            iConsole->Printf(KSuccess);
            }
        DoChanges(EVolumeNameChange);
        iStatus = iReqStatus; 
        if(!IsActive())
            {
            SetActive();
            }  
        iIsVolumeNameChange = ETrue;
        }
    else{
        _LIT(KResult,"Volume name already changed\n");
        iConsole->Printf(KResult);
        PressAnyKey();
        } 
    }


/*
 * Client receives notifications for multiple notifications requested.
 * 
 * The major differences between the old (RFs::NotifyChange) and new (CFsNotify) frameworks are:
 * The ability to request multiple notifications on differing paths, 
 * notification types,  combinations of paths/notification-types etc..,on the same request.
 * Guarantee that all notifiable operations will be served to the client (no race to re-request notifications). 
 * Client can query the notification to find out what change has occurred (no directory scanning required).
 */
void CChangeNotifier::MultipleNotifications()
    {
    if(!iIsMultipleNotifications)
        {
        TBuf<40> path;
        path.Append(iDrive);
        _LIT(KPath,":\\private\\e80000fb\\filechange\\");
        path.Append(KPath);
                               
        TBuf<20> filename;
        _LIT(KFileName,"notify.file");
        filename.Append(KFileName);
                               
        TInt err = iNotify->AddNotification((TUint)TFsNotification::ECreate,path,filename);
        if(err == KErrNone)
            {
            _LIT(KSuccess,"\nAdded 'Create File' type notification to  CFsNotify object\n");
            iConsole->Printf(KSuccess);
            }
        else
            {
            _LIT(KSuccess,"Added Notification failed\n");
            iConsole->Printf(KSuccess);
            }
    
        err = iNotify->AddNotification((TUint)TFsNotification::ERename,path,filename);
        if(err == KErrNone)
            {
            _LIT(KSuccess,"\nAdded 'Rename file' type notification to  CFsNotify object\n");
            iConsole->Printf(KSuccess);
            }
        else
            {
            _LIT(KSuccess,"Add Notification failed\n");
            iConsole->Printf(KSuccess);
            }
        iNotify->RequestNotifications(iReqStatus);
        DoChanges(EMultipleNotifications);
        iStatus = iReqStatus; 
        if(!IsActive())
            {
            SetActive();
            }  
        iIsMultipleNotifications = ETrue;
        }
    else
        {
        _LIT(KResult,"Mutliple notifications already done\n");
        iConsole->Printf(KResult);
        PressAnyKey();  
        }
    }

/*
 * Do the file/folder changes as per the notificaiton request posted by the CFsNotify client.
 * After the file/folder changes complete. RunL() will be called, successful completion of the request is displayed in RunL()
 * 
 * Note : The file changes can originate from any thread or a process.
 */
void CChangeNotifier::DoChanges(CChangeNotifier::TFileOperations aOperation)
    {
    TBuf<80> path;
    TBuf<20> filename;
    TBuf<80> fullname;
    RFile file;
    switch(aOperation)
        {
        case CChangeNotifier::ECreateFile :
            {
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\");
            path.Append(KPath);
                                        
            TBuf<20> filename;
            _LIT(KFileName,"create.file");
            filename.Append(KFileName);
                    
            fullname.Append(path);
            fullname.Append(filename);                  
            file.Replace(iFs,fullname,EFileWrite); //Replace produces Create notification
            
            _LIT(KTxtFileChanges,"\nNew file create.file created at %c drive at :\\private\\e80000fb\\filechange\\ location\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            return;
            }
            
        case CChangeNotifier::EChangeFileContents :
            {
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\");
            path.Append(KPath);    
            _LIT(KFileName,"file.txt");
            filename.Append(KFileName);
            fullname.Append(path);
            fullname.Append(filename);                  
            file.Replace(iFs,fullname,EFileWrite); //Replace produces Create notification
            _LIT8(KFileData,"Some Data");
            file.Write(KFileData);
            
            _LIT(KTxtFileChanges,"\nfile contents of file.txt at %c drive at :\\private\\e80000fb\\filechange\\ location has changed\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            return;
            }
            
        case CChangeNotifier::ERenameFile :
            {
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\");
            path.Append(KPath);  
            _LIT(KFileName,"create.file");
            filename.Append(KFileName);
            fullname.Append(path);
            fullname.Append(filename); 

            file.Replace(iFs,fullname,EFileWrite); //Replace produces Create notification
            _LIT(KRenameFile,"NewFileName");
            file.Rename(KRenameFile);
            
            _LIT(KTxtFileChanges,"\ncreate.file at %c drive at :\\private\\e80000fb\\filechange\\ location has been renamed to NewFileName\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            return;
            }
            
        case CChangeNotifier::EChangeFileAttributes  :
            {
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\");
            path.Append(KPath);
            _LIT(KFileName,"create.file");
            filename.Append(KFileName);
            fullname.Append(path);
            fullname.Append(filename);                  
            file.Replace(iFs,fullname,EFileWrite); //Replace produces Create notification
            file.SetSize(134);
            
            _LIT(KTxtFileChanges,"\ncreate.file at %c drive at :\\private\\e80000fb\\filechange\\ location has it attributes changed to 134bytes\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            return;
            }
        case CChangeNotifier::EDeleteFile :
            {
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\deletefolder\\");
            path.Append(KPath); 
            _LIT(KFileName,"file.txt");
            filename.Append(KFileName);
            fullname.Append(path);
            fullname.Append(filename);                                      
            iFs.Delete(fullname);
            
            _LIT(KTxtFileChanges,"\nfile.txt at %c drive at :\\private\\e80000fb\\filechange\\deletefolder\\ location has been deleted\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            return;
            }
        case CChangeNotifier::ECreateFolder :
            {
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\createfolder\\");
            path.Append(KPath);                                         
            iFs.MkDir(path);
            
            _LIT(KTxtFileChanges,"\nA new folder 'createfolder' at %c drive at :\\private\\e80000fb\\filechange\\ location has been created\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            return;
            }
        case CChangeNotifier::ERenameFolder :
            {
            path.Append(iDrive);
            _LIT(KPath,"\\private\\e80000fb\\filechange\\newfolder\\");
            path.Append(KPath);
            
            TBuf<80> newPath;
            newPath.Append(iDrive);
            _LIT(KNewPath,"\\private\\e80000fb\\filechange\\RenameFolder\\");
            newPath.Append(KNewPath);   
            iFs.Rename(path,newPath);
            
            _LIT(KTxtFileChanges,"\nnewfolder at %c drive at :\\private\\e80000fb\\filechange\\ location has been renamed to RenameFolder\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            return;
            }
        case CChangeNotifier::EDeleteFolder :
            {
            path.Append('U');
            _LIT(KPath,":\\private\\e80000fb\\filechange\\deletefolder\\");
            path.Append(KPath);        
            iFs.RmDir(path);
            
            _LIT(KTxtFileChanges,"\ndeletefolder at %c drive at :\\private\\e80000fb\\filechange\\ location has been deleted\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            return;
            }
        case CChangeNotifier::EVolumeNameChange :
            {
            _LIT(KVolumeLabel,"MyNewDrive");  
            iDriveNumber = (TChar)'U'- (TChar)'A';
            iFs.SetVolumeLabel(KVolumeLabel,iDriveNumber);
            _LIT(KResult,"Volume name changed to MyNewDrive \n");
            iConsole->Printf(KResult);
            return;
            }
        case CChangeNotifier::EMultipleNotifications :
            {
            // Create a file 'notify.file'
            path.Append(iDrive);
            _LIT(KPath,":\\private\\e80000fb\\filechange\\");
            path.Append(KPath);
                                                           
            TBuf<20> filename;
            _LIT(KFileName,"notify.file");
            filename.Append(KFileName);
                                       
            fullname.Append(path);
            fullname.Append(filename);                  
            file.Replace(iFs,fullname,EFileWrite); //Replace produces Create notification
                               
            _LIT(KTxtFileChanges,"\nNew file notify.file created at %c drive at :\\private\\e80000fb\\filechange\\ location\n");
            iConsole->Printf(KTxtFileChanges,iDrive);
            
            _LIT(KRenameFile,"rename.file");
            file.Rename(KRenameFile);
                   
            _LIT(KTxtFolderChanges,"\nnotify.file at %c drive at :\\private\\e80000fb\\filechange\\ location has been renamed to rename.file\n");
            iConsole->Printf(KTxtFolderChanges,iDrive);
            return;
            }     
        }
    file.Close();  
    }

void CChangeNotifier::PressAnyKey()
    {
    iOptionContinue = ETrue;
    _LIT(KPressAnyKey,"Press any key to continue");
    iConsole->Printf(KPressAnyKey);
    iConsole->Read(iStatus);
    if(!IsActive())
        {
        SetActive();
        }    
    }

void CChangeNotifier::DoCancel()
    {
    iNotify->CancelNotifications(iReqStatus);
    iConsole->ReadCancel();
    }


void CChangeNotifier::RequestCharacter()
    {   
    iOption = EFalse;
    iOptionContinue = EFalse;
    iDeleteFileContinue = EFalse;
    _LIT(KMenuL1,"\nChoose an option:\n");
    _LIT(KMenuL2,"1.Get notified when a file changes\n");
    _LIT(KMenuL3,"2.Get notified when a folder changes\n");
    _LIT(KMenuL4,"3.Get notified when the volume name changes\n");
    _LIT(KMenuL5,"4.Multiple notifications\n");
    
    iConsole->Printf(KMenuL1);
    iConsole->Printf(KMenuL2);
    iConsole->Printf(KMenuL3);
    iConsole->Printf(KMenuL4);
    iConsole->Printf(KMenuL5);

    TChar choice = iConsole->Getch();
    switch(choice)
        {
        case '1':
            {
            NotifyChangeToFile();
            break;
            }
        case '2':
            {
            NotifyChangeToFolder();
            break;
            }
        case '3':
            {
            VolumeNameChange();
            break;
            }
        case '4':
                   {
                   MultipleNotifications();
                   break;
                   }
        default:
            {
            iOption = ETrue;
            _LIT(KPressAnyKey,"Press any key to Exit");
            iConsole->Printf(KPressAnyKey);
            iConsole->Read(iStatus);
            if(!IsActive())
                {
                SetActive();
                } 
            break;
            }
        }
    }

static void MainL()
    {
    CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
    CleanupStack::PushL(scheduler);
    CActiveScheduler::Install(scheduler);
    CChangeNotifier* changeNotifier = CChangeNotifier::NewL();
    CleanupStack::PushL(changeNotifier);
    changeNotifier->RequestCharacter();
    CActiveScheduler::Start();
    
    CleanupStack::PopAndDestroy(changeNotifier);
    CleanupStack::PopAndDestroy(scheduler);
    }

extern TInt E32Main()
    {
    // Create cleanup stack
    __UHEAP_MARK;
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(cleanup == NULL)
        {
        return KErrNoMemory;
        }

    // Run application code inside a TRAP harness.
    TRAPD(mainError, MainL());
    if(mainError != KErrNone)
        {
        _LIT(KUserPanic,"Failed to complete");  
        User::Panic(KUserPanic, mainError);
        }
    delete cleanup;
    __UHEAP_MARKEND;
    return KErrNone;
    }
