// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

#include "threadclient.h"
#include "threadserver.h"
#include "threadclientserver.h"
#include<e32debug.h>

#define KMaxServerNameLength 256

static const TUid KUidThreadServer = {0x10283039}; // Thread Server UID
const TInt KCreateSessionRetryCount = 2; 			//CreateSession retry count
const TInt KServerDefaultMessageSlots = 2; 			//default async message slots

/**
	Starts the server in the current thread. Simultaneous launching
	of two such processes should be detected when the second one attempts to
	create the server object with its globally unique name.
	@return KErrNone if successful, otherwise one of the other system wide error codes.
*/
static TInt StartServer()
	{
	const TUidType serverUid(KNullUid,KNullUid,KUidThreadServer);
	TThreadFunction serverFunc = CThreadServer::StartThread;
	RThread server;
	
	//create a new thread
	TInt r = server.Create(_L(""), serverFunc, KThreadServerStackSize,
	#ifdef SYMBIAN_USE_SEPARATE_HEAPS
	                	KThreadServerInitHeapSize, KThreadServerMaxHeapSize,
	#else
						&User::Heap(), //shared heap is now default
	#endif
	                	NULL,EOwnerProcess);
	
	if (r!=KErrNone)
		return r;
	TRequestStatus stat;
	server.Rendezvous(stat);
	
	if (stat!=KRequestPending)
		server.Kill(KErrCouldNotConnect);		// abort startup
	else
		server.Resume();	// logon OK - start the server
	
	User::WaitForRequest(stat);		// wait for start or death
	// The server exit type may be a panic value and a panic value can be zero, which is the same value as KErrNone. 
	// So, the exit type is tested for a panic before being returned.
	r=(server.ExitType()==EExitPanic) ? KErrServerTerminated : stat.Int();
	server.Close();
	return r;
	}
/**
	Connect to the server, attempting to start it if necessary
*/
EXPORT_C TInt RThreadClient::Connect()
	{
	TInt retry=KCreateSessionRetryCount;
	FOREVER
		{
		// try to create a session with the server which has KServerDefaultMessageSlots async message slots
		TInt r=CreateSession(KThreadServerName,
							 TVersion(KThreadServerVersion,
									  KThreadServerMinorVersionNumber,
									  KThreadServerBuildVersionNumber),
							 KServerDefaultMessageSlots);
		if (r!=KErrNotFound && r!=KErrServerTerminated)
			return r;
		if (--retry==0)
			return r;
		r=StartServer();
		if (r!=KErrNone && r!=KErrAlreadyExists)
			return r;
		// 1. r=KErrNone means Server starts up successfully
		// 2. r=KErrAlreadyExists means duplicate server was trying to start up 
		// 		Note: if there are other functions called before CServer2::StartL() during construction,
		//      	other errors might happen when the duplicate server is trying to start
		//       	therefore, we recommend CServer2::StartL() is the first function in server construction
		//       	see CThreadServer::ConstructL()
	
		// NOTE: If users would like to retry start up server after other scenarios happened, e.g. panic,then go 
		//		 through the following steps:
		// 		  1. Increase the value of KCreateSessionRetryCount (only start up server once in this example)
		//		  2. Need another variable, e.g. TExitType, together with error code returned in StartServer()
		//			to distinguish these scenarios
		//		  3. modify the third if statement to stop exit when the required scenarios happens.
		//           
		}
	}
/**
	Open the device driver's logical channel
	@return KErrNone if successful, otherwise one of the other system wide error codes.
*/
EXPORT_C TInt RThreadClient::OpenDriver()
	{
	return SendReceive(EThreadServerOpenDriver);
	}
/**
	Send data to device (asynchronous request)
	@param aBuf A descriptor which contains the data to be sent
	@param aStatus A TRequestStatus reference
*/	
EXPORT_C void RThreadClient::Send(const TDesC8& aBuf, TRequestStatus& aStatus)
	{
	SendReceive(EThreadServerSendData, TIpcArgs(&aBuf), aStatus);
	}
/**
	Cancel sending request
*/	
EXPORT_C void RThreadClient::SendCancel()
	{
	SendReceive(EThreadServerSendDataCancel);
	}
/**
	Unload Device Driver
	@return KErrNone if successful, otherwise one of the other system wide error codes.
*/	
EXPORT_C TInt RThreadClient::UnloadDeviceDriver()
	{
	return SendReceive(EThreadServerUnloadDeviceDriver);
	}
/**
	Load Device Driver
	@return KErrNone if successful, otherwise one of the other system wide error codes.
*/	
EXPORT_C TInt RThreadClient::LoadDeviceDriver()
	{
	return SendReceive(EThreadServerLoadDeviceDriver);
	}

//EOF
