/*
 * ============================================================================
 *  Name     : CEComCalculatorAppUi from CEComCalculatorAppui.cpp
 *  Part of  : EComCalculator
 *  Created  : 29/05/2006 by Forum Nokia
 *  Version  : 2.0
 *  Copyright: Nokia Corporation
 * ============================================================================
 */

#include <avkon.hrh>
#include <aknnotewrappers.h>    // CAknInformationNote
#include "EComCalculatorAppUi.h"
#include "EComCalculatorAppView.h"
#include "EComCalculator.hrh"   // menu commands
#include "EComInterfaceDefinition.h"


// Names of plugin operations. Valid values are defined in plugin's
// resource file. See
//      ..\..\plugin\101F5465.RSS
// Typically plugin provider gives these in the plugin documentation.
_LIT8(KSumOperationName,"sum");
_LIT8(KMultiplyOperationName,"multiply");
_LIT(KImplNotFound,"Implementation not found: ");
_LIT(KEComErr,"ECOM error: ");
_LIT(KEComErrCont,". Is the Calculation plugin installed?");

const TInt KOpNameLength=32;
const TInt KOpLength = 64;
const TInt KErrMsgLength = 128;


// Perform the second phase construction. This is allowed to leave.
void CEComCalculatorAppUi::ConstructL()
    {
    BaseConstructL(EAknEnableSkin);

    // Create the view. Note that the view in this example is not CAknView,
    // but a simple Avkon UI compound control.
    iAppView = CEComCalculatorAppView::NewL(ClientRect());

    // Add the view to control stack so that it is able to receive key events.
    // The CEComCalculatorAppView handles them in the OfferKeyEventL method.
    AddToStackL(iAppView);
    }

// Construct the AppUi. This may not leave.
CEComCalculatorAppUi::CEComCalculatorAppUi()
    {
    // No implementation required
    }

// Destructor. Release reserved resources.
CEComCalculatorAppUi::~CEComCalculatorAppUi()
    {
    if (iAppView)
        {
        // Remember to remove the view from control stack.
        iEikonEnv->RemoveFromStack(iAppView);
        }

    delete iAppView;
    iAppView = NULL;

#ifdef __SERIES60_3X__
    REComSession::FinalClose();
#endif
    }

// Handle menu commands. The commands are defined in EComCalculator.hrh
void CEComCalculatorAppUi::HandleCommandL(TInt aCommand)
    {
    switch(aCommand)
        {
    case ECalculateSumCmd:
    case ECalculateMultiplyCmd:
        CalculateL(aCommand);
        break;

    case EAknSoftkeyExit:
    case EEikCmdExit:
        Exit();
        break;

    default:
        break;
        }
    }

// Perform calculation and tell the view to show the result.
void CEComCalculatorAppUi::CalculateL(TInt aCommand)
    {
    TBuf8<KOpNameLength> operationName; // Data used in ECOM resolution.
                             // Either "sum" or "multiply"
    TReal A, B, calcResult;
    _LIT(KFailedA,"Failed. A is not a valid number.");
    _LIT(KFailedB,"Failed. B is not a valid number.");

    // Gather required info to perform ECOM interface call.
    if( iAppView->GetA(A) != KErrNone )
        {
        ShowNoteL(KFailedA);
        return;
        }
    if( iAppView->GetB(B) != KErrNone )
        {
        ShowNoteL(KFailedB);
        return;
        }

    if(aCommand == ECalculateSumCmd)
        {
        operationName.Append(KSumOperationName);      // "sum"
        }
    else
        {
        operationName.Append(KMultiplyOperationName); // "multiply"
        }

    if(DoEComCalculationL(operationName, A, B, calcResult) == KErrNone)
        iAppView->ShowResultL(operationName, calcResult);
    }

// Let the ECOM plugin do the calculation.
TInt CEComCalculatorAppUi::DoEComCalculationL(const TDesC8& aOperationName,
    TReal aA, TReal aB, TReal& aResult)
    {
        // Create instance of our own ECOM interface definition. The NewL
        // method will ask ECOM framework to find appropriate implementation
        // for the interface. The file
        //     ..\..\interface\EComInterfaceDefinition.inl
        // defines, how the framework will search. It uses default resolver.
        //
        // The NewL fails, if there is no plugin installed in the system,
        // or if an implementation for aOperationName cannot be found
        // (resolved).
        //
        //  - aOperationName must match one of the names of plugin
        //    implementations. The valid names are defined in plugin
        //    implementation resource file, see
        //        ..\..\plugin\101F5465.RSS
        //    The field "default_data" specifies name for a implementation.
        //    (In more detail, it is the identification data used by resolver.
        //    Default resolver used in this example just checks the matching
        //    string.)
        //
        //  - In emulator, the plugin implementation project must be built.
        //    In device, the plugin dll providing implementations must be
        //    installed. See
        //        ..\..\plugin.
        //    If there is no plugin available, the ECOM framework causes leave.
        //
        CCalculationInterfaceDefinition* calculateInterface = NULL;
        TRAPD(error,
            calculateInterface =
                CCalculationInterfaceDefinition::NewL(aOperationName);
        )
        if( error )
        {
            HandleEComInitializationErrorL(error, aOperationName);
            return error;
        }
        CleanupStack::PushL(calculateInterface);

        // Use given interface implementation to perform calculation.
        aResult = calculateInterface->Calculate(aA, aB);

        CleanupStack::PopAndDestroy(calculateInterface);
        return KErrNone;
    }

// Show error note for ECOM errors
void CEComCalculatorAppUi::HandleEComInitializationErrorL(TInt aError,
    const TDesC8& aOperationName) const
    {

    TBuf<KOpLength> operationName16bit;
    TBuf<KErrMsgLength> errorMsg;
    operationName16bit.Copy(aOperationName);

    if(aError == KErrNotFound)  // Plugin implementation not found (no match)
        {
        errorMsg.Append(KImplNotFound);
        errorMsg.Append(operationName16bit);
        ShowNoteL(errorMsg);
        }
    else if(aError != KErrNone) // Some other ECOM error. See in SDK help
        {
        errorMsg.Append(KEComErr);
        errorMsg.AppendNum(aError);
        errorMsg.Append(KEComErrCont);
        ShowNoteL(errorMsg);
        }
    }

// Show a note. Note that successive frequent calls to this method results in
// showing the latest message only.
void CEComCalculatorAppUi::ShowNoteL(const TDesC& aMessage) const
    {

    CAknInformationNote* note = new(ELeave)CAknInformationNote;
    note->ExecuteLD(aMessage); // Deletes itself, when returns
    }

void CEComCalculatorAppUi::HandleResourceChangeL(TInt aType)
    {
    CAknAppUi::HandleResourceChangeL(aType); //call to upper class

    // ADDED FOR SCALABLE UI SUPPORT
    // *****************************
    if ( aType == KEikDynamicLayoutVariantSwitch )
        {
        iAppView->SetRect( ClientRect() );
        }
    }
