// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// The following class takes user input and sends it to a different 
// process called inverter.exe via InverterInQ message Queue.
//



/**
 @file
*/

#include "CCollector.h"
_LIT(KTitle,"InputMessage");
_LIT(KWelcome," Welcome to Message Queue Example application for the global queues\n");
_LIT(KEnterWords, "\n Enter some words and press enter key to send to inverter or Press Esc to exit.\n");
_LIT(KGlobalInverterInQ, "InverterInQ");
_LIT(KGlobalInverterOutQ, "InverterOutQ");
const TInt KNumberOfMsgs = 10;

/**
Constructor
*/
CCollector::CCollector(TInt aPriority):CActive( aPriority )
	{	
	CActiveScheduler::Add(this);
	}

CCollector::~CCollector()
	{
	Cancel();
	}

void CCollector::DoCancel()
	{
	}

/**
The constructor creates a console for requesting words from the user, creates two global
message queues, starts the other process named Inverter.exe and create another active object to 
recieve words from the inverter.  
*/
void CCollector::ConstructL()
	{
	iConsole = Console::NewL(KTitle, TSize(KConsFullScreen, KConsFullScreen));
	iConsole->Printf(KWelcome);
	iConsole->Printf(KEnterWords);
	
	//Create global message queue , which is used as input to the inverter and output to the collector.
	User::LeaveIfError(iInverterInQ.CreateGlobal(KGlobalInverterInQ, KNumberOfMsgs, EOwnerProcess));
		
	//Create global message queue , which is used as output to the inverter and input to the collector.
	User::LeaveIfError(iInverterOutQ.CreateGlobal(KGlobalInverterOutQ, KNumberOfMsgs, EOwnerProcess));
	
	//Start Inverter process.
	_LIT(KProcessName, "inverter.exe");
	User::LeaveIfError(process.Create(KProcessName, KNullDesC));
	process.Resume();
	
	iRcvInverterInQ= CMsgQActive::NewL(); 
	}

CCollector* CCollector::NewL(TInt aPriority)
	{
	CCollector* self=new(ELeave)CCollector(aPriority);
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

/**
Request for user input.
*/
void CCollector::RequestFunction()
	{
	iConsole->Read(iStatus);
	SetActive();
	}

/**
Store the characters input by the user in a buffer.
*/
void CCollector::RunL()
	{
	//Store the words input by the user in a buffer.
	_LIT(KChar,"%c");
	iConsole->Printf(KChar,(TUint8)(iConsole->KeyCode()));
	iBuf.Append(iConsole->KeyCode()); 	
	ProcessKeyPress(TChar(iConsole->KeyCode()));
	}

/**
The following function processes the key pressed by the user, stores in a buffer and sends 
the data to the inverter via collector. 
*/
void CCollector::ProcessKeyPress(TChar aChar)
    {    
    if (aChar == EKeyEscape)
        {
        this->Stop();
        }
    else if(aChar == EKeyEnter)
        {
    	_LIT(KNextLine , "\n");
    	iConsole->Printf(KNextLine);
    	this->SendMsgInQ(iBuf); 
    	iBuf.Zero();
    	this->RequestFunction();   	
        }
    else
    	{
    	this->RequestFunction();  
    	}	
    }

/**
The following function sends the stop message to the inverter via the message queue.
*/
void CCollector::Stop()
	{
    TBuf<1> escape;
    escape.Append(EKeyEscape); 
    //Sends stop message to the inverter.
    this->SendMsgInQ(escape); 
    
    //Cancel any outstanding request.
    iRcvInverterInQ->Cancel();
    delete iRcvInverterInQ;
    delete iConsole;
    CActiveScheduler::Stop();   
    escape.Zero();
	}

/**
The following function calls the asynchronous request to recieve the words from the 
inverter.
*/
void CCollector::StartRecieving()
	{
	iRcvInverterInQ->StartRecieving();
	}

/**
The following function sends the message to InverterInQ.
*/
void CCollector::SendMsgInQ(TBuf<100> buf)
	{
	TInt ret = iInverterInQ.OpenGlobal(KGlobalInverterInQ); 
	if (ret == KErrNone)
	{
	iInverterInQ.SendBlocking(buf);
	}
	}

LOCAL_C void DoStartL()
	{
	// Create active scheduler (to run active objects)
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);

	CCollector* iCollect= CCollector::NewL();
	CleanupStack::PushL(iCollect);
	
	 //Start getting the user input.
	iCollect->RequestFunction();
	
	//Recieve data from the inverter in another console.
	iCollect->StartRecieving();
	CActiveScheduler::Start();
	
	CleanupStack::PopAndDestroy(iCollect);
	CleanupStack::PopAndDestroy(scheduler);	
	}

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();
	if(cleanup == NULL)
		{
		return KErrNoMemory;
		}
	TRAPD(mainError, DoStartL());
	if(mainError != KErrNone)
		{
		_LIT(KUserPanic,"Failed to complete");	
		User::Panic(KUserPanic, mainError);
		}
		delete cleanup;
		__UHEAP_MARKEND;
		return KErrNone;
	}

